% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kd.R
\name{kd}
\alias{kd}
\title{Generate data via the Kaiser-Dickman (1962) algorithm.}
\usage{
kd(covmat, n, type = c("exact", "sample"))
}
\arguments{
\item{covmat}{a symmetric, positive definite covariance matrix}

\item{n}{the sample size for the data that will be generated}

\item{type}{type of data generation. \code{exact} generates data that
exactly correspond to \code{covmat}.  \code{sample} treats \code{covmat} as
a poulation covariance matrix, generating a sample of size \code{n}.}
}
\value{
\code{kd} returns a data matrix of dimension \code{n} by
\code{nrow(covmat)}.
}
\description{
Given a covariance matrix and sample size, generate raw data that correspond
to the covariance matrix.  Data can be generated to match the covariance
matrix exactly, or to be a sample from the population covariance matrix.
}
\details{
By default, R's \code{cov()} function divides by \code{n}-1.  The data
generated by this algorithm result in a covariance matrix that matches
\code{covmat}, but you must divide by \code{n} instead of \code{n}-1.
}
\examples{

#### First Example

## Get data
dat <- HolzingerSwineford1939[ , 7:15]
hs.n <- nrow(dat)

## Covariance matrix divided by n
hscov <- ((hs.n-1)/hs.n) * cov(dat)

## Generate new, raw data corresponding to hscov
newdat <- kd(hscov, hs.n)

## Difference between new covariance matrix and hscov is minimal
newcov <- (hs.n-1)/hs.n * cov(newdat)
summary(as.numeric(hscov - newcov))

## Generate sample data, treating hscov as population matrix
newdat2 <- kd(hscov, hs.n, type = "sample")

#### Another example

## Define a covariance matrix
covmat <- matrix(0, 3, 3)
diag(covmat) <- 1.5
covmat[2:3,1] <- c(1.3, 1.7)
covmat[3,2] <- 2.1
covmat <- covmat + t(covmat)

## Generate data of size 300 that have this covariance matrix
rawdat <- kd(covmat, 300)

## Covariances are exact if we compute sample covariance matrix by
## dividing by n (vs by n - 1)
summary(as.numeric((299/300)*cov(rawdat) - covmat))

## Generate data of size 300 where covmat is the population covariance matrix
rawdat2 <- kd(covmat, 300)

}
\references{
Kaiser, H. F. and Dickman, K. (1962).  Sample and population
score matrices and sample correlation matrices from an arbitrary population
correlation matrix.  \emph{Psychometrika, 27}(2), 179--182.
\doi{10.1007/BF02289635}
}
\author{
Ed Merkle (University of Missouri; \email{merklee@missouri.edu})
}
