/*-
 * Copyright (c) 2013  Peter Pentchev
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * stdsyslog - the main component of the utility for logging programs' output
 * to the system log
 */

#include <sys/types.h>
#include <sys/select.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "flexarr.h"

#ifndef __unused
#ifdef __GNUC__
#define __unused	__attribute__((unused))
#else
#define __unused
#endif
#endif

static struct {
	const char	* const name;
	int		 level;
} syslog_levels[] = {
#ifdef LOG_EMERG
	{"emerg",	LOG_EMERG},
#endif
#ifdef LOG_ALERT
	{"alert",	LOG_ALERT},
#endif
#ifdef LOG_CRIT
	{"crit",	LOG_CRIT},
#endif
#ifdef LOG_ERR
	{"err",		LOG_ERR},
#endif
#ifdef LOG_WARNING
	{"warning",	LOG_WARNING},
#endif
#ifdef LOG_NOTICE
	{"notice",	LOG_NOTICE},
#endif
#ifdef LOG_INFO
	{"info",	LOG_INFO},
#endif
#ifdef LOG_DEBUG
	{"debug",	LOG_DEBUG},
#endif
};
#define LEVELS_COUNT	(sizeof(syslog_levels) / sizeof(syslog_levels[0]))

static struct {
	const char	* const name;
	int		 facility;
} syslog_facilities[] = {
#ifdef LOG_AUTHPRIV
	{"authpriv",	LOG_AUTHPRIV},
#endif
#ifdef LOG_CRON
	{"cron",	LOG_CRON},
#endif
#ifdef LOG_DAEMON
	{"daemon",	LOG_DAEMON},
#endif
#ifdef LOG_FTP
	{"ftp",		LOG_FTP},
#endif
#ifdef LOG_KERN
	{"kern",	LOG_KERN},
#endif
#ifdef LOG_LOCAL0
	{"local0",	LOG_LOCAL0},
#endif
#ifdef LOG_LOCAL1
	{"local1",	LOG_LOCAL1},
#endif
#ifdef LOG_LOCAL2
	{"local2",	LOG_LOCAL2},
#endif
#ifdef LOG_LOCAL3
	{"local3",	LOG_LOCAL3},
#endif
#ifdef LOG_LOCAL4
	{"local4",	LOG_LOCAL4},
#endif
#ifdef LOG_LOCAL5
	{"local5",	LOG_LOCAL5},
#endif
#ifdef LOG_LOCAL6
	{"local6",	LOG_LOCAL6},
#endif
#ifdef LOG_LOCAL7
	{"local7",	LOG_LOCAL7},
#endif
#ifdef LOG_LPR
	{"lpr",		LOG_LPR},
#endif
#ifdef LOG_MAIL
	{"mail",	LOG_MAIL},
#endif
#ifdef LOG_NEWS
	{"news",	LOG_NEWS},
#endif
#ifdef LOG_SYSLOG
	{"syslog",	LOG_SYSLOG},
#endif
#ifdef LOG_USER
	{"user",	LOG_USER},
#endif
#ifdef LOG_UUCP
	{"uucp",	LOG_UUCP},
#endif
};
#define FACILITIES_COUNT	(sizeof(syslog_facilities) / sizeof(syslog_facilities[0]))

static struct {
	const char	* const name;
	int		 signal;
} signal_pass[] = {
#ifdef SIGHUP
	{"SIGHUP",	SIGHUP},
#endif
#ifdef SIGINT
	{"SIGINT",	SIGINT},
#endif
#ifdef SIGQUIT
	{"SIGQUIT",	SIGQUIT},
#endif
#ifdef SIGALRM
	{"SIGALRM",	SIGALRM},
#endif
#ifdef SIGUSR1
	{"SIGUSR1",	SIGUSR1},
#endif
#ifdef SIGUSR2
	{"SIGUSR2",	SIGUSR1},
#endif
#ifdef SIGWINCH
	{"SIGWINCH",	SIGWINCH},
#endif
};
#define SIGNAL_PASS_COUNT	(sizeof(signal_pass) / sizeof(signal_pass[0]))

struct fdspec {
	int	 fd;
	int	 prio;
};

struct fdinfo {
	int	 fd;
	int	 pipe[2];
	char	*rbuf;
	size_t	 ralloc, rpos;
};

static struct fdspec fdspec_default[2] = {
	{1, LOG_INFO},
	{2, LOG_ERR},
};
#define FDSPEC_DEFAULT_SIZE	(sizeof(fdspec_default) / sizeof(fdspec_default[0]))

static char readbuf[2048];

static pid_t	 	pid;
static volatile int	got_sigterm, got_sigchld;

static void	 usage(int ferr);
static void	 version(void);

static void	 process_read(struct fdinfo * const fd, char * const readbuf, size_t sz,
		const struct fdspec * const spec, size_t nspec);
static void	 process_line(const struct fdinfo * const fd, const char * const s,
		const struct fdspec * const spec, size_t nspec);

static int	 parse_spec(struct fdspec * const spec, const char * const s);
static int	 syslog_facility(const char * const s);

static void	 sig_pass(int sig);
static void	 sig_chld(int sig);
static void	 sig_term(int sig);
static void	 got_timeout(int * const sent_sigkill, struct timeval * const sig_timeout);

int
main(int argc, char * const argv[])
{
	int ch, hflag, Vflag, status;
	int facility;
	const char *pidfile;
	struct fdspec *spec, sp;
	size_t nspec, allocspec;
	struct fdinfo *fds;
	size_t nfds, allocfds, i, left; 
	pid_t npid;
	struct sigaction sa;
	int old_got_sigterm, old_got_sigchld, sent_sigkill;
	struct timeval sig_timeout;
	FILE *pidfp;

	hflag = Vflag = 0;
	nspec = 0;
	facility = LOG_DAEMON;
	pidfile = NULL;
	FLEXARR_INIT(spec, nspec, allocspec);
	/* FIXME: more options, filters... */
	while ((ch = getopt(argc, argv, "d:f:hlp:V")) != -1) {
		switch (ch) {
			case 'd':
				if (parse_spec(&sp, optarg) == -1)
					return (1);
				FLEXARR_ALLOC(spec, 1, nspec, allocspec);
				spec[nspec - 1] = sp;
				break;

			case 'f':
				facility = syslog_facility(optarg);
				if (facility == -1)
					errx(1,
					    "Unknown syslog facility '%s', "
					    "use '-f list' for a list",
					    optarg);
				break;

			case 'h':
				hflag = 1;
				break;

			case 'l':
				printf("Available levels (priorities):\n");
				for (i = 0; i < LEVELS_COUNT; i++)
					printf("%s\n", syslog_levels[i].name);
				return (0);
				break;

			case 'p':
				pidfile = optarg;
				break;

			case 'V':
				Vflag = 1;
				break;

			default:
				usage(1);
				/* NOTREACHED */
		}
	}
	if (Vflag)
		version();
	if (hflag)
		usage(0);
	if (Vflag || hflag)
		return (0);
	argc -= optind;
	argv += optind;
	if (argc < 1)
		usage(1);

	/* FIXME: Filters, log, priority, etc. */
	openlog(basename(argv[0]), LOG_PID, facility);

	/* Set up the file descriptors */
	if (nspec == 0) {
		spec = fdspec_default;
		nspec = FDSPEC_DEFAULT_SIZE;
	}
	FLEXARR_INIT(fds, nfds, allocfds);
	for (i = 0; i < nspec; i++) {
		int found = 0;
		size_t j;

		for (j = 0; j < nfds; j++)
			if (fds[j].fd == spec[i].fd) {
				found = 1;
				break;
			}
		if (found)
			continue;

		FLEXARR_ALLOC(fds, 1, nfds, allocfds);
		memset(fds + nfds - 1, 0, sizeof(*fds));
		fds[nfds - 1].fd = spec[i].fd;
		FLEXARR_INIT(fds[nfds - 1].rbuf, fds[nfds - 1].rpos,
		    fds[nfds - 1].ralloc);
	}

	/* Set up the pipes */
	for (i = 0; i < nfds; i++)
		if (pipe(fds[i].pipe) == -1)
			err(1, "Could not create a communication pipe");

	/* Set up the signal handlers (parent only, but must do it early) */
	memset(&sa, 0, sizeof(sa));
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = SA_RESTART;
	sa.sa_handler = sig_pass;
	for (i = 0; i < SIGNAL_PASS_COUNT; i++)
		if (sigaction(signal_pass[i].signal, &sa, NULL) == -1)
			err(1, "Could not set up the handler for %s (%d)",
			    signal_pass[i].name, signal_pass[i].signal);
	got_sigterm = old_got_sigterm = sent_sigkill = 0;
	sa.sa_handler = sig_term;
	if (sigaction(SIGTERM, &sa, NULL) == -1)
		err(1, "Could not set up the handler for SIGTERM (%d)",
		    SIGTERM);
	got_sigchld = old_got_sigchld = 0;
	sa.sa_flags = SA_NOCLDSTOP;
	sa.sa_handler = sig_chld;
	if (sigaction(SIGCHLD, &sa, NULL) == -1)
		err(1, "Could not set up the handler for SIGCHLD (%d)",
		    SIGCHLD);

	if (pidfile != NULL) {
		int pidfd;

		/**
		 * Ignore unlink() errors for the present.
		 * TODO: Add the -P pidfile option to run stalepid and,
		 * well, remove the process ID file if it is stale.
		 */
		unlink(pidfile);
		if (pidfd = open(pidfile, O_CREAT | O_EXCL | O_RDWR,
		    S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH), pidfd == -1)
			err(1, "Could not create the process ID file %s",
			    pidfile);
		if (pidfp = fdopen(pidfd, "w"), pidfp == NULL)
			err(1,
			    "Could not fdopen the process ID file %s at fd %d",
			    pidfile, pidfd);
	} else {
		pidfp = NULL;
	}

	pid = fork();
	if (pid == -1) {
		err(1, "Could not fork for %s", argv[0]);
	} else if (pid == 0) {
		if (pidfp != NULL) {
			fprintf(pidfp, "%ld\n", (long)getpid());
			fclose(pidfp);
		}

		for (i = 0; i < nfds; i++) {
			FILE *fp;

			close(fds[i].pipe[0]);
			close(fds[i].fd);
			if (dup2(fds[i].pipe[1], fds[i].fd) == -1) {
				fprintf((fds[i].fd == 2? stdout: stderr),
				    "Could not reopen file descriptor %d as our pipe writer %d: %s\n",
				    fds[i].fd, fds[i].pipe[1],
				    strerror(errno));
				exit(1);
			}

			/**
			 * Try to set line buffering, do not complain too
			 * loudly if it fails.
			 * Yes, this leaks FILE structures, but we really
			 * do not want to close the file descriptor!
			 */
			if (fp = fdopen(fds[i].fd, "w"), fp != NULL)
				setvbuf(fp, NULL, _IOLBF, 0);
		}
		execvp(argv[0], argv);
		err(1, "Could not execute %s", argv[0]);
		/* NOTREACHED */
	}
	if (pidfp != NULL)
		fclose(pidfp);

	/**
	 * Parent: close the write side of the pipe and set the read side
	 * to non-blocking.
	 */
	for (i = 0; i < nfds; i++) {
		int fd, flags;

		close(fds[i].pipe[1]);
		fd = fds[i].pipe[0];
		flags = fcntl(fd, F_GETFL, 0);
		if (flags == -1)
			err(1, "Could not obtain the flags for fd %d", fd);
		if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1)
			err(1, "Could not set the flags for fd %d", fd);
	}

	left = nfds;
	while (left > 0) {
		fd_set rfd, efd;
		int maxfd, n;
		struct timeval sel_timeout;

		FD_ZERO(&rfd);
		FD_ZERO(&efd);
		maxfd = 0;
		for (i = 0; i < nfds; i++) {
			int fd = fds[i].pipe[0];

			if (fd == -1)
				continue;
			FD_SET(fd, &rfd);
			FD_SET(fd, &efd);
			if (fd > maxfd)
				maxfd = fd;
		}
		if (got_sigterm) {
			struct timeval tv;

			if (gettimeofday(&tv, NULL) == -1)
				err(1, "Could not get the time of day");
			if (tv.tv_sec > sig_timeout.tv_sec) {
				got_timeout(&sent_sigkill, &sig_timeout);
				/* redo */
				continue;
			} else if (tv.tv_sec == sig_timeout.tv_sec) {
				sel_timeout.tv_sec = 0;
				if (tv.tv_usec >= sig_timeout.tv_usec) {
					got_timeout(&sent_sigkill, &sig_timeout);
					/* redo */
					continue;
				}
				sel_timeout.tv_usec = sig_timeout.tv_usec - tv.tv_usec;
			} else {
				sel_timeout.tv_sec = sig_timeout.tv_sec - tv.tv_sec;
				if (tv.tv_usec < sig_timeout.tv_usec) {
					sel_timeout.tv_usec = sig_timeout.tv_usec - tv.tv_usec;
				} else {
					sel_timeout.tv_usec = sig_timeout.tv_usec + 1000000L - tv.tv_usec;
					sel_timeout.tv_sec--;
				}
			}
		}
		n = select(maxfd + 1, &rfd, NULL, &efd,
		    got_sigterm? &sel_timeout: NULL);
		if (n == -1) {
			if (errno == EINTR) {
				/* Brief special processing for SIGCHLD. */
				if (got_sigchld && !old_got_sigchld) {
					/**
					 * When we get a SIGCHLD, we want to
					 * redo the select() loop just once.
					 * Thus, we go back to the start now
					 * (after possibly handling a SIGTERM),
					 * and we check for got_sigchld at
					 * the end.
					 */
					old_got_sigchld = 1;
				}

				/* The rest: special processing for SIGTERM. */
				if (!got_sigterm)
					continue;

				/* A second SIGTERM? */
				if (old_got_sigterm) {
					/**
					 * A second SIGTERM, somebody really
					 * wants us to pack up and leave.
					 */
					kill(pid, SIGKILL);
					/* Leave the rest to the timeout. */
					continue;
				}
				old_got_sigterm = 1;

				/* Nah, just set the timeout. */
				if (gettimeofday(&sig_timeout, NULL) == -1)
					err(1, "Could not get the time of day");
				sig_timeout.tv_sec += 5;
				/* And redo the select() immediately. */
				continue;
			} else {
				err(1, "select() error with %lu fds left",
				    (unsigned long)nfds);
			}
		} else if (n == 0) {
			got_timeout(&sent_sigkill, &sig_timeout);
		} else if (n < 1) {
			errx(1, "select(%lu) returned %d",
			    (unsigned long)nfds, n);
		}
		for (i = 0; i < nfds; i++) {
			int fd, closefd;

			fd = fds[i].pipe[0];
			if (fd == -1)
				continue;

			closefd = 0;
			if (FD_ISSET(fd, &rfd)) {
				ssize_t sz;

				sz = read(fd, readbuf, sizeof(readbuf));
				if (sz == -1)
					err(1, "read() error");
				else if (sz == 0)
					closefd = 1;
				else
					process_read(fds + i, readbuf, sz, spec, nspec);
			}
			
			if (FD_ISSET(fd, &efd))
				closefd = 1;

			if (closefd) {
				/* Close stuff and out */
				close(fd);
				fds[i].pipe[0] = -1;
				left--;
			}
		}

		if (got_sigchld)
			break;
	}

	/* Oof, yes, this is a bit naive.  Want better event handling! */
	for (i = 0; i < 3; i++) {
		npid = waitpid(pid, &status, 0);
		if (npid != -1 || errno != EINTR)
			break;
		if (got_sigterm && !old_got_sigterm)
			errx(1, "Oops, got SIGTERM during the waitpid() call");
		if (got_sigchld && !old_got_sigchld) {
			/* SIGCHLD is good, waitpid() ought to succeed now */
			old_got_sigchld = 1;
			i--;
		}
	}
	if (npid == -1)
		err(1, "Could not obtain the exit status of %s", argv[0]);
	else if (npid != pid)
		errx(1, "OS glitch: waitpid(%d) returned pid %d", pid, npid);
	if (WIFEXITED(status))
		exit(WEXITSTATUS(status));
	else if (WIFSIGNALED(status))
		exit(WTERMSIG(status) + 128);
	else if (WIFSTOPPED(status))
		/* Nah, this really shouldn't happen, should it... */
		exit(WSTOPSIG(status) + 128);
	else
		errx(1, "The %s child process neither exited nor was killed "
		    "or stopped; what does wait() status %d mean?!",
		    argv[0], status);
	/* NOTREACHED */
}

static void
usage(int ferr)
{
	const char *s = "Usage:\tstdsyslog [-d fd:level] [-f facility] [-p pidfile] command [arg...]\n"
	    "\tstdsyslog -f list\n"
	    "\tstdsyslog -l\n"
	    "\tstdsyslog -V | -h\n\n"
	    "\t-d\tspecify the level for messages on a file descriptor (more than once);\n"
	    "\t-h\tdisplay program usage information and exit;\n"
	    "\t-f\tspecify the syslog facility to use (or 'list' for info);\n"
	    "\t-l\tlist the available syslog levels;\n"
	    "\t-p\tspecify the file to write the child process's ID to;\n"
	    "\t-V\tdisplay program version information and exit.\n\n"
	    "Examples:\n"
	    "\tstdsyslog -d 1:notice -d 2:crit sprog some args\n"
	    "\tstdsyslog -p sprog.pid -d 1:info -d 2:err -d 5:crit sprog more args\n";

	if (ferr) {
		fprintf(stderr, "%s", s);
		exit(1);
	} else {
		printf("%s", s);
	}
}

static void
version(void)
{
	printf("stdsyslog 0.03\n");
}

static void
process_read(struct fdinfo * const fd, char * const rdbuf, size_t sz,
		const struct fdspec * const spec, size_t nspec)
{
	size_t i, cpos, linestart;
	char *p;

	/* Bah, this is weird, but oh well */
	for (i = 0; i < sz; i++)
		if (rdbuf[i] == '\0')
			rdbuf[i] = ' ';

	/* OK, tack it onto the end... */
	cpos = fd->rpos;
	FLEXARR_ALLOC(fd->rbuf, sz, fd->rpos, fd->ralloc);
	memcpy(fd->rbuf + cpos, rdbuf, sz);

	/* We only need to look for newlines from cpos onwards */
	linestart = 0;
	while (p = (char *)memchr(fd->rbuf + cpos, '\n', fd->rpos - cpos),
	    p != NULL) {
		*p = '\0';
		process_line(fd, fd->rbuf + linestart, spec, nspec);
		cpos = p + 1 - fd->rbuf;
		linestart = cpos;
		if (cpos == fd->rpos) {
			break;
		} else if (cpos > fd->rpos) {
			errx(1, "INTERNAL ERROR: fd %d: cpos overflow, cpos %lu rpos %lu rbuf %p p %p ralloc %lu\n", fd->fd, (unsigned long)cpos, (unsigned long)fd->rpos, fd->rbuf, p, (unsigned long)fd->ralloc);
		}
	}
	if (linestart > 0) {
		memmove(fd->rbuf, fd->rbuf + linestart, fd->rpos - linestart);
		fd->rpos -= linestart;
	}
}

static void
process_line(const struct fdinfo * const fd, const char * const s,
		const struct fdspec * const spec, size_t nspec)
{
	size_t i;

	/* FIXME: Rules, prefixes, etc. */
	for (i = 0; i < nspec; i++)
		if (spec[i].fd == fd->fd)
			break;
	if (i == nspec)
		errx(1, "INTERNAL ERROR: process_line(): no match on fd %d, line %s", fd->fd, s);

	syslog(spec[i].prio, "%s", s);
}

static int
syslog_facility(const char * const s)
{
	size_t i;

	if (!strcmp(s, "list")) {
		printf("Available facilities:\n");
		for (i = 0; i < FACILITIES_COUNT; i++)
			printf("%s\n", syslog_facilities[i].name);
		exit(0);
	}
	
	for (i = 0; i < FACILITIES_COUNT; i++)
		if (!strcmp(s, syslog_facilities[i].name))
			return (syslog_facilities[i].facility);
	return (-1);
}

static int
parse_spec(struct fdspec * const spec, const char * const s)
{
	char *copy, *tokctx, *part, *end;
	long long lfd;
	struct fdspec sp;
	size_t i;

	memset(&sp, 0, sizeof(sp));

	if (copy = strdup(s), copy == NULL)
		err(1, "Could not copy an option string");
	part = strtok_r(copy, ":", &tokctx);
	if (part == NULL)
		errx(1, "No file descriptor specified for -d");
	lfd = strtoll(part, &end, 10);
	if (*part == '\0' || *end != '\0' || lfd < 0 || lfd > INT_MAX)
		errx(1, "Invalid file descriptor specified for -d");
	sp.fd = lfd;

	part = strtok_r(NULL, ":", &tokctx);
	if (part == NULL || *part == '\0')
		errx(1, "No log priority specified for -d");
	for (i = 0; i < LEVELS_COUNT; i++)
		if (!strcmp(syslog_levels[i].name, part))
			break;
	if (i == LEVELS_COUNT)
		errx(1, "Invalid log priority specified for -d");
	sp.prio = syslog_levels[i].level;

	part = strtok_r(NULL, ":", &tokctx);
	if (part != NULL)
		errx(1, "Extra data at the end of the -d specification; "
		    "just fd:prio supported so far");

	memcpy(spec, &sp, sizeof(*spec));
	free(copy);
	return (0);
}

static void
sig_pass(int sig)
{
	if (pid != 0)
		kill(pid, sig);
}

static void
sig_chld(int sig __unused)
{
	got_sigchld = 1;
}

static void
sig_term(int sig)
{
	if (pid != 0)
		kill(pid, sig);
	got_sigterm = 1;
}

static void
got_timeout(int * const sent_sigkill, struct timeval * const sig_timeout)
{
	/* Timeout, apparently. */
	if (*sent_sigkill) {
		warnx("Process %ld would not die even after "
		    "a SIGKILL", (long)pid);
		exit(128 + SIGKILL);
	}

	/* OK, so reinitialize the structures for a SIGKILL */
	if (gettimeofday(sig_timeout, NULL) == -1)
		err(1, "Could not get the time of day");
	sig_timeout->tv_sec += 5;
	kill(pid, SIGKILL);
	*sent_sigkill = 1;
}
