use core:lang;
use lang:bs:macro;

/**
 * Base class for an SQL expression.
 */
class SQLExpr on Compiler {
	// Position in source.
	SrcPos pos;

	// Create.
	init(SrcPos pos) {
		init { pos = pos; }
	}

	// Resolve this expression. May return some other node. The default behavior is to return the
	// node itself, which is what is needed for literals for example.
	SQLExpr resolve(ResolveContext context) {
		this;
	}

	// Build this expression.
	void build(QueryBuilder into) : abstract;

	// Get the type of this expression. Computes the type the first time it is called, and may throw
	// a type-checking exception at that time.
	Value result() : final {
		if (!hasResult) {
			computedResult = computeResult();
			hasResult = true;
		}
		computedResult;
	}

	// Get a Storm expression that corresponds to this expression if available.
	lang:bs:Expr? stormExpr() {
		null;
	}

	// Compute the result.
	protected Value computeResult() : abstract;

	// Is the type computed?
	private Bool hasResult;

	// What is the cached value of the type.
	private Value computedResult;
}

/**
 * Literals.
 */
class SQLLiteral extends SQLExpr {
	init(SrcPos pos) {
		init(pos) {}
	}

	Bool same(SQLLiteral other) : abstract;
	void toSQL(StrBuf to) : abstract;

	Str toSQL() {
		StrBuf b;
		toSQL(b);
		b.toS;
	}

	void build(QueryBuilder into) : override {
		toSQL(into.query);
	}
}

/**
 * Numeric literal.
 */
class SQLInt extends SQLLiteral {
	// Integer value.
	Long value;

	// Create.
	init(SrcPos pos, Str value) {
		init(pos) {
			value = value.toLong;
		}
	}

	// Build.
	void toSQL(StrBuf to) : override {
		to << value;
	}

	// Get a Storm expression.
	lang:bs:Expr? stormExpr() : override {
		lang:bs:NumLiteral(pos, value);
	}

	// Result.
	Value computeResult() : override {
		// TODO: Use Long for all things here? Then we also need to promote other variables to Longs.
		named{Int};
	}

	// Output.
	void toS(StrBuf to) : override {
		to << value;
	}

	// Equality.
	Bool same(SQLLiteral other) : override {
		unless (other as SQLInt)
			return false;

		return value == other.value;
	}
}

/**
 * Float literal.
 */
class SQLFloat extends SQLLiteral {
	// Float value.
	Double value;

	// Create.
	init(SrcPos pos, Str value) {
		init(pos) {
			value = value.toDouble;
		}
	}

	// Build.
	void toSQL(StrBuf into) : override {
		// TODO: Ensure that the representation is appropriate...
		into << value;
	}

	// Get a Storm expression.
	lang:bs:Expr? stormExpr() : override {
		lang:bs:NumLiteral(pos, value);
	}

	// Result.
	Value computeResult() : override {
		named{Double};
	}

	// Output.
	void toS(StrBuf to) : override {
		to << value;
	}

	// Equality.
	Bool same(SQLLiteral other) : override {
		unless (other as SQLInt)
			return false;

		return value == other.value;
	}
}

/**
 * String literal.
 */
class SQLStr extends SQLLiteral {
	// String value.
	Str value;

	// Create.
	init(SrcPos pos, Str value) {
		init(pos) {
			value = value.unescape;
		}
	}

	// Build.
	void toSQL(StrBuf into) : override {
		// TODO: Proper escaping!
		into << "'";
		for (ch in value) {
			// Double any single quotes. That is enough.
			if (ch.codepoint == 0x27)
				into << ch << ch;
			else
				into << ch;
		}
		into << "'";
	}

	// Get a Storm expression.
	lang:bs:Expr? stormExpr() : override {
		lang:bs:StrLiteral(pos, value);
	}

	// Result.
	Value computeResult() : override {
		named{Str};
	}

	// Output.
	void toS(StrBuf to) : override {
		to << "\"" << value.escape() << "\"";
	}

	// Equality.
	Bool same(SQLLiteral other) : override {
		unless (other as SQLInt)
			return false;

		return value == other.value;
	}
}

/**
 * Name of something. Either something in Basic Storm, or something in SQL.
 *
 * We don't resolve this node immediately.
 */
class SQLName extends SQLExpr {
	// Table name, if present.
	Str? table;

	// The unresolved name.
	Str name;

	// Create.
	init(SrcPos pos, SStr value) {
		init(pos) {
			name = value.v;
		}
	}

	// Create, a dotted name.
	init(SrcPos pos, SStr table, SStr column) {
		init(pos) {
			table = table.v;
			name = column.v;
		}
	}

	// Resolve.
	SQLExpr resolve(ResolveContext context) : override {
		// Try to resolve the name.
		if (result = context.resolve(pos, table, name)) {
			return SQLColumn(pos, result.table, result.column);
		}

		if (table) {
			// If we have a table name, we can't do anything more.
			if (context.typed())
				throw SyntaxError(pos, "The name ${table}.${name} can not be resolved to a column.");
		} else {
			// Ask Basic Storm for guidance...
			var bsExpr = lang:bs:namedExpr(context.block, pos, name, lang:bs:Actuals());
			unless (bsExpr as lang:bs:UnresolvedName)
				return StormValue(pos, bsExpr);

			// If the above check failed and we are a typed expression, just return that one. It will
			// throw eventually.
			if (context.typed())
				return StormValue(pos, bsExpr);
		}

		// If untyped and we did not have a match yet, just use this node. Then we will fail at
		// runtime if it was invalid.
		this;
	}

	// Build. Only used in the "unknown" case.
	void build(QueryBuilder into) : override {
		if (table) {
			into.name(table);
			into.query << ".";
		}
		into.name(name);
	}

	// Result (we don't know the type).
	Value computeResult() : override {
		Value();
	}

	// Output.
	void toS(StrBuf to) : override {
		to << "unresolved:";
		if (table)
			to << table << ".";
		to << name;
	}
}

/**
 * A resolved Storm expression of some kind. Will be bound to the prepared statement.
 */
class StormValue extends SQLExpr {
	// The Basic Storm expression.
	lang:bs:Expr expr;

	// Create.
	init(SrcPos pos, lang:bs:Expr expr) {
		init(pos) {
			expr = expr;
		}
	}

	// Build.
	void build(QueryBuilder into) : override {
		into.query << "?";
		into.bind << expr;
	}

	// Get a Storm expression.
	lang:bs:Expr? stormExpr() : override {
		expr;
	}

	// Result.
	Value computeResult() : override {
		expr.result.type.asRef(false);
	}

	// Output.
	void toS(StrBuf to) : override {
		to << "bs:(" << expr << ")";
	}
}

/**
 * Escaped value from Basic Storm.
 */
class Escaped extends SQLExpr {
	// Captured syntax tree.
	lang:bs:SExpr expr;

	// Create.
	init(lang:bs:SExpr expr) {
		init(expr.pos) {
			expr = expr;
		}
	}

	// Resolve.
	SQLExpr resolve(ResolveContext context) : override {
		StormValue(pos, expr.transform(context.block));
	}

	// Build.
	void build(QueryBuilder into) : override {
		throw InternalError("An 'Escaped' node was not transformed.");
	}

	// Result.
	Value computeResult() : override {
		throw InternalError("An 'Escaped' node was not transformed.");
	}
}


/**
 * Reference to a column in SQL.
 */
class SQLColumn extends SQLExpr {
	// Table.
	Str table;

	// Column.
	Column column;

	init(SrcPos pos, Str table, Column column) {
		init(pos) {
			table = table;
			column = column;
		}
	}

	// Build.
	void build(QueryBuilder into) : override {
		into.name(table);
		into.query << ".";
		into.name(column.name);
	}

	// Result.
	Value computeResult() : override {
		column.datatype.storm;
	}

	// Output.
	void toS(StrBuf to) : override {
		to << "column:" << table << "." << column.name;
	}
}


/**
 * Built-in function in SQL.
 */
class SQLBuiltIn extends SQLExpr {
	private Str sql;
	private Value type;

	init(SrcPos pos, Str sql, Value type) {
		init(pos) {
			sql = sql;
			type = type;
		}
	}

	void build(QueryBuilder into) : override {
		into.query << sql;
	}

	Value computeResult() : override {
		type;
	}

	void toS(StrBuf to) : override {
		to << sql;
	}
}

// The CURRENT DATETIME function.
// Note: This currently only works in sqlite.
SQLBuiltIn currentDateTime(SrcPos pos) {
	SQLBuiltIn(pos, "datetime('now')", named{Str});
}
