use core:lang;
use lang:bs:macro;


/**
 * Operator that could be evaluated in Storm as well.
 */
class Operator extends SQLExpr {
	// Left hand side.
	SQLExpr lhs;

	// Right hand side.
	SQLExpr rhs;

	// Operator name in Storm (if available)
	Str stormOp;

	// Operator name in SQL.
	Str sqlOp;

	// Create.
	init(SQLExpr lhs, SQLExpr rhs, SStr op) {
		init(op.pos) {
			lhs = lhs;
			rhs = rhs;
			stormOp = op.v;
			sqlOp = op.v;
		}
	}

	// Create.
	init(SQLExpr lhs, SQLExpr rhs, SStr sql, Str storm) {
		init(sql.pos) {
			lhs = lhs;
			rhs = rhs;
			stormOp = storm;
			sqlOp = sql.v;
		}
	}

	SQLExpr resolve(ResolveContext context) : override {
		lhs = lhs.resolve(context);
		rhs = rhs.resolve(context);

		// If maybe types are involved, we bail out in trying to simplify.
		if (isMaybe(lhs.result) | isMaybe(rhs.result))
			return this;

		if (lStorm = lhs.stormExpr) {
			if (rStorm = rhs.stormExpr) {
				// Simplify!
				if (fn = findFunction()) {
					lang:bs:Actuals params;
					params.add(lStorm);
					params.add(rStorm);
					lang:bs:FnCall call(pos, Scope(), fn, params);
					return StormValue(pos, call);
				}
			}
		}

		this;
	}

	// Compute the result.
	Value computeResult() : override {
		unless (fn = findFunction()) {
			Value l = unwrapMaybe(lhs.result);
			Value r = unwrapMaybe(rhs.result);
			throw SyntaxError(pos, "Invalid parameter types for the operator: ${l} ${sqlOp} ${r}");
		}

		if (isMaybe(lhs.result) | isMaybe(rhs.result))
			wrapMaybe(fn.result);
		else
			fn.result;
	}

	// Build the operator.
	void build(QueryBuilder to) : override {
		to.query << "(";
		lhs.build(to);
		to.query << " " << sqlOp << " ";
		rhs.build(to);
		to.query << ")";
	}

	// To string.
	void toS(StrBuf to) : override {
		to << lhs << " " << sqlOp << " " << rhs;
	}

	// Figure out which function in Storm this operator would call.
	private Function? findFunction() {
		unless (lType = unwrapMaybe(lhs.result).type)
			throw SyntaxError(lhs.pos, "Unknown type of subexpression.");
		unless (rType = unwrapMaybe(rhs.result).type)
			throw SyntaxError(rhs.pos, "Unknown type of subexpression.");

		checkTypes(lType, rType);

		Value[] params = [Value(lType), Value(rType)];
		if (x = lType.find(stormOp, params, Scope()) as Function)
			return x;

		// Try one step up.
		if (parent = lType.parent) {
			if (x = parent.find(stormOp, params, Scope()) as Function)
				return x;
		}

		return null;
	}

	// Allow subclasses to override, in order to limit available types.
	protected void checkTypes(Type l, Type r) {}
}


// Create the equals operator, as we allow both = and ==.
Operator equalsOp(SQLExpr lhs, SQLExpr rhs, SStr op) on Compiler {
	Operator(lhs, rhs, SStr("=", op.pos), "==");
}


class StrConcatOp extends Operator {
	init(SQLExpr lhs, SQLExpr rhs, SStr op) {
		init(lhs, rhs, op, "+") {}
	}

	protected void checkTypes(Type l, Type r) : override {
		if ((l !is named{Str}) | (r !is named{Str}))
			throw SyntaxError(pos, "Both operands must be strings. Not ${Value(l)} and ${Value(r)}.");
	}
}

// Just like Operator, but disallows strings.
class NumOperator extends Operator {
	init(SQLExpr lhs, SQLExpr rhs, SStr op) {
		init(lhs, rhs, op) {}
	}

	protected void checkTypes(Type l, Type r) : override {
		if ((l is named{Str}) | (r is named{Str}))
			throw SyntaxError(pos, "The operator ${sqlOp} is not applicable to strings in SQL.");
	}
}

// Just like Operator, but only allows boolean values.
class BoolOperator extends Operator {
	init(SQLExpr lhs, SQLExpr rhs, SStr op, Str storm) {
		init(lhs, rhs, op, storm) {}
	}

	protected void checkTypes(Type l, Type r) : override {
		if ((l !is named{Bool}) | (r !is named{Bool}))
			throw SyntaxError(pos, "The operator ${sqlOp} requires both operands to be booleans.");
	}
}

// Create logical operators.
BoolOperator andOp(SQLExpr lhs, SQLExpr rhs, SStr op) {
	BoolOperator(lhs, rhs, op, "&");
}
BoolOperator orOp(SQLExpr lhs, SQLExpr rhs, SStr op) {
	BoolOperator(lhs, rhs, op, "|");
}

// NOT operator.
class NotOperator extends SQLExpr {
	SQLExpr operand;

	init(SrcPos pos, SQLExpr op) {
		init(pos) {
			operand = op;
		}
	}

	SQLExpr resolve(ResolveContext context) : override {
		operand = operand.resolve(context);

		if (storm = operand.stormExpr) {
			// For typechecking.
			computeResult();

			unless (fn = named{Bool}.find("!", [Value(named{Bool})], Scope()) as Function)
				throw InternalError("Unable to find the unary negation function.");

			lang:bs:FnCall call(pos, Scope(), fn, lang:bs:Actuals(storm));
			return StormValue(pos, call);
		}

		this;
	}

	Value computeResult() : override {
		if (unwrapMaybe(operand.result).type !is named{Bool})
			throw SyntaxError(operand.pos, "The operand to NOT must be a boolean.");

		return named{Bool};
	}

	void build(QueryBuilder to) : override {
		to.query << "(";
		lhs.build(to);
		to.query << " LIKE ";
		rhs.build(to);
		to.query << ")";
	}
}

// LIKE operator. This is special since it does not exist in Storm.
class LikeOperator extends SQLExpr {
	// Left- and right hand sides.
	SQLExpr lhs;
	SQLExpr rhs;

	// Create.
	init(SrcPos pos, SQLExpr l, SQLExpr r) {
		init(pos) {
			lhs = l;
			rhs = r;
		}
	}

	SQLExpr resolve(ResolveContext context) : override {
		lhs = lhs.resolve(context);
		rhs = rhs.resolve(context);

		this;
	}

	Value computeResult() : override {
		if (unwrapMaybe(lhs.result).type !is named{Str})
			throw SyntaxError(lhs.pos, "The left-hand side of a LIKE operator must be a string.");
		if (unwrapMaybe(rhs.result).type !is named{Str})
			throw SyntaxError(lhs.pos, "The right-hand side of a LIKE operator must be a string.");

		if (isMaybe(lhs.result) | isMaybe(rhs.result))
			return wrapMaybe(named{Bool});
		else
			return named{Bool};
	}

	void build(QueryBuilder to) : override {
		lhs.build(to);
		to.query << " LIKE ";
		rhs.build(to);
	}
}
