/* ====================================================================
 * Copyright (c) 2003-2007, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "ExportDialog.h"
#include "ExternProvider.h"
#include "RevisionWidget.h"
#include "Settings.h"
#include "commands/ExportParam.h"
#include "sublib/Gui.h"
#include "sublib/ExternButton.h"
#include "sublib/settings/LayoutSettings.h"
#include "util/String.h"

// qt
#include <QtGui/QLayout>
#include <QtGui/QPushButton>
#include <QtGui/QLabel>
#include <QtGui/QComboBox>
#include <QtGui/QToolTip>
#include <QtGui/QCheckBox>
#include <Qt3Support/Q3VButtonGroup>
#include <Qt3Support/Q3GroupBox>



ExportDialog::ExportDialog( ExternProvider* p, bool force, QWidget *parent )
: super( parent, 0, true, Qt::WStyle_Customize | Qt::WStyle_Dialog
  | Qt::WStyle_NormalBorder | Qt::WStyle_Title | Qt::WStyle_SysMenu ), _p(p)
{
  setName( "ExportDialog" );
  setCaption( _q("subcommander:export") );

  QVBoxLayout *vbl = new QVBoxLayout(this,5,8);
  vbl->setSpacing(10);
  {
    Q3GroupBox* gb = new Q3GroupBox(1,Qt::Vertical,this);
    gb->setTitle( _q("export options: ") );
    //gb->setInsideSpacing(10);
    gb->setInsideMargin(0);
    gb->setFlat(true);
    vbl->addWidget(gb);

    QGridLayout* gl = new QGridLayout(vbl,3,3);
    gl->setMargin(0);
    gl->setRowStretch( 1, 1 );
    {
      {
        QLabel* l = new QLabel(this);
        _src      = new QComboBox(this);
        _srcEx    = new ExternButton(this);

        l->setBuddy(_srcEx);
        l->setText( _q("&source path/url:") );
        l->setFixedWidth( l->sizeHint().width() );
        //_src->setEditable(true);
        _src->setEnabled(false);
        _src->setAutoCompletion(true);
        _srcEx->setEnabled(false);

        gl->addWidget(l,0,0);
        gl->addWidget(_src,0,1); 
        gl->addWidget(_srcEx,0,2); 

        connect( _src, SIGNAL(activated(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _src, SIGNAL(highlighted(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _src, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _srcEx, SIGNAL(clicked()), SLOT(srcExtern()) );

        QToolTip::add( _src, _q("the source path/url you like to export") );
      }
      {
        QLabel* l = new QLabel(this);
        _dst      = new QComboBox(this);
        _dstEx    = new ExternButton(this);

        l->setBuddy(_dstEx);
        l->setText( _q("&destination path:") );
        _dst->setEditable(true);
        _dst->setAutoCompletion(true);

        gl->addWidget(l,2,0);
        gl->addWidget(_dst,2,1); 
        gl->addWidget(_dstEx,2,2); 

        connect( _dst, SIGNAL(activated(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _dst, SIGNAL(highlighted(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _dst, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
        connect( _dstEx, SIGNAL(clicked()), SLOT(dstExtern()) );

        QToolTip::add( _dst, _q("the local destination folder for the export") );
      }
    }

    _srcRw = new RevisionWidget(false,"SND","H",0,this);
    vbl->addWidget(_srcRw);
    QToolTip::add( _srcRw, _q("revision to export") );

    QHBoxLayout* h1 = new QHBoxLayout;
    vbl->addLayout(h1);
    {
      _force = new QCheckBox(this);
      _force->setText( _q("&force") );
      _force->setChecked(force);
      h1->addWidget(_force,0);

      Q3GroupBox* gbeol = new Q3GroupBox(this);
      gbeol->setColumnLayout( 1, Qt::Horizontal );
      gbeol->setTitle( _q(" eol style ") );
      h1->addWidget(gbeol,1);
      {
        _eol = new QComboBox(gbeol);
        _eol->insertItem( "default" );
        _eol->insertItem( "LF" );
        // ?? _eol->insertItem( "CR" );
        _eol->insertItem( "CRLF" );

        QToolTip::add( _eol, _q("overwrite eol style") );
      }
    }

    QHBoxLayout* hu = new QHBoxLayout;
    vbl->addLayout(hu);
    {
      // eats extra space, so the buttons keep their size
      hu->addStretch(1); 

      _ok = new QPushButton(this);
      _ok->setEnabled(false);
      _ok->setText( _q("&Ok") );
      _ok->setDefault(true);
      hu->addWidget(_ok);
      QPushButton* ca = new QPushButton(this);
      ca->setText( _q("&Cancel") );
      hu->addWidget(ca);
      
      hu->addSpacing(getSizeGripSpacing());

      connect( _ok, SIGNAL(clicked()), SLOT(accept()) );
      connect( ca, SIGNAL(clicked()), SLOT(reject()) );
    }
  }

  Settings s;
  resize( s.layout().getSize( name(), QSize() ) );

  // don't resize vertically.
  setMaximumHeight( sizeHint().height() );
}

ExportDialog::~ExportDialog()
{
  Settings s;
  s.layout().setSize( name(), geometry().size() );
}

void ExportDialog::initSrcUrl( const QString& url )
{
  _type = repository;
  _src->insertItem( url, 0 );
  _srcRw->setEnabled(true);
}

void ExportDialog::initSrcPath( const QString& path )
{
  _type = workingcopy;
  _src->insertItem( path, 0 );
  _srcRw->setEnabled(false);
}

void ExportDialog::checkOk( const QString& text )
{
  QString src = _src->currentText();
  QString dst = _dst->currentText();

  if( ! src.isEmpty() && ! dst.isEmpty() )
  {
    _ok->setEnabled(true);
  }
  else
  {
    _ok->setEnabled(false);
  }
}

void ExportDialog::srcExtern()
{
  if( _type == workingcopy )
  {
    sc::String res;

    if( _p->selectPath( this, sc::String(_src->currentText().utf8()), res, ExternProvider::Dir ) )
    {
      _src->setCurrentText(QString::fromUtf8(res) );
      //_src->insertItem( QString::fromUtf8(res), 0 );
    }
  }
  else if( _type == repository )
  {
    sc::String res;

    if( _p->selectUrl( this, sc::String(_src->currentText().utf8()), res, ExternProvider::Dir ) )
    {
      _src->setCurrentText(QString::fromUtf8(res) );
    }
  }
}

void ExportDialog::dstExtern()
{
  sc::String res;

  if( _p->selectPath( this, sc::String(_dst->currentText().utf8()), res, ExternProvider::Dir ) )
  {
    _dst->insertItem( QString::fromUtf8(res), 0 );
  }
}

ExportParam* ExportDialog::getParameters() const
{
  sc::String eol;
  if( _eol->currentText() == "default" )
  {
    eol = "";
  }
  else
  {
    eol = sc::String(_eol->currentText().utf8());
  }

  return new ExportParam(
    sc::String(_src->currentText().utf8()),
    sc::String(_dst->currentText().utf8()),
    svn::RevisionPtr(_srcRw->getRevision()),
    _force->isChecked(), eol );
}

QString ExportDialog::getSrcPathOrUrl()
{
  return _src->currentText();
}

QString ExportDialog::getDstPath()
{
  return _dst->currentText();
}

svn::Revision* ExportDialog::getRevision()
{
  return _srcRw->getRevision();
}

bool ExportDialog::getForce()
{
  return _force->isChecked();
}

QString ExportDialog::getEol()
{
  if( _eol->currentText() == "default" )
  {
    return "";
  }
  else
  {
    return _eol->currentText();
  }
}
