# -*- coding: utf-8 -*-
#
# test_keyring - tests for ubuntu_kylin_sso.keyring
#
# Author: Alejandro J. Cura <alecu@canonical.com>
# Author: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Tests for the keyring common module."""

from __future__ import unicode_literals

from twisted.trial.unittest import TestCase

from ubuntu_kylin_sso import keyring


class TestGetHostname(TestCase):
    """Test the function that gets the hostname."""

    def test_get_hostname(self):
        """The common case."""
        fake_hostname = "fake hostname"
        self.patch(keyring.socket, "gethostname", lambda: fake_hostname)
        self.assertEqual(keyring.gethostname(), fake_hostname)

    def test_get_hostname_uses_filesystem_encoding(self):
        """The fs encoding is used to decode the name returned by socket."""
        fake_hostname = "Привет-ПК"
        hostname_koi8r = fake_hostname.encode("koi8-r")
        self.patch(keyring.socket, "gethostname", lambda: hostname_koi8r)
        self.patch(keyring.sys, "getfilesystemencoding", lambda: "koi8-r")
        self.assertEqual(keyring.gethostname(), fake_hostname)


class TestTokenNameBuilder(TestCase):
    """Test the function that builds the token name."""

    def check_build(self, sample_app_name, sample_hostname, expected_result):
        """Check the build of a given token."""
        self.patch(keyring, "gethostname", lambda *a: sample_hostname)
        result = keyring.get_token_name(sample_app_name)
        self.assertEqual(result, expected_result)

    def test_get_simple_token_name(self):
        """A simple token name is built right."""
        sample_app_name = "UbuntuTwo"
        sample_hostname = "Darkstar"
        expected_result = "UbuntuTwo @ Darkstar"
        self.check_build(sample_app_name, sample_hostname, expected_result)

    def test_get_complex_token_name_for_app_name(self):
        """A complex token name is built right too."""
        sample_app_name = "Ubuntu @ Eleven"
        sample_hostname = "Mate+Cocido"
        expected_result = "Ubuntu @ Eleven @ Mate+Cocido"
        self.check_build(sample_app_name, sample_hostname, expected_result)

    def test_get_complex_token_name_for_hostname(self):
        """A complex token name is built right too."""
        sample_app_name = "Ubuntu Eleven"
        sample_hostname = "Mate @ Cocido"
        expected_result = "Ubuntu Eleven @ Mate AT Cocido"
        self.check_build(sample_app_name, sample_hostname, expected_result)

    def test_get_unicode_appname_token_name(self):
        """A token name with unicode in the app name."""
        sample_app_name = "Ubuntu 四百六十九"
        sample_hostname = "Darkstar"
        expected_result = "Ubuntu 四百六十九 @ Darkstar"
        self.check_build(sample_app_name, sample_hostname, expected_result)

    def test_get_utf8_hostname_token_name(self):
        """A token name with utf8 in the host name."""
        sample_app_name = "Ubuntu Eleven"
        sample_hostname = "Привет-ПК"
        expected_result = "Ubuntu Eleven @ Привет-ПК"
        self.check_build(sample_app_name, sample_hostname, expected_result)
