# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Test the SSOWizardPage and related."""

from ubuntu_kylin_sso.qt import PREFERED_UI_SIZE, sso_wizard_page as gui
from ubuntu_kylin_sso.qt.tests import (
    APP_NAME,
    BaseTestCase,
    build_string_for_pixels,
    PageBaseTestCase,
)


MAX_WIDTH = 100


class WizardHeaderTestCase(BaseTestCase):

    """Tests for injected Header in each Wizard Page."""

    kwargs = dict(max_width=MAX_WIDTH)
    ui_class = gui.WizardHeader
    ui_wizard_class = None

    def test_label_state(self):
        """Check the title and subtitle properties."""
        self.assertTrue(self.ui.title_label.wordWrap())
        self.assertTrue(self.ui.subtitle_label.wordWrap())
        self.assertFalse(self.ui.title_label.isVisible())
        self.assertFalse(self.ui.subtitle_label.isVisible())

    def test_set_title(self):
        """Check if set_title works ok, showing the widget if necessary."""
        max_width = self.ui.max_title_width
        text = build_string_for_pixels(self.ui.title_label, max_width)

        self.ui.set_title(text)

        self.assert_title_correct(self.ui.title_label, text, max_width)

    def test_set_elided_title(self):
        """Check if set_title adds the ellipsis when necessary."""
        # add an extra letter so we ensure this needs to be trimmed
        max_width = self.ui.max_title_width
        text = build_string_for_pixels(self.ui.title_label, max_width + 10)

        self.ui.set_title(text)

        self.assert_title_correct(self.ui.title_label, text, max_width)

    def test_set_empty_title(self):
        """Check if the widget is hidden for empty title."""
        self.ui.set_title('')

        self.assertEqual(self.ui.title_label.toolTip(), '')
        self.assertFalse(self.ui.title_label.isVisible())

    def test_set_subtitle(self):
        """Check if set_subtitle works ok, showing the widget if necessary."""
        max_width = self.ui.max_subtitle_width
        text = build_string_for_pixels(self.ui.subtitle_label, max_width)

        self.ui.set_subtitle(text)

        self.assert_subtitle_correct(self.ui.subtitle_label, text, max_width)

    def test_set_elided_subtitle(self):
        """Check if set_subtitle adds the ellipsis when necessary."""
        max_width = self.ui.max_subtitle_width
        text = build_string_for_pixels(self.ui.subtitle_label, max_width + 10)

        self.ui.set_subtitle(text)

        self.assert_subtitle_correct(self.ui.subtitle_label, text, max_width)

    def test_set_empty_subtitle(self):
        """Check if the widget is hidden for empty subtitle."""
        self.ui.set_subtitle('')

        self.assertEqual(self.ui.subtitle_label.toolTip(), '')
        self.assertFalse(self.ui.subtitle_label.isVisible())


class BaseWizardPageTestCase(PageBaseTestCase):

    """Tests for SSOWizardPage."""

    kwargs = {}
    ui_class = gui.BaseWizardPage

    def test_max_width(self):
        """The max_width is correct."""
        self.assertEqual(self.ui.max_width, 0)


class SSOWizardPageTestCase(BaseWizardPageTestCase):

    """Tests for SSOWizardPage."""

    kwargs = dict(app_name=APP_NAME)
    ui_class = gui.SSOWizardPage

    def test_max_width(self):
        """The max_width is correct."""
        self.assertEqual(self.ui.max_width, PREFERED_UI_SIZE['width'])

    def test_show_error(self):
        """Test show_error with a normal length string."""
        message = 'error-message'
        self.ui.show_error(message)

        self.assert_error_correct(self.ui.form_errors_label, message,
                                  self.ui.header.max_title_width)

    def test_show_error_long_text(self):
        """Test show_error with a long length string."""
        message = build_string_for_pixels(self.ui.form_errors_label,
                                          self.ui.header.max_title_width + 10)

        self.ui.show_error(message)
        self.assert_error_correct(self.ui.form_errors_label, message,
                                  self.ui.header.max_title_width)

    def test_hide_error(self):
        """Test show_error with a long length string."""
        self.ui.hide_error()

        self.assertEqual(self.ui.form_errors_label.text(), ' ')

    def test_setup_page_with_failing_backend(self):
        """Test how the ui react with an invalid backend."""
        self.patch(gui.main, "get_sso_client", lambda: None)
        self.patch(self.ui, "show_error", self._set_called)
        self.ui.setup_page()
        reason = 'There was a problem trying to setup the page %r' % self.ui
        expected = ((reason,), {})
        self.assertEqual(expected, self._called)
        self.assertFalse(self.ui.isEnabled())
