/*
 * Copyright (C) 2014 Canonical Ltd.
 *
 * This file is part of unity-chromium-extension
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "service.h"

#include <QDebug>
#include <QList>
#include <QMetaMethod>
#include <QMetaObject>

using namespace UnityWebapps;

namespace UnityWebapps {

class ServicePrivate
{
    Q_DECLARE_PUBLIC(Service)

public:
    inline ServicePrivate(Service *service);
    inline ~ServicePrivate();

    void setIdle(bool isIdle);

private:
    QList<QObject *> m_handlers;
    bool m_isIdle;
    mutable Service *q_ptr;
};

} // namespace

ServicePrivate::ServicePrivate(Service *service):
    m_isIdle(true),
    q_ptr(service)
{
}

ServicePrivate::~ServicePrivate()
{
}

void ServicePrivate::setIdle(bool isIdle)
{
    Q_Q(Service);

    if (m_isIdle == isIdle) return;
    m_isIdle = isIdle;
    Q_EMIT q->isIdleChanged();
}

Service::Service(QObject *parent):
    QObject(parent),
    d_ptr(new ServicePrivate(this))
{
}

Service::~Service()
{
    delete d_ptr;
}

bool Service::isIdle() const {
    Q_D(const Service);
    return d->m_isIdle;
}

void Service::addHandler(QObject *handler)
{
    Q_D(Service);

    d->m_handlers.append(handler);
}

void Service::handleMessage(const QVariantMap &message)
{
    Q_D(Service);

    QByteArray method = message.value(QStringLiteral("method")).toString().toLatin1();
    if (Q_UNLIKELY(method.isEmpty())) return;

    QByteArray methodSignature = method + "(QVariantMap)";

    d->setIdle(false);

    QVariantMap reply;
    bool ok = false;
    Q_FOREACH(QObject *handler, d->m_handlers) {
        const QMetaObject *metaObject = handler->metaObject();
        int index = metaObject->indexOfMethod(methodSignature);
        if (index == -1) continue;

        QMetaMethod metaMethod = metaObject->method(index);
        ok = metaMethod.invoke(handler,
                               Q_RETURN_ARG(QVariantMap, reply),
                               Q_ARG(QVariantMap, message));
        if (ok) break;
    }

    if (ok) {
        Q_EMIT messageHandled(reply);
    }

    d->setIdle(true);
}
