/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "PostureRetrievalThread.h"
#include "Credentials.h"
#include "AMTPPSharedData.h"
#include "AMTPPDefs.h"
#include "UNSEventLogger.h"
#include "UNSConfig.h"

// Set the static constants values.
const unsigned long PostureRetrievalThread::DEFAULT_RETRIEVAL_INTERVAL = 5 * MINUTE;
const unsigned long PostureRetrievalThread::HECI_DOWNTIME_ON_RESUME = 10 * SECOND;
const unsigned long PostureRetrievalThread::MIN_GET_POSTURE_INTERVAL = 1 * SECOND;
const unsigned long PostureRetrievalThread::MAX_GET_POSTURE_INTERVAL = 24 * 60 * MINUTE;

PostureRetrievalThread::PostureRetrievalThread(UNSEventLogger *pLogger/*=NULL*/) :
	PeriodicActionThread("PostureRetrieval", DEFAULT_RETRIEVAL_INTERVAL, HECI_DOWNTIME_ON_RESUME)
{
	m_pLogger = pLogger;
	m_EACWsmanClient.SetLogger(pLogger);
	m_State = INITIAL_CLEANUP_STATE;
}

void PostureRetrievalThread::SetLogger(UNSEventLogger *pLogger)
{
	m_pLogger = pLogger;
	m_EACWsmanClient.SetLogger(pLogger);
}

bool PostureRetrievalThread::init(const UNSConfig &Config)
{
	unsigned long nGetPostureInterval = Config.GetPostureInterval();
	if (nGetPostureInterval < MIN_GET_POSTURE_INTERVAL)
	{
		char logMsg[512];
		sprintf(logMsg, "GetPostureInterval was too low, %lu. Setting it to %lu.\n", nGetPostureInterval, MIN_GET_POSTURE_INTERVAL);
		m_pLogger->WarningLog(logMsg);
		nGetPostureInterval = MIN_GET_POSTURE_INTERVAL;
	}
	else if (nGetPostureInterval > MAX_GET_POSTURE_INTERVAL)
	{
		char logMsg[512];
		sprintf(logMsg, "GetPostureInterval was too high, %lu. Setting it to %lu.\n", nGetPostureInterval, MAX_GET_POSTURE_INTERVAL);
		m_pLogger->WarningLog(logMsg);
		nGetPostureInterval = MAX_GET_POSTURE_INTERVAL;
	}

	setActionInterval(nGetPostureInterval);

	return true;
}

void PostureRetrievalThread::SetCredentials(const Credentials &cred)
{
	m_EACWsmanClient.SetCredentials(cred);
}

void PostureRetrievalThread::cleanup()
{
	AMTPPSharedData SharedPosture(AMT_PP_REG_POSTURE);
	AMTPPSharedData SharedHash(AMT_PP_REG_POSTURE_HASH);

	if (SharedPosture.Remove())
		printf("Cleanup: Shared posture was removed\n");

	if (SharedHash.Remove())
		printf("Cleanup: Shared hash was removed\n");

	m_pLogger = NULL;
}

/*InitialCleanup is intended for a case that the service was killed, and we didn't have
a chance to remove the shared data.*/
PostureRetrievalThread::State PostureRetrievalThread::InitialCleanup()
{
	cleanup();
	return CheckEACEnablement(); //Proceed to check if EAC is enabled
}

PostureRetrievalThread::State PostureRetrievalThread::CheckEACEnablementFailure()
{
	return CheckEACEnablement(false);
}

PostureRetrievalThread::State PostureRetrievalThread::CheckEACEnablement(bool bLogFailure /*= true*/)
{
	bool bEACEnabled = false;
	if (!m_EACWsmanClient.IsEACEnabled(bEACEnabled))
	{
		//We couldn't get EAC status - next time we'll try again.
		if (bLogFailure && m_pLogger)
		{
			//Only log this message once before going into failure state
			m_pLogger->GetEACStatusFailedError();
		}
		return CHECK_EAC_ENABLEMENT_FAILURE_STATE;
	}

	if (bEACEnabled)
	{
		printf("EAC is enabled - trying to get posture.\n");
		//EAC is enabled - try to get posture now.
		return GetPosture();
	}
	else
	{
		printf("EAC is disabled - will check again if enabled next time.\n");
		//EAC is not enabled - next time we'll check again if it's enabled.
		return CHECK_EAC_ENABLEMENT_STATE;
	}
}

void PostureRetrievalThread::SetIP(const string &ip)
{
	m_EACWsmanClient.SetIP(ip);
	return;
}

PostureRetrievalThread::State PostureRetrievalThread::GetPosture()
{
	AMTPPSharedData SharedPosture(AMT_PP_REG_POSTURE, true, true);
	AMTPPSharedData SharedHash(AMT_PP_REG_POSTURE_HASH, true, true);

	/* Read Posture */
	unsigned char *pPosture = NULL;
	unsigned char *pHash = NULL;
	unsigned int nPostureLen = 0;
	unsigned int nHashLen = 0;

	if (!m_EACWsmanClient.GetPosture(&pPosture, nPostureLen, &pHash, nHashLen))
	{
		if (m_pLogger)
			m_pLogger->GetPostureFailedError();
		//Get posture failed - next time we'll check if EAC is enabled.
		return CHECK_EAC_ENABLEMENT_STATE;
	}

	/*Write Posture and Hash to registry*/
	if (!SharedPosture.Write(pPosture, nPostureLen, true))
	{
		printf("SharedPosture.Write failed. lastError=%d.\n", SharedPosture.LastError());
		if (m_pLogger)
			m_pLogger->WriteSharedDataError();
		return GETTING_POSTURE_STATE; //Stay in this state
	}
	printf("Shared Posture was written\n");

	if (!SharedHash.Write(pHash, nHashLen, true))
	{
		printf("SharedHash.Write failed. lastError=%d.\n", SharedHash.LastError());
		if (m_pLogger)
			m_pLogger->WriteSharedDataError();
		return GETTING_POSTURE_STATE; //Stay in this state
	}
	printf("Shared Hash was written\n");

	/*Hash and posture were successfully written, so we can release ownership of
	  them, and allow them to not be removed.*/
	SharedPosture.ReleaseOwnership();
	SharedHash.ReleaseOwnership();
	return GETTING_POSTURE_STATE;
}

void PostureRetrievalThread::action()
{
	switch (m_State)
	{
		case INITIAL_CLEANUP_STATE:
			m_State = InitialCleanup();
			break;

		case CHECK_EAC_ENABLEMENT_STATE:
			m_State = CheckEACEnablement();
			break;

		case CHECK_EAC_ENABLEMENT_FAILURE_STATE:
			m_State = CheckEACEnablementFailure();
			break;

		case GETTING_POSTURE_STATE:
			m_State = GetPosture();
			break;
	}
}
