/*
 *   This file is part of VBA Express.
 *
 *   Copyright (c) 2005-2006 Achraf cherti <achrafcherti@gmail.com>
 * 
 *   VBA Express is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   VBA Express is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with VBA Express; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

/*****************************************
 * Fichier Main du Configurateur de 
 * VisualBoyAvance
 *
 * Licence: GPL
 *
 * (c) Achraf cherti 2006
 * <achrafcherti@gmail.com>
 *
 *****************************************
 * Ce code source est distribue
 * selon la licence de la GNU GPL
 * version 2 ou ultrieure.
 *****************************************
 * http://vbaexpress.tuxfamily.org/
 *****************************************/

#include <stdlib.h>
#include "interface.h"
#include "vbaconfig.h"
#include "spath.h"
#include "astring.h"
#include "lcfg.h"
#include "sdl_key.h"
#include "Fl_Nice.H"
#include <FL/Fl.H>
#include <FL/Fl_Preferences.H>
#include <FL/Fl_File_Chooser.H>

//opendir/readdir/closedir
#include <sys/types.h>
#include <dirent.h>

//stat
#include <sys/stat.h>
#include <unistd.h>

#ifdef USE_FLU
#include <FLU/Flu_File_Chooser.h>
#endif

#include <sys/types.h> //fork
#include <unistd.h> // fork/exec*

InterfaceConfig *interfaceconfig=0;
Interface *I=0; //l'interface principale
About *about=0;
VbaConfig *vbaconfig=0; //Communication entre fentre et config...
AString *pVisualBoyAdvance_cfg=0;
AString *pVisualBoyAdvance_exec=0;
TextViewer *textviewer=0;
SkinManager *skin_manager=0; //fentre pour configurer les skins

int lang_index=0;

//les messages d'alertes
char *msg_trouve_pas_home="";
char *msg_vba_non_trouve="";
char *msg_vba_nexist="";
char *msg_save="";
char *msg_cfg_nexist_creer="";
char *msg_file_chooser_saveas="";
char *msg_file_chooser_open="";
char *msg_file_exists_replace="";
char *msg_error_creation_file="";
char *msg_cfg_important="";
char *msg_sure_effacer_chemin_sauvegardes="";
char *msg_sure_effacer_chemin_gameboy="";
char *msg_sure_effacer_chemin_captures="";
char *msg_sure_effacer_chemin_sauvegardes_dir="";
char *msg_sure_effacer_chemin_gameboy_dir="";
char *msg_sure_effacer_chemin_sauvegardes_vide="";
char *msg_sure_effacer_chemin_gameboy_vide="";
char *msg_sure_effacer_chemin_captures_dir="";
char *msg_sure_effacer_chemin_captures_vide="";
char *msg_file_chooser_dir="";
char *msg_error_open_dir="";
char *msg_please_select_skin="";

//quelques fonc
void show_licence_fr();
void show_licence_en();
void show_readme_fr(TextViewer *);
void show_readme_en(TextViewer *);
void load_skin();

/***********************************************************************
 * Fonctions de sauvegarde
 ***********************************************************************/
void save(const char *filename)
{
	vbaconfig->Get(I);
	//si quelqu'un l'efface alors que le programme tourne
	if(!sp_exists(filename)) {
		if(!fn_ask(msg_cfg_nexist_creer,filename))
			return;
	}
	//et maintenant il sauvegarde en se basant sur ce qu'il existe
	vbaconfig->SaveConfig(filename);
	//et dfini le tout comme tant non modifi
	I->changed=0;
}
void save()
{
	save(pVisualBoyAdvance_cfg->ptr());
}

void save_as()
{
	FN_PATTERN p[] = { {"Configuration","cfg"}, {0,0}};
	const char *filename = fn_save_chooser(msg_file_chooser_saveas,p,pVisualBoyAdvance_cfg->ptr());
	if(!filename) return;
	if(sp_exists(filename)) {
		if(!fn_ask(msg_file_exists_replace,filename)) 
			return;	
	}
	save(filename);
}

void ask_save()
{
	if(I->changed) {
		if(interfaceconfig->check_sauvegarde_automatique->value() || fn_ask(msg_save,pVisualBoyAdvance_cfg->ptr())) 
			save();
	}
}


/***********************************************************************
 * Translation vers la langue Anglaise
 ***********************************************************************/
void TranslateToEnglish()
{
	msg_please_select_skin="Please select a skin";
	msg_error_open_dir="The directory \"%s\"\nIs not exists.";

	skin_manager->btn_annuler->label("&Cancel");
	skin_manager->label_config_skin->label("Skin configuration");

	I->tab_affichage->label("Graphic");
	I->tab_son->label("Sound");
	I->tab_controle->label("Keys");
	I->tab_autre->label("Other");
	I->tab_path->label("Paths");

	I->check_plein_ecran->label("Fullscreen");
	I->check_plein_ecran->tooltip("Use fullscreen mode");
	
	I->label_interframe->label("Interframe");
	I->check_interframe_aucune->label("None");
	I->check_interframe_aucune->tooltip("Disable interframe blending");
	
	I->check_interframe_motion->label("Motion Blur");
	I->check_interframe_motion->tooltip("Motion Blur interframe blending");

	I->check_interframe_smart->label("Smart");
	I->check_interframe_smart->tooltip("Smart interframe blending");

	I->check_auto_skip->label("Auto Skip");
	I->check_auto_skip->tooltip("Enable/Disable auto frame skip");
	
	I->slide_frame_skip->label("Frame Skip");
	I->slide_frame_skip->tooltip("Emulation frame skip");

	I->slide_gb_skip->label("GB Frame Skip");
	I->slide_gb_skip->tooltip("Internal Gameboy frame skip settings");

	I->browser_filter->label("Gfx Filter:");

	I->check_son_active->label("Sound On");
	I->label_qualite_son->label("Quality");
	I->label_divers->label("Misc");
	I->check_reverse_stereo->label("Reverse Stereo");
	I->check_low_pass_filter->label("Low Pass Filter");
	I->check_echo->label("Echo");
	I->slide_volume->label("Volume:");

	I->label_screenshot->label("Screenshot");

	I->mnu_fichier->label("&File");
	I->mnu_ouvrir_jeu->label("&Open a Game");
	I->mnu_importer_config->label("&Import a configuration");
	I->mnu_enregistrer_config->label("&Save this configuration");
	I->mnu_enregistrer_config_sous->label("Save this &configuration AS...");
	I->mnu_quitter->label("&Quit");
	I->mnu_aide_menu->label("&Help");
	I->mnu_a_propos->label("&About VBA Express");

	I->mnu_lang->label("&Language");
	I->mnu_lang_fr->label("&Franais");
	I->mnu_lang_en->label("&English");

	I->btn_quitter->label("&Quit");
	I->btn_jouer->label("Play!");

	I->label_afficher_rapidite->label("Show Speed");
	I->radio_speed_none->label("None");
	I->radio_speed_none->tooltip("Disable showing speed");
	I->radio_speed_percentage->label("Percentage");
	I->radio_speed_percentage->tooltip("Show speed with a percentage");
	I->radio_speed_detailed->label("Detailed");
	I->radio_speed_detailed->tooltip("Show speed detailed");
	I->check_speed_transparent->label("Transparent");

	I->check_color_option->label("Color Option");
	I->check_color_option->tooltip("Enable washed colors");
	I->check_pause_si_inactif->label("Pause if inactive");
	I->check_agb_print->label("GameBoy Printer");

	I->check_disable_status->label("Disable status messages");
	I->check_disable_status->tooltip("Disable status messages");

	I->check_enable_mmx->label("MMX");
	I->check_enable_mmx->tooltip("Enable MMX support (optimisation)");

	I->check_enable_rtc->label("RTC");
	I->check_enable_rtc->tooltip("Enable GBA RTC support");
	I->check_enhanced_detection->label("E.Detection");
	I->check_enhanced_detection->tooltip("Enhanced save type Detection\n(this option is no longuer used...)");

	I->check_bordure_auto->label("Border Auto");
	I->check_bordure_auto->tooltip("Controls automatic Super Gameboy border");

	msg_trouve_pas_home="The $PATH environnement variable not exists...";
	msg_vba_non_trouve="VisualBoyAdvance not found in $PATH";
	msg_vba_nexist="WARNING: VisualBoyAdvance executable not found in $PATH";
	msg_save="Do you want to save your modifications in:\n%s\n??";
	msg_cfg_nexist_creer="The file:\n%s\nis not exist. Do you want do creat it?";
	msg_file_chooser_saveas="Select a file to write this configuration";
	msg_file_chooser_open="Select a file to open";
	msg_file_exists_replace="This file:\n%s\nexists. Replace it?";
	msg_error_creation_file="ERROR: Unable to create:\n%s";
	msg_cfg_important="VisualBoyAdvance.cfg is very important.\nThe programme can't continue if it doesn't exist.";

	//about
	about->image_gba->label("VBA Express is a configurator for Visual Boy Advance (a game boy advance emulator)");
	about->btn_plus_info->label("Click here for more info");
	about->label_auteur->label("Author");
	about->label_site->label("Site");
	about->label_titre_licence->label("Licence");
	about->label_licence->label("VBA Express is a free software. Its licence is the GNU General Public Licence version 2.");
	about->btn_lire_licence->label("Click here to read the licence");
	about->btn_fermer->label("Close");

	//interface config
	interfaceconfig->check_sauvegarde_automatique->label("Save configuration automatically");
	interfaceconfig->check_fenetre_centree->label("Center the window on start");
	interfaceconfig->check_sauvegarde_automatique->tooltip("");
	interfaceconfig->check_fenetre_centree->tooltip("");
	interfaceconfig->btn_ok->label("&Ok");
	interfaceconfig->btn_annuler->label("&Cancel");
	interfaceconfig->label_config_interface->label("Interface configuration");
	interfaceconfig->check_souvenir_onglet->label("Remember the selected tab");

	I->mnu_config->label("&Configuration");
	I->mnu_config_interface->label("&Interface");
	I->mnu_config_skin->label("&Skins");

	//concernant les chemins
	I->output_chemin_sauvegardes->label("Save Directory (save state)");
	I->output_chemin_gameboy->label("Game Boy Advance Saves (battery)");
	I->output_chemin_captures->label("Screenshots directory");
	I->output_chemin_bios->label("Bios file");
	I->btn_effacer_sauvegardes->tooltip("If you click here, all the saves (save state) will be in the same directory of the Roms (games)");
	I->btn_effacer_gameboy->tooltip("If you click here, all the saves (battery gameboy saves) will be in the same directory of the Roms (games)");
	I->btn_effacer_captures->tooltip("If you click here, all the captures (screenshots) will be in the same directory of the Roms (games)");

	msg_sure_effacer_chemin_sauvegardes="WARNING!!\nAre you sure to clean the save Directory (save state)?\n\nAll the saves will be in the same directory of the\nRoms (games) insteaded of:\n\n%s";
	msg_sure_effacer_chemin_gameboy="WARNING\nAre you sure to clean the Battery save directory?\n\nAll the GameBoy saves will be in the same directory of the\nRoms (games) insteaded of:\n\n%s";
	msg_sure_effacer_chemin_captures="Are you sure?\n\nAll captures will be in the same directory of\nthe Roms (games) insteaded of:\n\n%s";

	msg_sure_effacer_chemin_sauvegardes_vide="WARNING!!\n\nIf you choose a new save (save state) directory, all\nthe saves who are in the Roms (games) directory\nwill be ignored by VisualBoyAdvance...";
	msg_sure_effacer_chemin_gameboy_vide="WARNING!!\n\nIf you choose a new save (battery gameboy save) directory, all\nthe saves who are in the Roms (games) directory\nwill be ignored by VisualBoyAdvance...";
	msg_sure_effacer_chemin_sauvegardes_dir="WARNING!!\n\nAre you sre to change this directory?\nIf you do that, all the saves (save state) who are in:\n\n%s\n\nwill be ignored VisualBoyAdvance...";
	msg_sure_effacer_chemin_gameboy_dir="WARNING!!\n\nAre you sre to change this directory?\nIf you do that, All the saves (battery gameboy save) who are in:\n\n%s\n\nwill be ignored by VisualBoyAdvance...";
	msg_sure_effacer_chemin_captures_dir="WARNING!!\n\nAre you sure to change the capture directory?";
	msg_sure_effacer_chemin_captures_vide="WARNING!!\n\nAre you sure to change the capture directory?";

	msg_file_chooser_dir = "Select a directory...";

	lang_index=0;
}

void TranslateToFrench()
{
	msg_please_select_skin="Vous devez slectionner un skin";
	msg_error_open_dir="Erreur lors de l'ouverture du rpertoire\n%s";

	skin_manager->btn_annuler->label("&Annuler");
	skin_manager->label_config_skin->label("Revtement");
	
	I->tab_affichage->label("Affichage");
	I->tab_son->label("Son");
	I->tab_controle->label("Contrle");
	I->tab_path->label("Chemins");
	I->tab_autre->label("Autre");

	I->check_plein_ecran->label("Plein cran");
	I->check_plein_ecran->tooltip("Utiliser le mode plein cran\ns'il est dcoch, le jeu va apparatre\nen forme de fentre...");
	
	I->label_interframe->label("Interframe");
	I->check_interframe_aucune->label("Aucune");
	I->check_interframe_aucune->tooltip("Dsactiver l'affichage des interframes");
	
	I->check_interframe_motion->label("Motion Blur");
	I->check_interframe_motion->tooltip("Effet motion blur entre les frames, cela vous donne un effet trs lisse dans l'animations (un peu comme l'cran LCD de la game boy)");

	I->check_interframe_smart->label("Intelligent");
	I->check_interframe_smart->tooltip("Effet intelligent. VisualBoyAdvance dcide\ns'il doit ou non faire un Motion Blur\nentre les frames...");

	I->check_auto_skip->label("Auto Skip");
	I->check_auto_skip->tooltip("Activer ou dsactiver l'ajustement automatique des frames\nTrs utile si vous avez un ordinateur qui est trop rapide\net que vous souhaitez que VisualBoyAdvance vous calcule la\nVitesse favorable au lieu de la spcifier vous mme.");
	
	I->slide_frame_skip->label("Frame Skip");
	I->slide_frame_skip->tooltip("Le frame skip manuel.\nSi un jeu tourne trop lentement chez vous avec un Auto Skip essayer d'ajuster cette valeur...\nMais ATTENTION une grande valeur peut faire subir de grand \"SAUTS\" dans l'affichage.");

	I->slide_gb_skip->label("GB Frame Skip");
	I->slide_gb_skip->tooltip("Le frame skip de la Game Boy, peut tre aussi utilie si\nun tourne lentement chez vous\nAttention  ne pas mettre une grande valeur (qui pourrait rentre l'animation mauvaise)");

	I->browser_filter->label("Filtre Graphique:");

	I->check_son_active->label("Le son Activ");
	I->label_qualite_son->label("Qualit");
	I->label_divers->label("Autre");
	I->check_reverse_stereo->label("Reverse Stereo");
	I->check_low_pass_filter->label("Low Pass Filter");
	I->check_echo->label("Echo");
	I->slide_volume->label("Volume:");

	I->label_screenshot->label("Capture");

	I->mnu_fichier->label("&Fichier");
	I->mnu_ouvrir_jeu->label("&Ouvrir un jeu");
	I->mnu_importer_config->label("&Importer une configuration");
	I->mnu_enregistrer_config->label("&Enregistrer cette configuration");
	I->mnu_enregistrer_config_sous->label("Enregistrer cette &configuration sous...");
	I->mnu_quitter->label("&Quitter");
	I->mnu_aide_menu->label("&Aide");
	I->mnu_a_propos->label("&A propos de VbaExpress");

	I->mnu_lang->label("&Langage");
	I->mnu_lang_fr->label("&Franais");
	I->mnu_lang_en->label("&English");

	I->btn_quitter->label("&Quitter");
	I->btn_jouer->label("Jouer!");

	I->label_afficher_rapidite->label("Show Speed");
	I->radio_speed_none->label("None");
	I->radio_speed_none->tooltip("Ne pas afficher la rapidit de l'mulation");
	I->radio_speed_percentage->label("Pourcentage");
	I->radio_speed_percentage->tooltip("Afficher la rapidit du jeu avec un pourcentage\nNormalement 100% veut dire que le jeu tourne\navec une bonne rapidit\nTrs utile pour savoir si la configuration actuelle est adapte \nvotre ordinateur.");
	I->radio_speed_detailed->label("Dtaill");
	I->radio_speed_detailed->tooltip("Afficher tous les dtails, pourcentage et FPS (frame par seconde)\nUtile pour savoir si la configuration actuelle est adapte  votre ordinateur");
	I->check_speed_transparent->label("Transparent");

	I->check_color_option->label("Color Option");
	I->check_color_option->tooltip("Enable washed colors");
	I->check_pause_si_inactif->label("Pause si inactive");
	I->check_agb_print->label("Imprimante");

	I->check_disable_status->label("Dsactiver message status");
	I->check_disable_status->tooltip("Dsactiver les messages de status\ncomme par exemple quand vous faites un save state\navec ALT+F1 ou encore que le jeu a fait une sauvegarde\nIL EST DECONSEILLE DE CHOCHER CETTE CASE car les messages de status vous informent de plusieurs choses trs importantes.");

	I->check_enable_mmx->label("MMX");
	I->check_enable_mmx->tooltip("Activer les optimisation MMX\nSi cette option vous cause quelque chose d'anormal\nn'hsitez pas  la dsactiver.");

	I->check_enable_rtc->label("RTC");
	I->check_enable_rtc->tooltip("Activer le support GBA RTC");
	I->check_enhanced_detection->label("E.Detection");
	I->check_enhanced_detection->tooltip("Activer la dtection avance des sauvegardes");
	
	I->check_bordure_auto->label("Bordure Auto");
	I->check_bordure_auto->tooltip("Si cette case est coche VisualBoyAdvance va contrle automatiquement l'affichage de la bordure SuperGameboy");

	msg_trouve_pas_home="La variable d'environnement $PATH n'existe pas...";
	msg_vba_non_trouve="VisualBoyAdvance n'a pas t trouv dans $PATH";
	msg_vba_nexist="ATTENTION: Le fichier excutable VisualBoyAdvance n'a pas t trouv dans $PATH";
	msg_save="Voulez-vous enregistrer vos modifications dans le fichier:\n%s\n??";
	msg_cfg_nexist_creer="Le fichier:\n%s\nn'existe pas. Voulez vous le crer?";
	msg_file_chooser_saveas="Selectionnez un fichier pour enregistrer la configuration";
	msg_file_chooser_open="Selectionnez un fichier  ouvrir";
	msg_file_exists_replace="Le fichier:\n%s\nexiste. Voulez-vous le remplacer?";
	msg_error_creation_file="ERREUR, impossible de creer le fichier:\n%s";
	msg_cfg_important="Le fichier VisualBoyAdvance.cfg est trs important car il\nva contenir toute la configuration\nncessaire pour muler vos roms avec vos paramtres.\nLe programme ne peut continuer si ce fichier n'existe pas.";

	//about...
	about->image_gba->label("VBA Express est un logiciel qui vous facilitera la configuration de Visual Boy Advance sous Linux");
	about->btn_plus_info->label("Cliquez ici pour plus d'infos");
	about->label_auteur->label("Auteur");
	about->label_site->label("Site");
	about->label_titre_licence->label("Licence");
	about->label_licence->label("VBA Express est un logiciel libre distribu sous la licence GNU General Public Licence version 2.");
	about->btn_lire_licence->label("Cliquez ici pour lire la licence");
	about->btn_fermer->label("Fermer");

	//interface config
	interfaceconfig->check_sauvegarde_automatique->label("Sauvegarder la configuration automatiquement");
	interfaceconfig->check_fenetre_centree->label("Dmarrer fentre centre");
	interfaceconfig->check_sauvegarde_automatique->tooltip("Enregistrer automatiquement la configuration actuelle dans le fichier VisualBoyAdvance.cfg quand vous quittez le logiciel ou encore quand vous lancez un jeu avec le bouton \"jouer!\".");
	interfaceconfig->check_fenetre_centree->tooltip("Centrer automatiquement la fentre principale au dmarrage de l'application (au lieu de laisser le windows manager dcider par lui mme)");
	interfaceconfig->btn_ok->label("&Ok");
	interfaceconfig->btn_annuler->label("&Annuler");
	interfaceconfig->label_config_interface->label("Configurer L'interface");
	interfaceconfig->check_souvenir_onglet->label("Se souvenir de l'onglet slectionn");

	//config
	I->mnu_config->label("&Configuration");
	I->mnu_config_skin->label("&Revtement");

	//concernant les chemins
	I->output_chemin_sauvegardes->label("Chemins des sauvegardes (save state)");
	I->output_chemin_gameboy->label("Chemins des sauvegardes game boy (battery)");
	I->output_chemin_captures->label("Chemin des captures d'cran");
	I->output_chemin_bios->label("Chemin du bios");
	I->btn_effacer_sauvegardes->tooltip("Si vous cliquez ici, toutes les sauvegardes (save state) seront localises dans le mme rpertoire que les Roms (jeux)");
	I->btn_effacer_gameboy->tooltip("Si vous cliquez ici, toutes les sauvegardes (game boy) seront mises dans le mme rpertoire que les Roms (jeux)");
	I->btn_effacer_captures->tooltip("Si vous cliquez ici, toutes les captures d'cran seront mises dans le rpertoire ou sont les Roms (jeux)");

	msg_sure_effacer_chemin_sauvegardes="ATTENTION!\n\nSi vous videz le chemin des sauvegardes (save state) et que vous\navez des sauvegardes dans:\n\n%s\n\nVisualBoyAdvance va les ignorer...";
	msg_sure_effacer_chemin_gameboy="ATTENTION!\n\nSi vous videz le chemin des sauvegardes (save state) et que vous\navez des sauvegardes dans:\n\n%s\n\nVisualBoyAdvance va les ignorer...";
	msg_sure_effacer_chemin_captures="tes vous sre de changer le chemin des captures?\nLes captures ne se feront plus dans:\n%s";

	msg_file_chooser_dir = "Selectionnez un rpertoire...";

	msg_sure_effacer_chemin_sauvegardes_dir="ATTENTION!!\n\ntes vous sre de changer le rpertoire?\n\nSi vous avez dj des sauvegardes (save state) dans:\n\n%s\n\nsachez que VisualBoyAdvance va les ignorer...";
	msg_sure_effacer_chemin_gameboy_dir="ATTENTION!!\n\ntes vous sre de changer le rpertoire?\n\nSi vous avez dj des sauvegardes (batterie game boy) dans:\n\n%s\n\nsachez que VisualBoyAdvance va les ignorer...";
	msg_sure_effacer_chemin_sauvegardes_vide="ATTENTION!!\n\ntes vous sre de choisir un autre rpertoire?\n\nSi vous avez dj des sauvegardes (save state) dans le mme\nrpertoire que les Roms sachez que VisualBoyAdvance va les ignorer...";
	msg_sure_effacer_chemin_gameboy_vide="ATTENTION!!\n\ntes vous sre de changer le rpertoire?\n\nSi vous avez dj des sauvegardes (batterie game boy) dans le mme\nrpertoire que les Roms sachez que VisualBoyAdvance va les ignorer...";
	msg_sure_effacer_chemin_captures_dir="ATTENTION!!\n\nLes captures ne se feront plus dans:\n\n%s";
	msg_sure_effacer_chemin_captures_vide="ATTENTION!!\n\ntes vous sre de mettre un autre chemin pour les captures?\n(autre que le rpertoire ou sont des Roms)";

	lang_index=1;
}

//quelques fonctions utiles pour le fichier de configuration
int get_language()
{
	return lang_index;
}

void set_language(int i)
{
	if(i==0) TranslateToEnglish();
	else if(i==1) TranslateToFrench();
	I->MainWindow->redraw();
	Fl::check();
}

/***********************************************************************
 * Centrer une fentre
 ***********************************************************************/
void center(Fl_Window *window)
{
	window->position(
		/* x */			
		Fl::x()+
		(Fl::w()-window->w())/2

		/* y */ ,
		Fl::y()+
		(Fl::h()-window->h())/2
	);
}

/***********************************************************************
 * center une fenetre fille selon sa mre
 ***********************************************************************/
void center_in_window(Fl_Window *pere, Fl_Window *fils)
{
	fils->position(
			/* x */
			pere->x() + ((pere->w()-fils->w())/2),
			/* y */
			pere->y() + ((pere->h()-fils->h())/2)
	);
}

/***********************************************************************
 * A propos de...
 ***********************************************************************/
void mnu_a_propos_click(Fl_Widget *)
{
	center_in_window(I->MainWindow,about->MainWindow);
	about->show();
	I->dommage();
}

/***********************************************************************
 * Menu ou bouton quitter
 ***********************************************************************/
void mnu_quitter_click(Fl_Widget *)
{
	exit(0);
}

/***********************************************************************
 * Retourne le numro d'Onglet Actuel
 ***********************************************************************/
int get_onglet_index()
{
	if(I->onglets->value()==I->tab_affichage) return 0;
	if(I->onglets->value()==I->tab_son) return 1;
	if(I->onglets->value()==I->tab_controle) return 2;
	if(I->onglets->value()==I->tab_autre) return 3;
	if(I->onglets->value()==I->tab_path) return 4;
	return 0;
}

void set_onglet_index(int i)
{
	if(i==0) I->onglets->value(I->tab_affichage);
	if(i==1) I->onglets->value(I->tab_son);
	if(i==2) I->onglets->value(I->tab_controle);
	if(i==3) I->onglets->value(I->tab_autre);
	if(i==4) I->onglets->value(I->tab_path);
}

/***********************************************************************
 * SAVE/Load Configuration (config de l'interface)
 ***********************************************************************/
#define CONFIG_ROOT        Fl_Preferences::USER
#define CONFIG_VENDOR      "AchrafCherti"
#define CONFIG_APPLICATION "vbaexpress"

//Load
void load_config()
{
	int i;
	static int first=1;

	//on charge le skin
	load_skin();

	Fl_Preferences *pref = new Fl_Preferences(CONFIG_ROOT,CONFIG_VENDOR,CONFIG_APPLICATION);
	
	pref->get("souvenir_onglet",i,1);
	interfaceconfig->check_souvenir_onglet->value(i); 

	if(first) {
		//si il doit se souvenir de l'onglet
		if(i) {
			pref->get("onglet",i,0);
			set_onglet_index(i);
		}

		pref->get("lang_index",i,0);
		set_language(i);
	}

	//les autres trucs qui peuvent se charger  chaque fois :-)
	
	pref->get("fenetre_centree",i,0);
	if(first && i) center(I->MainWindow);
	interfaceconfig->check_fenetre_centree->value(i);
	
	pref->get("sauvegarde_automatique",i,0);
	interfaceconfig->check_sauvegarde_automatique->value(i);

	delete pref;
	first=0;
}

//Save
void save_config()
{
	Fl_Preferences *pref = new Fl_Preferences(CONFIG_ROOT,CONFIG_VENDOR,CONFIG_APPLICATION);

	pref->set("lang_index",get_language());

	pref->set("fenetre_centree", interfaceconfig->check_fenetre_centree->value()?1:0);
	pref->set("sauvegarde_automatique", interfaceconfig->check_sauvegarde_automatique->value()?1:0);
	pref->set("souvenir_onglet", interfaceconfig->check_souvenir_onglet->value()?1:0);
	if(interfaceconfig->check_souvenir_onglet->value())
		pref->set("onglet",get_onglet_index());
	
	delete pref;
}

/***********************************************************************
 * Pour trouver VisualBoyAdvance dans $PATH
 * retourne true si erreur
 ***********************************************************************/
bool chercher_VisualBoyAdvance()
{
	//cherche le fichier excutable de visualboyadvance
	pVisualBoyAdvance_exec = new AString;
	char *exec = sp_which("VisualBoyAdvance");
	if(exec) {
		*pVisualBoyAdvance_exec = exec;
		free(exec);
		return false;
	}
	else {
		fn_alert(msg_vba_non_trouve);
	}
	return true;
}

/***********************************************************************
 * Menu pour enregistrer la configuration
 ***********************************************************************/
void mnu_enregistrer_config_click(Fl_Widget *)
{
	save();
	I->dommage();
}

void mnu_enregistrer_config_sous_click(Fl_Widget *)
{
	save_as();
	I->dommage();
}

/***********************************************************************
 * Play a game
 ***********************************************************************/
void play_game(const char *filename)
{
	//s'il n'existe pas
	if(!sp_exists(pVisualBoyAdvance_exec->ptr())) {
		if(chercher_VisualBoyAdvance()) {
			fn_alert(msg_vba_nexist);
			return;
		}
	}

	//excute la commande...
	pid_t pid=fork();
	if(pid==0) { //le fils
		execlp(pVisualBoyAdvance_exec->ptr(),pVisualBoyAdvance_exec->ptr(),"-c",pVisualBoyAdvance_cfg->ptr(),filename,NULL);
		exit(1);
	}
	else if(pid<0) {
		fn_alert("Erreur lors de la cration du processus fils...");
	}
}

void mnu_importer_config_click(Fl_Widget *)
{
	FN_PATTERN p[] = {{"Configuration","cfg"},{0,0}};
	const char *filename = fn_file_chooser(msg_file_chooser_open,p,pVisualBoyAdvance_cfg->ptr());
	if(filename && sp_exists(filename)) {
		vbaconfig->LoadConfig(filename);
		vbaconfig->Put(I);
	}
	I->dommage();
}

/***********************************************************************
 * tout redessiner aprs 1 seconde
 ***********************************************************************/
void redraw_callback(void *)
{
	I->MainWindow->redraw();

	//profiter pour sauver si c'est possible toutes les ~5 secondes
	if(I->changed) {
		static int i=0;
		i++;
		
		//si on doit faire une sauvegarde automatique
		if(i>5 && interfaceconfig->check_sauvegarde_automatique->value()) {
			save();
			i=0;
		}
	}

	Fl::repeat_timeout(1.0,redraw_callback);
}

/***********************************************************************
 * Fonctions pour changer la langage
 ***********************************************************************/
void mnu_lang_fr_callback(Fl_Widget *)
{ TranslateToFrench(); }

void mnu_lang_en_callback(Fl_Widget *)
{ TranslateToEnglish(); }

/***********************************************************************
 * Pour ouvrir directement un jeu
 ***********************************************************************/
void mnu_ouvrir_jeu_click(Fl_Widget *)
{
	char *precedente_rom="";

	//il demande d'enregistrer avant de lancer le jeu
	ask_save();
	
	//charger la prcdente rom
	Fl_Preferences *pref = new Fl_Preferences(CONFIG_ROOT,CONFIG_VENDOR,CONFIG_APPLICATION);
	pref->get("precedente_rom",precedente_rom,"");

	//charger le filename
#if USE_FLU
	const char *filename = flu_file_chooser(msg_file_chooser_open,"GBA Rom (*.{zip,gba*})",precedente_rom);
#else
	FN_PATTERN p[] = {{"GBA Rom","gba*"},{"Zip","zip"},{0,0}};
	const char *filename = fn_file_chooser(msg_file_chooser_open,p,precedente_rom);
#endif

	//enregistrer le prcdent fichier
	if(filename)
		pref->set("precedente_rom",filename);

	delete pref;

	//et dsalloque le tout
	free(precedente_rom);

	//jouer!
	if(filename) play_game(filename);
}

/***********************************************************************
 * Lire la licence et plus d'infos dans about
 ***********************************************************************/
void mnu_plus_info_click(Fl_Widget *)
{
	if(lang_index==1) { //pour le franais
		show_readme_fr(textviewer);
	}
	else {
		show_readme_en(textviewer);
	}
}

void mnu_lire_licence_click(Fl_Widget *)
{
	if(lang_index==1) {
		show_licence_fr();
	}
	else {
		show_licence_en();
	}
}

/***********************************************************************
 * Load skin (on charge le skin depuis le fichier pref)
 ***********************************************************************/
#define SKIN_DIR "/usr/share/vbaexpress/skin"
void load_skin()
{
	AString tmp_str;

	//on charge le nom du skin du fichier pref
	Fl_Preferences *pref = new Fl_Preferences(CONFIG_ROOT,CONFIG_VENDOR,CONFIG_APPLICATION);
	char *skin=0;
	pref->get("skin",skin,"");
	delete pref;

	//au cas ou...
	if(!skin) return;

	//on va crer le chemin des skins
	tmp_str << SKIN_DIR << "/" << skin << ".png";

	//on change le skin
	I->change_skin(tmp_str.ptr());
}

/***********************************************************************
 * Config skin
 ***********************************************************************/
void mnu_config_skin_click(Fl_Widget *)
{
	center_in_window(I->MainWindow,skin_manager->MainWindow);

	//on charge la liste des skins
	DIR *d = opendir(SKIN_DIR);
	if(!d) {
		fn_alert(msg_error_open_dir,SKIN_DIR);
		return;
	}

	//la liste
	AString tmp_str;
	
	struct dirent *_dirent;

	skin_manager->list->clear();
	skin_manager->list->add("Par dfaut");

	//lecture de tous les fichiers
	while((_dirent=readdir(d))) {
		tmp_str=SKIN_DIR;
		tmp_str << "/";
		tmp_str << _dirent->d_name;
		
		//test si c'est un fichier
		static struct stat st;
		if(stat(tmp_str.ptr(),&st)) 
			continue;
		
		//si c'est un fichier texte
		if(!S_ISREG(st.st_mode)) 
			continue;

		//test extention
		tmp_str=_dirent->d_name;
		if(! (tmp_str.length()>4 && !strcmp(tmp_str.ptr()+tmp_str.length()-4 ,".png"))) //si ce n'est pas png, continue
			continue;

		//en enlve l'extention
		tmp_str.substr(0,tmp_str.length()-4);

		//on ajoute l'lment dans liste
		skin_manager->list->add(tmp_str.ptr());
	}

	//on ferme le rpertoire
	closedir(d);

	//on charge le skin actuellement slectionn
	Fl_Preferences *pref = new Fl_Preferences(CONFIG_ROOT,CONFIG_VENDOR,CONFIG_APPLICATION);
	char *actual_skin=0;
	pref->get("skin",actual_skin,"");
	delete pref;

	skin_manager->list->value(1); //on slectionne le premier (par dfaut)

	//on cherche si actual_skin existe
	if(actual_skin) {
		int i;
		const char *str;
		for(i=0;i<skin_manager->list->size();i++) {
			str = skin_manager->list->text(i+1);
			if(!strcmp(actual_skin,str)) {
				skin_manager->list->value(i+1);
				break;
			}
		}
		//on doit de toute faon slectionner un skin!
		free(actual_skin);
	}
	
	//maintenant on affiche notre fentre de skins
	if(skin_manager->show(msg_please_select_skin)) {
		//on sauvegarde le skin slectionn
		const char *skin = skin_manager->list->text(skin_manager->list->value());

		//aucun skin
		if(!skin) return;
		if(skin_manager->list->value()==0) 
			return;
		//skin par dfaut
		if(skin_manager->list->value()==1) {
			I->default_skin();
			skin=""; //pour enregistrer le ""
		}
		//autre skin personnalis
		else {
			tmp_str = SKIN_DIR;
			tmp_str << "/";
			tmp_str << skin;
			tmp_str << ".png";
			I->change_skin(tmp_str.ptr());
		}

		if(skin) {
			Fl_Preferences *pref = new Fl_Preferences(CONFIG_ROOT,CONFIG_VENDOR,CONFIG_APPLICATION);
			pref->set("skin",skin);
			delete pref;
		}
	}

}

/***********************************************************************
 * Config interface
 ***********************************************************************/
void mnu_config_interface_click(Fl_Widget *)
{
	center_in_window(I->MainWindow,interfaceconfig->MainWindow);
	load_config();
	if(interfaceconfig->show())
		save_config();
}

/***********************************************************************
 * toute ce qui concerne les chemins
 ***********************************************************************/
void btn_chemin_sauvegardes_click(Fl_Widget *)
{
	const char *filename;
	if(*I->output_chemin_sauvegardes->value()) {
		if(!fn_ask(msg_sure_effacer_chemin_sauvegardes_dir,I->output_chemin_sauvegardes->value())) return;
	}
	else {
		if(!fn_ask(msg_sure_effacer_chemin_sauvegardes_vide,I->output_chemin_sauvegardes->value())) return;
	}

	filename = fn_dir_chooser(msg_file_chooser_dir,I->output_chemin_sauvegardes->value());
	if(filename) {
		I->changed=1;
		I->output_chemin_sauvegardes->value(filename);
	}
}

void btn_chemin_bios_click(Fl_Widget *)
{
	FN_PATTERN pattern[] = {{"All Files","*"},{0,0}};
	const char *filename = fn_file_chooser(msg_file_chooser_dir,pattern,I->output_chemin_bios->value());
	if(filename) {
		I->changed=1;
		I->output_chemin_bios->value(filename);
	}
}

void btn_chemin_captures_click(Fl_Widget *)
{
	const char *filename;
	if(*I->output_chemin_captures->value()) {
		if(!fn_ask(msg_sure_effacer_chemin_captures_dir,I->output_chemin_captures->value())) return;
	}
	else {
		if(!fn_ask(msg_sure_effacer_chemin_captures_vide,I->output_chemin_captures->value())) return;
	}
	
	filename = fn_dir_chooser(msg_file_chooser_dir,I->output_chemin_captures->value());
	if(filename) {
		I->changed=1;
		I->output_chemin_captures->value(filename);
	}
}

void btn_chemin_gameboy_click(Fl_Widget *)
{
	const char *filename;
	if(*I->output_chemin_gameboy->value()) {
		if(!fn_ask(msg_sure_effacer_chemin_gameboy_dir,I->output_chemin_gameboy->value())) return;
	}
	else {
		if(!fn_ask(msg_sure_effacer_chemin_gameboy_vide,I->output_chemin_gameboy->value())) return;
	}
	filename = fn_dir_chooser(msg_file_chooser_dir,I->output_chemin_gameboy->value());
	if(filename) {
		I->changed=1;
		I->output_chemin_gameboy->value(filename);
	}
}

void btn_effacer_sauvegardes_click(Fl_Widget *)
{
	if(*I->output_chemin_sauvegardes->value() && fn_ask(msg_sure_effacer_chemin_sauvegardes,I->output_chemin_sauvegardes->value())) {
		I->changed=1;
		I->output_chemin_sauvegardes->value("");
	}
}

void btn_effacer_gameboy_click(Fl_Widget *)
{
	if(*I->output_chemin_gameboy->value() && fn_ask(msg_sure_effacer_chemin_gameboy,I->output_chemin_gameboy->value())){
		I->output_chemin_gameboy->value("");
		I->changed=1;
	}
}

void btn_effacer_captures_click(Fl_Widget *)
{
	if(*I->output_chemin_captures->value() && fn_ask(msg_sure_effacer_chemin_captures,I->output_chemin_captures->value())) {
		I->changed=1;
		I->output_chemin_captures->value("");
	}
}

void btn_effacer_bios_click(Fl_Widget *)
{
	if(*I->output_chemin_bios->value()) {
		I->changed=1;
		I->output_chemin_bios->value("");
	}
}

/***********************************************************************
 * Premire Initialisation de l'interface
 ***********************************************************************/
void init(int argc, char **argv)
{
	//Initialisation de l'interface
	I = new Interface;
	about=new About;

	//skin manager
	skin_manager = new SkinManager;

	//le text viewer
	textviewer = new TextViewer;

	//interface config
	interfaceconfig = new InterfaceConfig;

	//timer
	Fl::add_timeout(1.0,redraw_callback);

	//les Callback
	I->mnu_ouvrir_jeu->callback(mnu_ouvrir_jeu_click);
	I->btn_jouer->callback(mnu_ouvrir_jeu_click);
	I->mnu_a_propos->callback(mnu_a_propos_click);
	I->mnu_quitter->callback(mnu_quitter_click);
	I->btn_quitter->callback(mnu_quitter_click);
	I->MainWindow->callback(mnu_quitter_click);
	I->mnu_enregistrer_config->callback(mnu_enregistrer_config_click);
	I->mnu_enregistrer_config_sous->callback(mnu_enregistrer_config_sous_click);
	I->mnu_importer_config->callback(mnu_importer_config_click);
	I->mnu_lang_fr->callback(mnu_lang_fr_callback);
	I->mnu_lang_en->callback(mnu_lang_en_callback);
	about->btn_plus_info->callback(mnu_plus_info_click);
	about->btn_lire_licence->callback(mnu_lire_licence_click);
	I->mnu_config_interface->callback(mnu_config_interface_click);
	I->mnu_config_skin->callback(mnu_config_skin_click);

	//les chemins
	I->btn_chemin_sauvegardes->callback(btn_chemin_sauvegardes_click);
	I->btn_chemin_captures->callback(btn_chemin_captures_click);
	I->btn_chemin_gameboy->callback(btn_chemin_gameboy_click);
	I->btn_effacer_sauvegardes->callback(btn_effacer_sauvegardes_click);
	I->btn_effacer_captures->callback(btn_effacer_captures_click);
	I->btn_effacer_gameboy->callback(btn_effacer_gameboy_click);

	//chemin du bios
	I->btn_effacer_bios->callback(btn_effacer_bios_click);
	I->btn_chemin_bios->callback(btn_chemin_bios_click);
	
	//Initialisation de la configuration
	vbaconfig = new VbaConfig;
	vbaconfig->Put(I); //Le premier changement de configuration par dfaut

	//changement=aucun
	I->changed=0;

	//chargement de configuration de l'interface
	load_config();

	//fonction pour rechercher VisualBoyAdvance et
	//puis le mettre dans spVisualBoyAdvance
	chercher_VisualBoyAdvance();

	//cherche le fichier VisualBoyAdvance.cfg principal
	pVisualBoyAdvance_cfg = new AString;
	char *PATH = getenv("HOME");
	if(!PATH) {
		fn_alert(msg_trouve_pas_home);
		exit(1);
	}
	else {
		*pVisualBoyAdvance_cfg << PATH << "/VisualBoyAdvance.cfg";
	}

	// le petit message pour crer visual*cfg
	if(!sp_exists((*pVisualBoyAdvance_cfg).ptr())) {
		if(fn_ask(msg_cfg_nexist_creer,pVisualBoyAdvance_cfg->ptr())) {
			if(vbaconfig->CreateConfig(pVisualBoyAdvance_cfg->ptr())) {
				fn_alert(msg_error_creation_file);
				exit(1);
			}
		}
		else {
			fn_alert(msg_cfg_important);
			exit(1);
		}
	}
	
	//chargement du fichier config :-)
	vbaconfig->LoadConfig(pVisualBoyAdvance_cfg->ptr());
	vbaconfig->Put(I);

	//changement des couleurs
	Fl::background(0xEE,0xEE,0xEE);

	//affichage
	I->show(argc,argv);
	I->MainWindow->redraw();

	//changements
	I->changed=0;
}

/***********************************************************************
 * bye
 ***********************************************************************/
void bye()
{
	static bool first=true;

	//on ne fait un free que lors du premier appel
	if(first) {
		first=false;
		
		if(pVisualBoyAdvance_cfg){
			save_config(); //sauvegarde avant de quitter
			ask_save();
		}
		delete I;
		delete interfaceconfig;
		delete textviewer;
		delete about;
		delete vbaconfig;
		delete pVisualBoyAdvance_cfg;
		delete pVisualBoyAdvance_exec;
		delete skin_manager;
	}
}

/***********************************************************************
 * Main (fonction principale)
 ***********************************************************************/
int main(int argc, char **argv)
{
	//atexit (important)
	atexit(bye);
	
	//width et height
	vba_getkey_wh(320,200);

	//initialisation
	lcfg_set_uncomment_var(1);
	lcfg_set_autoindent(0);
	init(argc,argv);

	//fltk
	return Fl::run();
}

/***********************************************************************
 * Affichage de la licence 
 ***********************************************************************/
#define ADD(s) textviewer->browser->add(s);
void show_licence_fr()
{
	textviewer->browser->clear();
	ADD("GPL");
	ADD("");
	ADD("Introduction");
	ADD("");
	ADD("This is an unofficial translation of the GNU General Public License into French. It was not published by the Free Software Foundation, and does not legally");
	ADD("state the distribution terms for software that uses the GNU GPL--only the original English text of the GNU GPL does that. However, we hope that this");
	ADD("translation will help French speakers understand the GNU GPL better. ");
	ADD("");
	ADD("Voici (http://www.linux-france.org/article/these/gpl.html) une adaptation non officielle de la Licence Publique Gnrale du projet GNU. Elle n'a pas t");
	ADD("publie par la Free Software Foundation et son contenu n'a aucune porte lgale car seule la version anglaise de ce document dtaille le mode de distribution");
	ADD("des logiciels sous GNU GPL. Nous esprons cependant qu'elle permettra aux francophones de mieux comprendre la GPL. ");
	ADD("");
	ADD("Licence Publique Gnrale GNU Version 2, Juin 1991");
	ADD("");
	ADD("Copyright  Free Software Foundation, Inc.");
	ADD("59 Temple Place, Suite 330, Boston, MA 02111-1307");
	ADD("tats-Unis, 1989, 1991.");
	ADD("La copie et la distribution de copies exactes de ce document sont autorises, mais aucune modification n'est permise. ");
	ADD("");
	ADD("Prambule");
	ADD("");
	ADD("Les licences d'utilisation de la plupart des programmes sont dfinies pour limiter ou supprimer toute libert  l'utilisateur.  l'inverse, la Licence Publique");
	ADD("Gnrale (General Public License) est destine  vous garantir la libert de partager et de modifier les logiciels libres, et de s'assurer que ces logiciels sont");
	ADD("effectivement accessibles  tout utilisateur.");
	ADD("");
	ADD("Cette Licence Publique Gnrale s'applique  la plupart des programmes de la Free Software Foundation, comme  tout autre programme dont l'auteur l'aura");
	ADD("dcid (d'autres logiciels de la FSF sont couverts pour leur part par la Licence Publique Gnrale pour Bibliothques GNU (LGPL)). Vous pouvez aussi");
	ADD("appliquer les termes de cette Licence  vos propres programmes, si vous le dsirez.");
	ADD("");
	ADD("Libert des logiciels ne signifie pas ncessairement gratuit. Notre Licence est conue pour vous assurer la libert de distribuer des copies des programmes,");
	ADD("gratuitement ou non, de recevoir le code source ou de pouvoir l'obtenir, de modifier les programmes ou d'en utiliser des lments dans de nouveaux");
	ADD("programmes libres, en sachant que vous y tes autoris.");
	ADD("");
	ADD("Afin de garantir ces droits, nous avons d introduire des restrictions interdisant  quiconque de vous les refuser ou de vous demander d'y renoncer. Ces");
	ADD("restrictions vous imposent en retour certaines obligations si vous distribuez ou modifiez des copies de programmes protgs par la Licence. En d'autre termes,");
	ADD("il vous incombera en ce cas de : ");
	ADD("");
	ADD("    transmettre aux destinataires tous les droits que vous possdez, ");
	ADD("    expdier aux destinataires le code source ou bien tenir celui-ci  leur disposition, ");
	ADD("    leur remettre cette Licence afin qu'ils prennent connaissance de leurs droits. ");
	ADD("");
	ADD("Nous protgeons vos droits de deux faons : d'abord par le copyright du logiciel, ensuite par la remise de cette Licence qui vous autorise lgalement  copier,");
	ADD("distribuer et/ou modifier le logiciel.");
	ADD("");
	ADD("En outre, pour protger chaque auteur ainsi que la FSF, nous affirmons solennellement que le programme concern ne fait l'objet d'aucune garantie. Si un tiers");
	ADD("le modifie puis le redistribue, tous ceux qui en recevront une copie doivent savoir qu'il ne s'agit pas de l'original afin qu'une copie dfectueuse n'entache pas");
	ADD("la rputation de l'auteur du logiciel.");
	ADD("");
	ADD("Enfin, tout programme libre est sans cesse menac par des dpts de brevets. Nous souhaitons  tout prix viter que des distributeurs puissent dposer des");
	ADD("brevets sur les Logiciels Libres pour leur propre compte. Pour viter cela, nous stipulons bien que tout dpt ventuel de brevet doit accorder expressment ");
	ADD("tout un chacun le libre usage du produit.");
	ADD("");
	ADD("Les dispositions prcises et les conditions de copie, de distribution et de modification de nos logiciels sont les suivantes : ");
	ADD("");
	ADD("Stipulations et conditions relatives  la copie, la distribution et la modification");
	ADD("");
	ADD("");
	ADD("");
	ADD("    Article 0");
	ADD("    La prsente Licence s'applique  tout Programme (ou autre travail) o figure une note, place par le dtenteur des droits, stipulant que ledit Programme");
	ADD("    ou travail peut tre distribu selon les termes de la prsente Licence. Le terme Programme dsigne aussi bien le Programme lui-mme que tout travail");
	ADD("    qui en est driv selon la loi, c'est--dire tout ouvrage reproduisant le Programme ou une partie de celui-ci,  l'identique ou bien modifi, et/ou traduit");
	ADD("    dans une autre langue (la traduction est considre comme une modification). Chaque personne concerne par la Licence Publique Gnrale sera");
	ADD("    dsigne par le terme Vous.");
	ADD("");
	ADD("    Les activits autres que copie, distribution et modification ne sont pas couvertes par la prsente Licence et sortent de son cadre. Rien ne restreint");
	ADD("    l'utilisation du Programme et les donnes issues de celui-ci ne sont couvertes que si leur contenu constitue un travail bas sur le logiciel");
	ADD("    (indpendemment du fait d'avoir t ralis en lanant le Programme). Tout dpend de ce que le Programme est cens produire.");
	ADD("");
	ADD("");
	ADD("    Article 1.");
	ADD("    Vous pouvez copier et distribuer des copies conformes du code source du Programme, tel que Vous l'avez reu, sur n'importe quel support,  condition");
	ADD("    de placer sur chaque copie un copyright appropri et une restriction de garantie, de ne pas modifier ou omettre toutes les stipulations se rfrant  la");
	ADD("    prsente Licence et  la limitation de garantie, et de fournir avec toute copie du Programme un exemplaire de la Licence.");
	ADD("");
	ADD("    Vous pouvez demander une rtribution financire pour la ralisation de la copie et demeurez libre de proposer une garantie assure par vos soins,");
	ADD("    moyennant finances.");
	ADD("");
	ADD("");
	ADD("    Article 2.");
	ADD("    Vous pouvez modifier votre copie ou vos copies du Programme ou partie de celui-ci, ou d'un travail bas sur ce Programme, et copier et distribuer ces");
	ADD("    modifications selon les termes de l'article 1,  condition de Vous conformer galement aux conditions suivantes : ");
	ADD("        a) Ajouter aux fichiers modifis l'indication trs claire des modifications effectues, ainsi que la date de chaque changement. ");
	ADD("        b) Distribuer sous les termes de la Licence Publique Gnrale l'ensemble de toute ralisation contenant tout ou partie du Programme, avec ou sans");
	ADD("        modifications. ");
	ADD("        c) Si le Programme modifi lit des commandes de manire interactive lors de son excution, faire en sorte qu'il affiche, lors d'une invocation");
	ADD("        ordinaire, le copyright appropri en indiquant clairement la limitation de garantie (ou la garantie que Vous Vous engagez  fournir Vous-mme),");
	ADD("        qu'il stipule que tout utilisateur peut librement redistribuer le Programme selon les conditions de la Licence Publique Gnrale GNU, et qu'il montre");
	ADD("         tout utilisateur comment lire une copie de celle-ci (exception : si le Programme original est interactif mais n'affiche pas un tel message en temps");
	ADD("        normal, tout travail driv de ce Programme ne sera pas non plus contraint de l'afficher). ");
	ADD("");
	ADD("    Toutes ces conditions s'appliquent  l'ensemble des modifications. Si des lments identifiables de ce travail ne sont pas drivs du Programme et");
	ADD("    peuvent tre raisonnablement considrs comme indpendants, la prsente Licence ne s'applique pas  ces lments lorsque Vous les distribuez seuls.");
	ADD("    Mais, si Vous distribuez ces mmes lments comme partie d'un ensemble cohrent dont le reste est bas sur un Programme soumis  la Licence, ils lui");
	ADD("    sont galement soumis, et la Licence s'tend ainsi  l'ensemble du produit, quel qu'en soit l'auteur.");
	ADD("");
	ADD("    Cet article n'a pas pour but de s'approprier ou de contester vos droits sur un travail entirement ralis par Vous, mais plutt d'ouvrir droit  un contrle");
	ADD("    de la libre distribution de tout travail driv ou collectif bas sur le Programme.");
	ADD("");
	ADD("    En outre, toute fusion d'un autre travail, non bas sur le Programme, avec le Programme (ou avec un travail driv de ce dernier), effectue sur un");
	ADD("    support de stockage ou de distribution, ne fait pas tomber cet autre travail sous le contrle de la Licence.");
	ADD("");
	ADD("");
	ADD("    Article 3.");
	ADD("    Vous pouvez copier et distribuer le Programme (ou tout travail driv selon les conditions nonces dans l'article 1) sous forme de code objet ou");
	ADD("    excutable, selon les termes des articles 0 et 1,  condition de respecter les clauses suivantes : ");
	ADD("        a) Fournir le code source complet du Programme, sous une forme lisible par un ordinateur et selon les termes des articles 0 et 1, sur un support");
	ADD("        habituellement utilis pour l'change de donnes ; ou, ");
	ADD("        b) Faire une offre crite, valable pendant au moins trois ans, prvoyant de donner  tout tiers qui en fera la demande une copie, sous forme lisible");
	ADD("        par un ordinateur, du code source correspondant, pour un tarif n'excdant pas le cot de la copie, selon les termes des articles 0 et 1, sur un support");
	ADD("        couramment utilis pour l'change de donnes informatiques ; ou, ");
	ADD("        c) Informer le destinataire de l'endroit o le code source peut tre obtenu (cette solution n'est recevable que dans le cas d'une distribution non");
	ADD("        commerciale, et uniquement si Vous avez reu le Programme sous forme de code objet ou excutable avec l'offre prvue  l'alina b ci-dessus). ");
	ADD("");
	ADD("    Le code source d'un travail dsigne la forme de cet ouvrage sous laquelle les modifications sont les plus aises. Sont ainsi dsigns la totalit du code");
	ADD("    source de tous les modules composant un Programme excutable, de mme que tout fichier de dfinition associ, ainsi que les scripts utiliss pour");
	ADD("    effectuer la compilation et l'installation du Programme excutable. Toutefois, l'environnement standard de dveloppement du systme d'exploitation mis");
	ADD("    en oeuvre (source ou binaire) -- compilateurs, bibliothques, noyau, etc. -- constitue une exception, sauf si ces lments sont diffuss en mme temps que");
	ADD("    le Programme excutable.");
	ADD("");
	ADD("    Si la distribution de l'excutable ou du code objet consiste  offrir un accs permettant de copier le Programme depuis un endroit particulier, l'offre d'un");
	ADD("    accs quivalent pour se procurer le code source au mme endroit est considr comme une distribution de ce code source, mme si l'utilisateur choisit");
	ADD("    de ne pas profiter de cette offre.");
	ADD("");
	ADD("");
	ADD("    Article 4.");
	ADD("    Vous ne pouvez pas copier, modifier, cder, dposer ou distribuer le Programme d'une autre manire que l'autorise la Licence Publique Gnrale. Toute");
	ADD("    tentative de ce type annule immdiatement vos droits d'utilisation du Programme sous cette Licence. Toutefois, les tiers ayant reu de Vous des copies du");
	ADD("    Programme ou le droit d'utiliser ces copies continueront  bnficier de leur droit d'utilisation tant qu'ils respecteront pleinement les conditions de la");
	ADD("    Licence.");
	ADD("");
	ADD("");
	ADD("    Article 5.");
	ADD("    Ne l'ayant pas signe, Vous n'tes pas oblig d'accepter cette Licence. Cependant, rien d'autre ne Vous autorise  modifier ou distribuer le Programme ou");
	ADD("    quelque travaux drivs : la loi l'interdit tant que Vous n'acceptez pas les termes de cette Licence. En consquence, en modifiant ou en distribuant le");
	ADD("    Programme (ou tout travail bas sur lui), Vous acceptez implicitement tous les termes et conditions de cette Licence.");
	ADD("");
	ADD("");
	ADD("    Article 6.");
	ADD("    La diffusion d'un Programme (ou de tout travail driv) suppose l'envoi simultan d'une licence autorisant la copie, la distribution ou la modification du");
	ADD("    Programme, aux termes et conditions de la Licence. Vous n'avez pas le droit d'imposer de restrictions supplmentaires aux droits transmis au");
	ADD("    destinataire. Vous n'tes pas responsable du respect de la Licence par un tiers.");
	ADD("");
	ADD("");
	ADD("    Article 7.");
	ADD("    Si,  la suite d'une dcision de Justice, d'une plainte en contrefaon ou pour toute autre raison (lie ou non  la contrefaon), des conditions Vous sont");
	ADD("    imposes (que ce soit par ordonnance, accord amiable ou autre) qui se rvlent incompatibles avec les termes de la prsente Licence, Vous n'tes pas");
	ADD("    pour autant dgag des obligations lies  celle-ci : si Vous ne pouvez concilier vos obligations lgales ou autres avec les conditions de cette Licence,");
	ADD("    Vous ne devez pas distribuer le Programme.");
	ADD("");
	ADD("    Si une partie quelconque de cet article est invalide ou inapplicable pour quelque raison que ce soit, le reste de l'article continue de s'appliquer et");
	ADD("    l'intgralit de l'article s'appliquera en toute autre circonstance.");
	ADD("");
	ADD("    Le prsent article n'a pas pour but de Vous pousser  enfreindre des droits ou des dispositions lgales ni en contester la validit ; son seul objectif est de");
	ADD("    protger l'intgrit du systme de distribution du Logiciel Libre. De nombreuses personnes ont gnreusement contribu  la large gamme de Programmes");
	ADD("    distribue de cette faon en toute confiance ; il appartient  chaque auteur/donateur de dcider de diffuser ses Programmes selon les critres de son");
	ADD("    choix.");
	ADD("");
	ADD("");
	ADD("    Article 8.");
	ADD("    Si la distribution et/ou l'utilisation du Programme est limite dans certains pays par des brevets ou des droits sur des interfaces, le dtenteur original des");
	ADD("    droits qui place le Programme sous la Licence Publique Gnrale peut ajouter explicitement une clause de limitation gographique excluant ces pays.");
	ADD("    Dans ce cas, cette clause devient une partie intgrante de la Licence.");
	ADD("");
	ADD("");
	ADD("    Article 9.");
	ADD("    La Free Software Foundation se rserve le droit de publier priodiquement des mises  jour ou de nouvelles versions de la Licence. Rdiges dans le");
	ADD("    mme esprit que la prsente version, elles seront cependant susceptibles d'en modifier certains dtails  mesure que de nouveaux problmes se font jour.");
	ADD("");
	ADD("    Chaque version possde un numro distinct. Si le Programme prcise un numro de version de cette Licence et  toute version ultrieure , Vous avez le");
	ADD("    choix de suivre les termes et conditions de cette version ou de toute autre version plus rcente publie par la Free Software Foundation. Si le Programme");
	ADD("    ne spcifie aucun numro de version, Vous pouvez alors choisir l'une quelconque des versions publies par la Free Software Foundation.");
	ADD("");
	ADD("");
	ADD("    Article 10.");
	ADD("    Si Vous dsirez incorporer des lments du Programme dans d'autres Programmes libres dont les conditions de distribution diffrent, Vous devez crire");
	ADD("     l'auteur pour lui en demander la permission. Pour ce qui est des Programmes directement dposs par la Free Software Foundation, crivez-nous : une");
	ADD("    exception est toujours envisageable. Notre dcision sera base sur notre volont de prserver la libert de notre Programme ou de ses drivs et celle de");
	ADD("    promouvoir le partage et la rutilisation du logiciel en gnral.");
	ADD("");
	ADD("                                                    LIMITATION DE GARANTIE");
	ADD("");
	ADD("    Article 11.");
	ADD("    Parce que l'utilisation de ce Programme est libre et gratuite, aucune garantie n'est fournie, comme le permet la loi. Sauf mention crite, les dtenteurs du");
	ADD("    copyright et/ou les tiers fournissent le Programme en l'tat, sans aucune sorte de garantie explicite ou implicite, y compris les garanties de");
	ADD("    commercialisation ou d'adaptation dans un but particulier. Vous assumez tous les risques quant  la qualit et aux effets du Programme. Si le Programme");
	ADD("    est dfectueux, Vous assumez le cot de tous les services, corrections ou rparations ncessaires.");
	ADD("");
	ADD("");
	ADD("    Article 12.");
	ADD("    Sauf lorsqu'explicitement prvu par la Loi ou accept par crit, ni le dtenteur des droits, ni quiconque autoris  modifier et/ou redistribuer le");
	ADD("    Programme comme il est permis ci-dessus ne pourra tre tenu pour responsable de tout dommage direct, indirect, secondaire ou accessoire (pertes");
	ADD("    financires dues au manque  gagner,  l'interruption d'activits ou  la perte de donnes, etc., dcoulant de l'utilisation du Programme ou de");
	ADD("    l'impossibilit d'utiliser celui-ci). ");
	ADD("");
	ADD("");
	ADD("                                               FIN DES TERMES ET CONDITIONS");

	textviewer->show();
}

void show_licence_en()
{
	textviewer->browser->clear();
	ADD("Copyright (C) Achraf cherti 2005-2006");
	ADD("");
	ADD("Ce programme est distribu selon les termes de la GPL v2.");
	ADD("");
	ADD("		    GNU GENERAL PUBLIC LICENSE");
	ADD("		       Version 2, June 1991");
	ADD("");
	ADD(" Copyright (C) 1989, 1991 Free Software Foundation, Inc.");
	ADD("               59 Temple Place, Suite 330, Boston, MA 02111-1307, USA");
	ADD(" Everyone is permitted to copy and distribute verbatim copies");
	ADD(" of this license document, but changing it is not allowed.");
	ADD("");
	ADD("			    Preamble");
	ADD("");
	ADD("  The licenses for most software are designed to take away your");
	ADD("freedom to share and change it.  By contrast, the GNU General Public");
	ADD("License is intended to guarantee your freedom to share and change free");
	ADD("software--to make sure the software is free for all its users.  This");
	ADD("General Public License applies to most of the Free Software");
	ADD("Foundation's software and to any other program whose authors commit to");
	ADD("using it.  (Some other Free Software Foundation software is covered by");
	ADD("the GNU Library General Public License instead.)  You can apply it to");
	ADD("your programs, too.");
	ADD("");
	ADD("  When we speak of free software, we are referring to freedom, not");
	ADD("price.  Our General Public Licenses are designed to make sure that you");
	ADD("have the freedom to distribute copies of free software (and charge for");
	ADD("this service if you wish), that you receive source code or can get it");
	ADD("if you want it, that you can change the software or use pieces of it");
	ADD("in new free programs; and that you know you can do these things.");
	ADD("");
	ADD("  To protect your rights, we need to make restrictions that forbid");
	ADD("anyone to deny you these rights or to ask you to surrender the rights.");
	ADD("These restrictions translate to certain responsibilities for you if you");
	ADD("distribute copies of the software, or if you modify it.");
	ADD("");
	ADD("  For example, if you distribute copies of such a program, whether");
	ADD("gratis or for a fee, you must give the recipients all the rights that");
	ADD("you have.  You must make sure that they, too, receive or can get the");
	ADD("source code.  And you must show them these terms so they know their");
	ADD("rights.");
	ADD("");
	ADD("  We protect your rights with two steps: (1) copyright the software, and");
	ADD("(2) offer you this license which gives you legal permission to copy,");
	ADD("distribute and/or modify the software.");
	ADD("");
	ADD("  Also, for each author's protection and ours, we want to make certain");
	ADD("that everyone understands that there is no warranty for this free");
	ADD("software.  If the software is modified by someone else and passed on, we");
	ADD("want its recipients to know that what they have is not the original, so");
	ADD("that any problems introduced by others will not reflect on the original");
	ADD("authors' reputations.");
	ADD("");
	ADD("  Finally, any free program is threatened constantly by software");
	ADD("patents.  We wish to avoid the danger that redistributors of a free");
	ADD("program will individually obtain patent licenses, in effect making the");
	ADD("program proprietary.  To prevent this, we have made it clear that any");
	ADD("patent must be licensed for everyone's free use or not licensed at all.");
	ADD("");
	ADD("  The precise terms and conditions for copying, distribution and");
	ADD("modification follow.");
	ADD("");
	ADD("		    GNU GENERAL PUBLIC LICENSE");
	ADD("   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION");
	ADD("");
	ADD("  0. This License applies to any program or other work which contains");
	ADD("a notice placed by the copyright holder saying it may be distributed");
	ADD("under the terms of this General Public License.  The \"Program\", below,");
	ADD("refers to any such program or work, and a \"work based on the Program\"");
	ADD("means either the Program or any derivative work under copyright law:");
	ADD("that is to say, a work containing the Program or a portion of it,");
	ADD("either verbatim or with modifications and/or translated into another");
	ADD("language.  (Hereinafter, translation is included without limitation in");
	ADD("the term \"modification\".)  Each licensee is addressed as \"you\".");
	ADD("");
	ADD("Activities other than copying, distribution and modification are not");
	ADD("covered by this License; they are outside its scope.  The act of");
	ADD("running the Program is not restricted, and the output from the Program");
	ADD("is covered only if its contents constitute a work based on the");
	ADD("Program (independent of having been made by running the Program).");
	ADD("Whether that is true depends on what the Program does.");
	ADD("");
	ADD("  1. You may copy and distribute verbatim copies of the Program's");
	ADD("source code as you receive it, in any medium, provided that you");
	ADD("conspicuously and appropriately publish on each copy an appropriate");
	ADD("copyright notice and disclaimer of warranty; keep intact all the");
	ADD("notices that refer to this License and to the absence of any warranty;");
	ADD("and give any other recipients of the Program a copy of this License");
	ADD("along with the Program.");
	ADD("");
	ADD("You may charge a fee for the physical act of transferring a copy, and");
	ADD("you may at your option offer warranty protection in exchange for a fee.");
	ADD("");
	ADD("  2. You may modify your copy or copies of the Program or any portion");
	ADD("of it, thus forming a work based on the Program, and copy and");
	ADD("distribute such modifications or work under the terms of Section 1");
	ADD("above, provided that you also meet all of these conditions:");
	ADD("");
	ADD("    a) You must cause the modified files to carry prominent notices");
	ADD("    stating that you changed the files and the date of any change.");
	ADD("");
	ADD("    b) You must cause any work that you distribute or publish, that in");
	ADD("    whole or in part contains or is derived from the Program or any");
	ADD("    part thereof, to be licensed as a whole at no charge to all third");
	ADD("    parties under the terms of this License.");
	ADD("");
	ADD("    c) If the modified program normally reads commands interactively");
	ADD("    when run, you must cause it, when started running for such");
	ADD("    interactive use in the most ordinary way, to print or display an");
	ADD("    announcement including an appropriate copyright notice and a");
	ADD("    notice that there is no warranty (or else, saying that you provide");
	ADD("    a warranty) and that users may redistribute the program under");
	ADD("    these conditions, and telling the user how to view a copy of this");
	ADD("    License.  (Exception: if the Program itself is interactive but");
	ADD("    does not normally print such an announcement, your work based on");
	ADD("    the Program is not required to print an announcement.)");
	ADD("");
	ADD("These requirements apply to the modified work as a whole.  If");
	ADD("identifiable sections of that work are not derived from the Program,");
	ADD("and can be reasonably considered independent and separate works in");
	ADD("themselves, then this License, and its terms, do not apply to those");
	ADD("sections when you distribute them as separate works.  But when you");
	ADD("distribute the same sections as part of a whole which is a work based");
	ADD("on the Program, the distribution of the whole must be on the terms of");
	ADD("this License, whose permissions for other licensees extend to the");
	ADD("entire whole, and thus to each and every part regardless of who wrote it.");
	ADD("");
	ADD("Thus, it is not the intent of this section to claim rights or contest");
	ADD("your rights to work written entirely by you; rather, the intent is to");
	ADD("exercise the right to control the distribution of derivative or");
	ADD("collective works based on the Program.");
	ADD("");
	ADD("In addition, mere aggregation of another work not based on the Program");
	ADD("with the Program (or with a work based on the Program) on a volume of");
	ADD("a storage or distribution medium does not bring the other work under");
	ADD("the scope of this License.");
	ADD("");
	ADD("  3. You may copy and distribute the Program (or a work based on it,");
	ADD("under Section 2) in object code or executable form under the terms of");
	ADD("Sections 1 and 2 above provided that you also do one of the following:");
	ADD("");
	ADD("    a) Accompany it with the complete corresponding machine-readable");
	ADD("    source code, which must be distributed under the terms of Sections");
	ADD("    1 and 2 above on a medium customarily used for software interchange; or,");
	ADD("");
	ADD("    b) Accompany it with a written offer, valid for at least three");
	ADD("    years, to give any third party, for a charge no more than your");
	ADD("    cost of physically performing source distribution, a complete");
	ADD("    machine-readable copy of the corresponding source code, to be");
	ADD("    distributed under the terms of Sections 1 and 2 above on a medium");
	ADD("    customarily used for software interchange; or,");
	ADD("");
	ADD("    c) Accompany it with the information you received as to the offer");
	ADD("    to distribute corresponding source code.  (This alternative is");
	ADD("    allowed only for noncommercial distribution and only if you");
	ADD("    received the program in object code or executable form with such");
	ADD("    an offer, in accord with Subsection b above.)");
	ADD("");
	ADD("The source code for a work means the preferred form of the work for");
	ADD("making modifications to it.  For an executable work, complete source");
	ADD("code means all the source code for all modules it contains, plus any");
	ADD("associated interface definition files, plus the scripts used to");
	ADD("control compilation and installation of the executable.  However, as a");
	ADD("special exception, the source code distributed need not include");
	ADD("anything that is normally distributed (in either source or binary");
	ADD("form) with the major components (compiler, kernel, and so on) of the");
	ADD("operating system on which the executable runs, unless that component");
	ADD("itself accompanies the executable.");
	ADD("");
	ADD("If distribution of executable or object code is made by offering");
	ADD("access to copy from a designated place, then offering equivalent");
	ADD("access to copy the source code from the same place counts as");
	ADD("distribution of the source code, even though third parties are not");
	ADD("compelled to copy the source along with the object code.");
	ADD("");
	ADD("  4. You may not copy, modify, sublicense, or distribute the Program");
	ADD("except as expressly provided under this License.  Any attempt");
	ADD("otherwise to copy, modify, sublicense or distribute the Program is");
	ADD("void, and will automatically terminate your rights under this License.");
	ADD("However, parties who have received copies, or rights, from you under");
	ADD("this License will not have their licenses terminated so long as such");
	ADD("parties remain in full compliance.");
	ADD("");
	ADD("  5. You are not required to accept this License, since you have not");
	ADD("signed it.  However, nothing else grants you permission to modify or");
	ADD("distribute the Program or its derivative works.  These actions are");
	ADD("prohibited by law if you do not accept this License.  Therefore, by");
	ADD("modifying or distributing the Program (or any work based on the");
	ADD("Program), you indicate your acceptance of this License to do so, and");
	ADD("all its terms and conditions for copying, distributing or modifying");
	ADD("the Program or works based on it.");
	ADD("");
	ADD("  6. Each time you redistribute the Program (or any work based on the");
	ADD("Program), the recipient automatically receives a license from the");
	ADD("original licensor to copy, distribute or modify the Program subject to");
	ADD("these terms and conditions.  You may not impose any further");
	ADD("restrictions on the recipients' exercise of the rights granted herein.");
	ADD("You are not responsible for enforcing compliance by third parties to");
	ADD("this License.");
	ADD("");
	ADD("  7. If, as a consequence of a court judgment or allegation of patent");
	ADD("infringement or for any other reason (not limited to patent issues),");
	ADD("conditions are imposed on you (whether by court order, agreement or");
	ADD("otherwise) that contradict the conditions of this License, they do not");
	ADD("excuse you from the conditions of this License.  If you cannot");
	ADD("distribute so as to satisfy simultaneously your obligations under this");
	ADD("License and any other pertinent obligations, then as a consequence you");
	ADD("may not distribute the Program at all.  For example, if a patent");
	ADD("license would not permit royalty-free redistribution of the Program by");
	ADD("all those who receive copies directly or indirectly through you, then");
	ADD("the only way you could satisfy both it and this License would be to");
	ADD("refrain entirely from distribution of the Program.");
	ADD("");
	ADD("If any portion of this section is held invalid or unenforceable under");
	ADD("any particular circumstance, the balance of the section is intended to");
	ADD("apply and the section as a whole is intended to apply in other");
	ADD("circumstances.");
	ADD("");
	ADD("It is not the purpose of this section to induce you to infringe any");
	ADD("patents or other property right claims or to contest validity of any");
	ADD("such claims; this section has the sole purpose of protecting the");
	ADD("integrity of the free software distribution system, which is");
	ADD("implemented by public license practices.  Many people have made");
	ADD("generous contributions to the wide range of software distributed");
	ADD("through that system in reliance on consistent application of that");
	ADD("system; it is up to the author/donor to decide if he or she is willing");
	ADD("to distribute software through any other system and a licensee cannot");
	ADD("impose that choice.");
	ADD("");
	ADD("This section is intended to make thoroughly clear what is believed to");
	ADD("be a consequence of the rest of this License.");
	ADD("");
	ADD("  8. If the distribution and/or use of the Program is restricted in");
	ADD("certain countries either by patents or by copyrighted interfaces, the");
	ADD("original copyright holder who places the Program under this License");
	ADD("may add an explicit geographical distribution limitation excluding");
	ADD("those countries, so that distribution is permitted only in or among");
	ADD("countries not thus excluded.  In such case, this License incorporates");
	ADD("the limitation as if written in the body of this License.");
	ADD("");
	ADD("  9. The Free Software Foundation may publish revised and/or new versions");
	ADD("of the General Public License from time to time.  Such new versions will");
	ADD("be similar in spirit to the present version, but may differ in detail to");
	ADD("address new problems or concerns.");
	ADD("");
	ADD("Each version is given a distinguishing version number.  If the Program");
	ADD("specifies a version number of this License which applies to it and \"any");
	ADD("later version\", you have the option of following the terms and conditions");
	ADD("either of that version or of any later version published by the Free");
	ADD("Software Foundation.  If the Program does not specify a version number of");
	ADD("this License, you may choose any version ever published by the Free Software");
	ADD("Foundation.");
	ADD("");
	ADD("  10. If you wish to incorporate parts of the Program into other free");
	ADD("programs whose distribution conditions are different, write to the author");
	ADD("to ask for permission.  For software which is copyrighted by the Free");
	ADD("Software Foundation, write to the Free Software Foundation; we sometimes");
	ADD("make exceptions for this.  Our decision will be guided by the two goals");
	ADD("of preserving the free status of all derivatives of our free software and");
	ADD("of promoting the sharing and reuse of software generally.");
	ADD("");
	ADD("			    NO WARRANTY");
	ADD("");
	ADD("  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY");
	ADD("FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN");
	ADD("OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES");
	ADD("PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED");
	ADD("OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF");
	ADD("MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS");
	ADD("TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE");
	ADD("PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,");
	ADD("REPAIR OR CORRECTION.");
	ADD("");
	ADD("  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING");
	ADD("WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR");
	ADD("REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,");
	ADD("INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING");
	ADD("OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED");
	ADD("TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY");
	ADD("YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER");
	ADD("PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE");
	ADD("POSSIBILITY OF SUCH DAMAGES.");
	ADD("");
	ADD("		     END OF TERMS AND CONDITIONS");
	ADD("");
	ADD("");
	ADD("	    How to Apply These Terms to Your New Programs");
	ADD("");
	ADD("  If you develop a new program, and you want it to be of the greatest");
	ADD("possible use to the public, the best way to achieve this is to make it");
	ADD("free software which everyone can redistribute and change under these terms.");
	ADD("");
	ADD("  To do so, attach the following notices to the program.  It is safest");
	ADD("to attach them to the start of each source file to most effectively");
	ADD("convey the exclusion of warranty; and each file should have at least");
	ADD("the \"copyright\" line and a pointer to where the full notice is found.");
	ADD("");
	ADD("    <one line to give the program's name and a brief idea of what it does.>");
	ADD("    Copyright (C) 19yy  <name of author>");
	ADD("");
	ADD("    This program is free software; you can redistribute it and/or modify");
	ADD("    it under the terms of the GNU General Public License as published by");
	ADD("    the Free Software Foundation; either version 2 of the License, or");
	ADD("    (at your option) any later version.");
	ADD("");
	ADD("    This program is distributed in the hope that it will be useful,");
	ADD("    but WITHOUT ANY WARRANTY; without even the implied warranty of");
	ADD("    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the");
	ADD("    GNU General Public License for more details.");
	ADD("");
	ADD("    You should have received a copy of the GNU General Public License");
	ADD("    along with this program; if not, write to the Free Software");
	ADD("    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA");
	ADD("");
	ADD("");
	ADD("Also add information on how to contact you by electronic and paper mail.");
	ADD("");
	ADD("If the program is interactive, make it output a short notice like this");
	ADD("when it starts in an interactive mode:");
	ADD("");
	ADD("    Gnomovision version 69, Copyright (C) 19yy name of author");
	ADD("    Gnomovision comes with ABSOLUTELY NO WARRANTY; for details type `show w'.");
	ADD("    This is free software, and you are welcome to redistribute it");
	ADD("    under certain conditions; type `show c' for details.");
	ADD("");
	ADD("The hypothetical commands `show w' and `show c' should show the appropriate");
	ADD("parts of the General Public License.  Of course, the commands you use may");
	ADD("be called something other than `show w' and `show c'; they could even be");
	ADD("mouse-clicks or menu items--whatever suits your program.");
	ADD("");
	ADD("You should also get your employer (if you work as a programmer) or your");
	ADD("school, if any, to sign a \"copyright disclaimer\" for the program, if");
	ADD("necessary.  Here is a sample; alter the names:");
	ADD("");
	ADD("  Yoyodyne, Inc., hereby disclaims all copyright interest in the program");
	ADD("  `Gnomovision' (which makes passes at compilers) written by James Hacker.");
	ADD("");
	ADD("  <signature of Ty Coon>, 1 April 1989");
	ADD("  Ty Coon, President of Vice");
	ADD("");
	ADD("This General Public License does not permit incorporating your program into");
	ADD("proprietary programs.  If your program is a subroutine library, you may");
	ADD("consider it more useful to permit linking proprietary applications with the");
	ADD("library.  If this is what you want to do, use the GNU Library General");
	ADD("Public License instead of this License.");
	textviewer->show();
}
#undef ADD

