/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#include "_xitk.h"
#include "doublebox.h"
#include "inputtext.h"
#include "button.h"
#include "default_font.h"

typedef enum {
  /* keep order */
  _W_input = 0,
  _W_more,
  _W_less,
  /* /keep order */
  _W_LAST
} _W_t;

typedef struct {
  xitk_widget_t                   w;

  xitk_widget_t                  *iw[_W_LAST];

  double                          step;
  double                          value;

  xitk_state_double_callback_t    callback;

  uint32_t                        last_pos;
} _doublebox_private_t;

/*
 *
 */
static void _db_paint (_doublebox_private_t *wp) {
  XITK_HV_INIT;

  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    if (wp->w.pos.w != wp->last_pos) {
      int bx, ih, iw;

      wp->last_pos = wp->w.pos.w;
      iw = xitk_get_widget_width (wp->iw[_W_input]);
      ih = xitk_get_widget_height (wp->iw[_W_input]);
      xitk_set_widget_pos (wp->iw[_W_input], XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
      bx = XITK_HV_H (wp->w.pos) + iw;
      xitk_set_widget_pos (wp->iw[_W_more], bx, XITK_HV_V (wp->w.pos));
      xitk_set_widget_pos (wp->iw[_W_less], bx, XITK_HV_V (wp->w.pos) + (ih >> 1));
    }
  }
  xitk_widgets_state (wp->iw + _W_input, 3, XITK_WIDGET_STATE_VISIBLE, wp->w.state);
}

static int notify_event (xitk_widget_t *w, const widget_event_t *event) {
  _doublebox_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      _db_paint (wp);
      break;
    case WIDGET_EVENT_CHANGE_SKIN:
      break;
    case WIDGET_EVENT_DESTROY:
      xitk_widgets_delete (wp->iw + _W_input, 3);
      break;
    case WIDGET_EVENT_ENABLE:
      xitk_widgets_state (wp->iw + _W_input, 3, XITK_WIDGET_STATE_ENABLE, (wp->w.state & XITK_WIDGET_STATE_ENABLE));
      break;
    case WIDGET_EVENT_TIPS_TIMEOUT:
      if (wp->iw[_W_input])
        xitk_set_widget_tips_and_timeout (wp->iw[_W_input], wp->w.tips_string, event->tips_timeout);
      break;
    default: ;
  }
  return 0;
}

/*
 *
 */
static void doublebox_change_value(xitk_widget_t *x, void *data, const char *string) {
  _doublebox_private_t *wp = (_doublebox_private_t *)data;
  char buf[256];

  if (!wp || !string)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return;

  (void)x;
  wp->value = strtod (string, NULL);
  memset (&buf, 0, sizeof (buf));
  snprintf (buf, sizeof (buf), "%lf", wp->value);
  xitk_inputtext_change_text (wp->iw[_W_input], buf);
  if (wp->callback)
    wp->callback (&wp->w, wp->w.userdata, wp->value);
}

/*
 *
 */
void xitk_doublebox_set_value(xitk_widget_t *w, double value) {
  _doublebox_private_t *wp;
  char buf[256];

  xitk_container (wp, w, w);
  if (!wp)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return;

  wp->value = value;
  memset (&buf, 0, sizeof (buf));
  snprintf (buf, sizeof (buf), "%lf", wp->value);
  xitk_inputtext_change_text (wp->iw[_W_input], buf);
}

/*
 *
 */
double xitk_doublebox_get_value(xitk_widget_t *w) {
  _doublebox_private_t *wp;
  const char *strval;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return 0;

  strval = xitk_inputtext_get_text (wp->iw[_W_input]);
  wp->value = strtod (strval, NULL);
  return wp->value;
}

/*
 *
 */
static void doublebox_minus(xitk_widget_t *x, void *data) {
  _doublebox_private_t *wp = (_doublebox_private_t *)data;
  char buf[256];

  if (!wp)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return;

  (void)x;
  wp->value -= wp->step;
  memset (&buf, 0, sizeof (buf));
  snprintf (buf, sizeof (buf), "%lf", wp->value);
  xitk_inputtext_change_text (wp->iw[_W_input], buf);
  if (wp->callback)
    wp->callback (&wp->w, wp->w.userdata, wp->value);
}

/*
 *
 */
static void doublebox_plus(xitk_widget_t *x, void *data) {
  _doublebox_private_t *wp = (_doublebox_private_t *)data;
  char buf[256];

  if (!wp)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_DOUBLEBOX)
    return;

  (void)x;
  wp->value += wp->step;
  memset (&buf, 0, sizeof (buf));
  snprintf (buf, sizeof (buf), "%lf", wp->value);
  xitk_inputtext_change_text (wp->iw[_W_input], buf);
  if (wp->callback)
    wp->callback (&wp->w, wp->w.userdata, wp->value);
}

/*
 *
 */
xitk_widget_t *xitk_noskin_doublebox_create (const xitk_doublebox_widget_t *ib, int x, int y, int width, int height) {
  _doublebox_private_t *wp;
  char buf[256], fontname[128];
  XITK_HV_INIT;

  wp = (_doublebox_private_t *)xitk_widget_new (&ib->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  XITK_HV_H (wp->w.size) = width;
  XITK_HV_V (wp->w.size) = height;
  wp->w.state     &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  wp->w.type         = WIDGET_GROUP | WIDGET_TYPE_DOUBLEBOX;
  wp->w.event        = notify_event;

  wp->callback          = ib->callback;
  wp->step              = ib->step;
  wp->value             = ib->value;
  /* Create inputtext and buttons (not skinable) */

  memset (&buf, 0, sizeof (buf));
  snprintf (buf, sizeof (buf), "%lf", ib->value);

  {
    xitk_inputtext_widget_t inp = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = ib->nw.add_state ? ib->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX
      },
      .text = buf,
      .max_length = 16,
      .callback = doublebox_change_value
    };

    XITK_DEFAULT_FONT_NAME (fontname, DEFAULT_FONT_FMT, height);
    wp->iw[_W_input] = xitk_noskin_inputtext_create (&inp,
      x, y, (width - 10), height, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, fontname);
  }

  {
    xitk_button_widget_t b = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = ib->nw.add_state ? ib->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_DOUBLEBOX
      },
      .symbol = XITK_SYMBOL_PLUS
    };

    b.callback = doublebox_plus;
    wp->iw[_W_more] = xitk_noskin_button_create (&b, x + width - (height >> 1), y, height >> 1, height >> 1);

    b.symbol = XITK_SYMBOL_MINUS;
    b.callback = doublebox_minus;
    wp->iw[_W_less] = xitk_noskin_button_create (&b, x + width - (height >> 1), y + (height >> 1), height >> 1, height >> 1);
  }

  _xitk_new_widget_apply (&ib->nw, &wp->w);

  return &wp->w;
}
