! This file is part of xtb.
!
! Copyright (C) 2023 Marcel Mueller
!
! xtb is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! xtb is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with xtb.  If not, see <https://www.gnu.org/licenses/>.

#ifndef WITH_TBLITE
#define WITH_TBLITE 0
#endif

!> PTB parametrization data
module xtb_ptb_param
#if WITH_TBLITE
   use xtb_mctc_accuracy, only: wp
   use xtb_ptb_data, only: init, &
   & TPTBData, TCorePotentialData, THamiltonianData, &
      TEEQData, TPauliXCData, TCoulombData, TPlusU, TResponse
   use xtb_type_param, only: TPTBParameter
   use xtb_mctc_convert, only: aatoau

   use xtb_ptb_vdzp, only: max_elem, nshell, max_shell, ang_shell

   implicit none
   private

   !> Global and essential parameters
   public :: initPTB, ptbGlobals
   !> Atomic radii for CN
   public :: rf
   !> Exponent scaling parameters for overlap matrices
   public :: kalphah0l, klalphaxc

   public :: setPTBReferenceOcc
   interface initPTB
      module procedure :: initData
      module procedure :: initHamiltonian
      module procedure :: initCorePotential
      module procedure :: initEEQ
      module procedure :: initPauliXC
      module procedure :: initCoulomb
      module procedure :: initPlusU
      module procedure :: initResponse
   end interface initPTB

   type(TPTBParameter), parameter :: ptbGlobals = TPTBParameter( &
      kpol=0.2272957611_wp, &
      kpolres=0.6291885570_wp, &
      kitr=0.0407296920_wp, &
      geps0=1.2209123751_wp, &
      kitocod=1.0926562388_wp, &
      geps=-0.1771695250_wp, &
      kerfcn=2.0_wp, &
      kerfcn_eeq=7.5_wp, &
      mlmix = 0.33333333333333_wp)

   !> Maximum angular momentum supported by PTB
   integer, parameter :: max_angmom = maxval(ang_shell)

   !----------- H0 DATA --------------

   !> Radius-like parameter in H0
   !> see Eq. 7 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kr(max_elem) = [&
   &  0.0526621033_wp, -0.1259789251_wp, -0.0009919449_wp, 0.1038653034_wp, -0.0013797952_wp, &
   & -0.0105524688_wp, 0.0305573244_wp, 0.0399300788_wp, -0.0000776731_wp, -0.0554411947_wp, &
   & -0.0856156068_wp, 0.0273394800_wp, 0.0017285337_wp, -0.2070450754_wp, -0.1544716111_wp, &
   & -0.1048578352_wp, -0.0925797761_wp, -0.1146631679_wp, -0.0390512368_wp, 0.0063365009_wp, &
   & -0.2357654649_wp, -0.3225969806_wp, -0.4567287642_wp, -0.3160355266_wp, -0.1923924379_wp, &
   & -0.3506012470_wp, -0.3763369708_wp, -0.1902205786_wp, -0.1121995817_wp, -0.0757997447_wp, &
   & -0.0778122987_wp, -0.1447676354_wp, -0.1433224343_wp, -0.1696459346_wp, -0.0079415656_wp, &
   &  0.0010872527_wp, -0.0397371478_wp, 0.0082424834_wp, -0.2830360762_wp, -0.0425719559_wp, &
   & -0.2563295136_wp, -0.2117541280_wp, -0.1189283930_wp, -0.3378556868_wp, -0.3728305415_wp, &
   & -0.4370684092_wp, -0.1631039630_wp, -0.0475780027_wp, -0.2192666427_wp, -0.1520873530_wp, &
   & -0.1541932897_wp, -0.2058776450_wp, -0.0647952675_wp, 0.0002216001_wp, -0.0175202529_wp, &
   &  0.0180104985_wp, -0.1704895691_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.0331391675_wp, -0.0935714439_wp, -0.2808411704_wp, &
   & -0.0615718239_wp, -0.2601246709_wp, -0.3573936184_wp, -0.4095341394_wp, -0.2447579231_wp, &
   & -0.0388921785_wp, -0.1456856492_wp, -0.2267507878_wp, -0.1474041482_wp, -0.1138167374_wp, &
   & -0.3419533891_wp, -0.0007046794_wp]

   !> Wolfsberg parameter in H0
   !> see Eq. 7 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kla(max_angmom+1, max_elem) = reshape([&
   &  2.3456350394_wp, 2.5867439077_wp, 0.0000000000_wp, &
   &  1.8790862592_wp, 2.2362835825_wp, 0.0000000000_wp, &
   &  2.1034538375_wp, 2.0807166747_wp, 2.3889199775_wp, &
   &  1.6673860226_wp, 1.7878102040_wp, 2.7545304887_wp, &
   &  2.1816204306_wp, 2.5034433319_wp, 3.0614684088_wp, &
   &  2.4901117474_wp, 2.5318511515_wp, 2.5882016720_wp, &
   &  2.2153858761_wp, 2.2506806859_wp, 2.5417554001_wp, &
   &  1.9891869596_wp, 1.9762959767_wp, 2.5410037490_wp, &
   &  1.8183342519_wp, 1.8253693454_wp, 3.1657459294_wp, &
   &  3.6537709658_wp, 3.3585490721_wp, 2.2012785567_wp, &
   &  1.1535381826_wp, 1.9879380966_wp, 1.4455181297_wp, &
   &  1.2302266132_wp, 1.5716145838_wp, 1.4084648480_wp, &
   &  2.2191870246_wp, 2.4672201339_wp, 3.2973306960_wp, &
   &  2.1067688965_wp, 2.3858238020_wp, 2.4904081609_wp, &
   &  1.6531335753_wp, 2.1258583981_wp, 2.5460180529_wp, &
   &  1.6325060511_wp, 2.0360355703_wp, 2.4450345570_wp, &
   &  1.9931234087_wp, 1.7846705508_wp, 2.1513854320_wp, &
   &  2.1334923191_wp, 1.9825621072_wp, 3.9632328294_wp, &
   &  1.1848664375_wp, 1.3549825724_wp, 1.4477061135_wp, &
   &  0.9865236798_wp, 1.5283209388_wp, 1.8700182615_wp, &
   &  1.8205372192_wp, 2.4284443247_wp, 2.4530493010_wp, &
   &  2.8234174850_wp, 2.9399023484_wp, 2.8336948463_wp, &
   &  2.8611711462_wp, 2.3898626865_wp, 3.0059421494_wp, &
   &  2.9131689050_wp, 2.1756336205_wp, 2.4949359627_wp, &
   &  4.8707288666_wp, 2.5905973963_wp, 2.4705712800_wp, &
   &  2.8644583686_wp, 3.4262956379_wp, 3.0201936230_wp, &
   &  2.4464480414_wp, 2.7562889024_wp, 2.6641354331_wp, &
   &  1.6099054933_wp, 2.2455282663_wp, 1.8644328806_wp, &
   &  1.7641694148_wp, 1.8656869746_wp, 1.3611817553_wp, &
   &  1.3310487425_wp, 1.6295328235_wp, 1.2841041754_wp, &
   &  2.0615588693_wp, 2.3775162864_wp, 3.4394202693_wp, &
   &  2.0474740203_wp, 2.5571719148_wp, 3.1854384995_wp, &
   &  1.5725824318_wp, 2.1133961137_wp, 2.7839743651_wp, &
   &  1.6429833868_wp, 2.0541171961_wp, 2.4184686776_wp, &
   &  1.8034595171_wp, 1.7653076080_wp, 2.0287986452_wp, &
   &  1.8184128577_wp, 2.3799056409_wp, 2.8894341045_wp, &
   &  1.0861278906_wp, 1.3159812140_wp, 2.1590825626_wp, &
   &  1.0142998159_wp, 1.1882684709_wp, 2.1595521716_wp, &
   &  2.1772217982_wp, 1.9359587636_wp, 2.7662090525_wp, &
   &  2.1890876375_wp, 2.2144625960_wp, 2.4852145019_wp, &
   &  3.1922376247_wp, 2.3244320906_wp, 2.6217692173_wp, &
   &  3.7769491607_wp, 2.7850019842_wp, 3.0387838560_wp, &
   &  4.6077642668_wp, 3.2694644117_wp, 2.9618591034_wp, &
   &  3.7764761735_wp, 3.4041346265_wp, 2.6718372480_wp, &
   &  3.5504342609_wp, 3.3398222092_wp, 3.0409515982_wp, &
   &  2.6549280155_wp, 2.8568585437_wp, 2.3848594654_wp, &
   &  1.8353413224_wp, 1.6094164245_wp, 1.7579105474_wp, &
   &  1.2311728806_wp, 1.9004495701_wp, 1.2295483708_wp, &
   &  1.9349936099_wp, 2.6573497507_wp, 2.6623065724_wp, &
   &  2.0736807228_wp, 2.6042372305_wp, 2.3559868420_wp, &
   &  1.7024712356_wp, 2.2172051306_wp, 2.3019641026_wp, &
   &  1.9210857297_wp, 2.3338717392_wp, 2.4618262441_wp, &
   &  2.2301872182_wp, 1.7833203990_wp, 1.9452052235_wp, &
   &  1.9484102024_wp, 1.8823512697_wp, 3.5879763305_wp, &
   &  1.1992070417_wp, 1.4185928209_wp, 1.4259214284_wp, &
   &  0.9860307592_wp, 1.1885926301_wp, 2.4863325322_wp, &
   &  2.3209167436_wp, 1.7991758534_wp, 2.3595936393_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  1.9629574353_wp, 2.1399737383_wp, 2.3197376145_wp, &
   &  2.8413083203_wp, 2.5460907528_wp, 2.5592166850_wp, &
   &  3.3628821272_wp, 2.8731284621_wp, 3.0913726463_wp, &
   &  3.6090001725_wp, 2.3661107340_wp, 2.8154505963_wp, &
   &  4.3040217876_wp, 2.7805487296_wp, 2.1624205641_wp, &
   &  3.6287568736_wp, 3.4858428696_wp, 2.9196206174_wp, &
   &  2.7759686458_wp, 2.7441718718_wp, 2.4624567409_wp, &
   &  2.3007947568_wp, 2.4012150677_wp, 2.2081253425_wp, &
   &  1.9759143413_wp, 1.8656635712_wp, 1.8842149894_wp, &
   &  2.1134755360_wp, 2.2405578898_wp, 3.2382026735_wp, &
   &  2.0734066535_wp, 2.7250852423_wp, 2.4208855087_wp, &
   &  1.7660303983_wp, 2.2763235506_wp, 2.2620396823_wp, &
   &  1.5612264003_wp, 1.9549754244_wp, 1.7260013615_wp, &
   &  1.7732987911_wp, 1.7970606069_wp, 1.6560069382_wp, &
   &  1.5276643927_wp, 1.5596596971_wp, 2.8644541356_wp], shape(kla))

   !> Scale one-center-off-diagonal elements of H0
   !> see Eq. 8 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: ksla(max_angmom+1, max_elem) = reshape([&
   &  2.6278779271_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  2.6548058980_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  2.7971693029_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.7941295629_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  2.4437187461_wp,  2.3818804786_wp,  0.0000000000_wp, &
   &  2.5895392224_wp,  2.4532474089_wp,  0.0000000000_wp, &
   &  2.3237410904_wp,  2.1523580721_wp,  0.0000000000_wp, &
   &  2.2641335748_wp,  1.8939100899_wp,  0.0000000000_wp, &
   &  1.7801055506_wp,  1.8632846279_wp,  0.0000000000_wp, &
   &  1.9483441897_wp,  2.1180785437_wp,  0.0000000000_wp, &
   &  1.4492458927_wp,  2.1483400696_wp,  0.0000000000_wp, &
   &  1.2148177309_wp,  1.5114111261_wp,  0.0000000000_wp, &
   &  2.3085223892_wp,  2.2890765552_wp,  0.0000000000_wp, &
   &  2.0895459151_wp,  2.3292691372_wp,  0.0000000000_wp, &
   &  1.8007484889_wp,  1.6515202948_wp,  0.0000000000_wp, &
   &  1.8107478605_wp,  1.7101444132_wp,  0.0000000000_wp, &
   &  1.9364673588_wp,  1.5762255639_wp,  0.0000000000_wp, &
   &  1.8376007902_wp,  1.8730257943_wp,  0.0000000000_wp, &
   &  1.1346682390_wp,  1.6427534580_wp,  0.0000000000_wp, &
   &  1.0530106478_wp,  1.6943365697_wp,  0.0000000000_wp, &
   &  1.1468597704_wp,  2.2175288977_wp,  1.9909389922_wp, &
   &  1.3065248800_wp,  1.9659122739_wp,  2.0421330603_wp, &
   &  0.9382805193_wp,  1.3304898721_wp,  2.2870260423_wp, &
   &  0.9967641611_wp,  1.2440957260_wp,  2.7988696520_wp, &
   &  1.1719575130_wp,  1.8740725272_wp,  2.3604065743_wp, &
   &  1.0448751633_wp,  2.2612946045_wp,  2.8976920563_wp, &
   &  1.0829229072_wp,  2.0727188844_wp,  2.6729721755_wp, &
   &  1.0789592260_wp,  2.0616387003_wp,  2.4396908930_wp, &
   &  1.1152669999_wp,  2.0320880046_wp,  2.4135267568_wp, &
   &  1.1480935443_wp,  1.7307553373_wp,  2.3635880178_wp, &
   &  2.1472603037_wp,  2.5127422302_wp,  0.0000000000_wp, &
   &  2.1220568627_wp,  2.3857496427_wp,  0.0000000000_wp, &
   &  1.6756480961_wp,  1.7184731983_wp,  0.0000000000_wp, &
   &  1.6917110507_wp,  1.7113130317_wp,  0.0000000000_wp, &
   &  1.9463481557_wp,  1.5872399128_wp,  0.0000000000_wp, &
   &  1.6205472457_wp,  2.3061390731_wp,  0.0000000000_wp, &
   &  1.1933432181_wp,  1.7978082592_wp,  0.0000000000_wp, &
   &  0.9890225423_wp,  1.4027681814_wp,  0.0000000000_wp, &
   &  0.8667470892_wp,  1.7543845557_wp,  2.3355545650_wp, &
   &  0.9802225585_wp,  1.9208177114_wp,  3.4304194995_wp, &
   &  1.3430014665_wp,  1.7752183785_wp,  2.7239116050_wp, &
   &  1.8336969369_wp,  1.7476495616_wp,  3.3558008183_wp, &
   &  1.3056003883_wp,  3.8631170933_wp,  2.3865440116_wp, &
   &  1.5337295832_wp,  2.1077569436_wp,  3.1931098073_wp, &
   &  1.5162751190_wp,  2.1128123206_wp,  3.2392272848_wp, &
   &  1.1295554602_wp,  1.9160097663_wp,  2.6729500429_wp, &
   &  1.0814323340_wp,  1.6470117840_wp,  2.2201512757_wp, &
   &  1.1471660410_wp,  1.5980123553_wp,  2.1452349661_wp, &
   &  1.7334939737_wp,  3.1700055124_wp,  0.0000000000_wp, &
   &  2.3050188519_wp,  2.4095683958_wp,  0.0000000000_wp, &
   &  1.8196504388_wp,  1.8762219938_wp,  0.0000000000_wp, &
   &  1.9418802251_wp,  1.9884626183_wp,  0.0000000000_wp, &
   &  1.9245677119_wp,  1.6283628430_wp,  0.0000000000_wp, &
   &  1.8978408967_wp,  1.9059951787_wp,  0.0000000000_wp, &
   &  1.2824443035_wp,  2.3052334175_wp,  0.0000000000_wp, &
   &  1.1039427002_wp,  1.4291023688_wp,  0.0000000000_wp, &
   &  1.3351100232_wp,  1.7215677877_wp,  2.6599792703_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  1.0188716288_wp,  1.8337013972_wp,  2.1515354393_wp, &
   &  1.2479329409_wp,  1.8820348050_wp,  2.5698875700_wp, &
   &  1.0392882102_wp,  2.1432037357_wp,  3.1221259932_wp, &
   &  1.3493866765_wp,  2.2702289418_wp,  2.7990035728_wp, &
   &  1.4686547844_wp,  2.1421068729_wp,  2.8267036917_wp, &
   &  1.5180326218_wp,  2.1762684756_wp,  2.8592416728_wp, &
   &  1.0656110397_wp,  1.9664948346_wp,  2.7844579605_wp, &
   &  0.8698163577_wp,  1.8264896420_wp,  2.4737358697_wp, &
   &  0.6358496402_wp,  1.5842554558_wp,  2.5691520373_wp, &
   &  2.0916469390_wp,  2.0946889068_wp,  0.0000000000_wp, &
   &  2.2195218972_wp,  2.5317469824_wp,  0.0000000000_wp, &
   &  1.8920418208_wp,  1.9323639694_wp,  0.0000000000_wp, &
   &  1.5794408592_wp,  1.5889917945_wp,  0.0000000000_wp, &
   &  1.9515137129_wp,  1.3761513931_wp,  0.0000000000_wp, &
   &  1.7515368226_wp,  1.6207993053_wp,  0.0000000000_wp], shape(ksla))

   !>  OCOD scaling in HO
   !> see Eq. 8 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kocod(max_elem) = [&
   &  0.0000000000_wp,  0.0000000000_wp, -0.0009968994_wp, -0.0125826604_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0351623755_wp, -0.1083543693_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, -0.0246201555_wp, -0.1796602234_wp, &
   & -0.0248475753_wp, -0.0488370607_wp, -0.0600020819_wp, -0.0456063281_wp, -0.0237428667_wp, &
   & -0.0228545980_wp, -0.0377471313_wp, -0.0379219024_wp, -0.0308117528_wp, -0.0438115179_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp, -0.0334177664_wp, -0.0842136926_wp, -0.0344384968_wp, -0.0471440329_wp, &
   & -0.0426325820_wp, -0.0084987127_wp, -0.0362124331_wp, -0.0565094099_wp, -0.0107785225_wp, &
   & -0.0281178941_wp, -0.0358649454_wp, -0.0319912156_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, -0.0224090199_wp, &
   & -0.1153296390_wp, -0.0182602125_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.0223700418_wp, -0.0436931188_wp, -0.0220301456_wp, &
   & -0.0289954489_wp, -0.0431578337_wp, -0.0181580254_wp, -0.0417051128_wp, -0.0663298112_wp, &
   & -0.0351086320_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp]

   !> H0 parameter in H0
   !> see Eq. 10 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: hla(max_shell, max_elem) = reshape([&
   & -0.6189949819_wp, -0.2731232971_wp, -0.2140591474_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.5147194542_wp, -0.3080371362_wp, -0.2986812848_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.6072372877_wp, -0.4610203270_wp, -0.2893831828_wp, -0.4714389699_wp, &
   & -0.2876218020_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.8692282840_wp, -0.7228346490_wp, -0.3998683644_wp, -0.5656727681_wp, &
   & -0.2191625369_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.7945777320_wp, -0.2703863470_wp, -0.5402351134_wp, -0.2882068698_wp, &
   & -0.1780879745_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9696268375_wp, -0.2103174627_wp, -0.5234006344_wp, -0.2694783119_wp, &
   & -0.2487304058_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.1642884412_wp, -0.2787536514_wp, -0.5730632667_wp, -0.2876464405_wp, &
   & -0.3181137333_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.2764157602_wp, -0.3174071855_wp, -0.6229546424_wp, -0.3482941872_wp, &
   & -0.3513288262_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.2052876406_wp, -0.4590522767_wp, -0.7029246816_wp, -0.4128494091_wp, &
   & -0.3156955960_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.5111833839_wp, -0.5607122694_wp, -0.6158090880_wp, -0.5170951828_wp, &
   & -0.3175382141_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9797527831_wp, -0.3188751807_wp, -0.6986794728_wp, -1.0241645167_wp, &
   & -0.5283035311_wp, -0.5151538354_wp,  0.0000000000_wp, &
   & -1.6192979164_wp, -1.0332699057_wp, -0.7490276782_wp, -1.6695592906_wp, &
   & -0.6208654189_wp, -0.3628143670_wp,  0.0000000000_wp, &
   & -0.8264535721_wp, -0.2285432581_wp, -0.5028599523_wp, -0.2784576299_wp, &
   & -0.2427471112_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9827992316_wp, -0.2732562425_wp, -0.6153244991_wp, -0.2383637597_wp, &
   & -0.3509311236_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9538458874_wp, -0.4038716275_wp, -0.5854061296_wp, -0.3653557101_wp, &
   & -0.3720368936_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.0195935584_wp, -0.4137514200_wp, -0.5893271444_wp, -0.3840545029_wp, &
   & -0.4864459449_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.0223523975_wp, -0.3918652912_wp, -0.7107937549_wp, -0.4391842368_wp, &
   & -0.5677615788_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.0426545626_wp, -0.8473897881_wp, -0.7049525139_wp, -0.3098499031_wp, &
   & -0.3515672839_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.8202351222_wp, -0.6073146200_wp, -0.6587075491_wp, -0.7814869153_wp, &
   & -0.5662748508_wp, -0.6103781467_wp,  0.0000000000_wp, &
   & -0.7719726571_wp, -0.6863929524_wp, -0.6592139232_wp, -1.0059152667_wp, &
   & -0.5059511715_wp, -0.4520847926_wp,  0.0000000000_wp, &
   & -0.8297148371_wp, -1.0511339874_wp, -0.6411965006_wp, -0.3529604590_wp, &
   & -0.2881869712_wp, -0.4747623571_wp, -0.3196497629_wp, &
   & -0.9226732466_wp, -1.1252571016_wp, -0.5730940221_wp, -0.4690012491_wp, &
   & -0.4009771527_wp, -0.4736050593_wp, -0.3342622776_wp, &
   & -0.9959312756_wp, -1.1558807846_wp, -0.5291895543_wp, -0.6158093460_wp, &
   & -0.4862502482_wp, -0.3291916784_wp, -0.3327113543_wp, &
   & -0.9312648097_wp, -1.3361455910_wp, -0.4989697551_wp, -1.0460201453_wp, &
   & -0.3537221567_wp, -0.3541665961_wp, -0.2164960532_wp, &
   & -1.0836980352_wp, -1.1726904762_wp, -0.2054311111_wp, -1.2555750415_wp, &
   & -0.4321302473_wp, -0.3799873054_wp, -0.2531347911_wp, &
   & -1.1035791540_wp, -1.2310082571_wp, -0.5906776151_wp, -1.4214749459_wp, &
   & -0.3663713843_wp, -0.3344040444_wp, -0.2492968977_wp, &
   & -1.0445573633_wp, -1.2287099974_wp, -0.7135463161_wp, -1.6372744288_wp, &
   & -0.4399579667_wp, -0.3212933238_wp, -0.2842088958_wp, &
   & -1.0213679456_wp, -1.2028670218_wp, -0.6806878030_wp, -1.6888957887_wp, &
   & -0.4815734215_wp, -0.3353333179_wp, -0.2993182930_wp, &
   & -1.0817857474_wp, -1.2150830235_wp, -0.6991297349_wp, -1.7102207085_wp, &
   & -0.5289140233_wp, -0.3450348504_wp, -0.2846078971_wp, &
   & -1.0894268027_wp, -1.3314350539_wp, -0.7484933582_wp, -1.7287965803_wp, &
   & -0.6269824658_wp, -0.3456878211_wp, -0.2843974859_wp, &
   & -0.8083233137_wp, -0.3058065537_wp, -0.5729516520_wp, -0.3025125574_wp, &
   & -0.2205512269_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9199407605_wp, -0.2755246252_wp, -0.5237524549_wp, -0.2602280435_wp, &
   & -0.2425727658_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.0441588230_wp, -0.3857463843_wp, -0.6359493373_wp, -0.3567003616_wp, &
   & -0.3612692792_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.0317020782_wp, -0.3920012951_wp, -0.6647978665_wp, -0.3695966128_wp, &
   & -0.4777352397_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.1148828775_wp, -0.3339816119_wp, -0.7326672849_wp, -0.4536173804_wp, &
   & -0.5584685569_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.2640358338_wp, -0.3861942615_wp, -0.5344546236_wp, -0.3028395358_wp, &
   & -0.3745918338_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.7958997596_wp, -0.9421240574_wp, -0.6968932071_wp, -0.8156090568_wp, &
   & -0.5849713033_wp, -0.4433683623_wp,  0.0000000000_wp, &
   & -0.5251732063_wp, -0.4052136160_wp, -0.6520960289_wp, -0.9379626515_wp, &
   & -0.5992384465_wp, -0.4552256800_wp,  0.0000000000_wp, &
   & -0.8108622315_wp, -1.0234572340_wp, -0.5743476805_wp, -0.4512124338_wp, &
   & -0.4063055075_wp, -0.4845993806_wp, -0.2807610463_wp, &
   & -0.7162283948_wp, -1.0254916043_wp, -0.6288664220_wp, -0.7907043237_wp, &
   & -0.3814841151_wp, -0.5050554750_wp, -0.2707326403_wp, &
   & -0.8782465545_wp, -0.9916710561_wp, -0.6323354251_wp, -1.0408141676_wp, &
   & -0.4282401489_wp, -0.4836898720_wp, -0.2533719460_wp, &
   & -1.1359039443_wp, -1.0513666028_wp, -0.4787791998_wp, -0.9112289839_wp, &
   & -0.3462483912_wp, -0.4919572440_wp, -0.1576286083_wp, &
   & -1.1527412508_wp, -0.8982347801_wp, -0.3046809796_wp, -0.9247312567_wp, &
   & -0.2537854170_wp, -0.4862106545_wp, -0.2301098067_wp, &
   & -1.1929663552_wp, -1.0475297883_wp, -0.4493644648_wp, -1.2323004974_wp, &
   & -0.3316881594_wp, -0.4976682324_wp, -0.2159081587_wp, &
   & -1.2159652815_wp, -1.2260127130_wp, -0.5880646454_wp, -0.9529038567_wp, &
   & -0.3343711030_wp, -0.4721464028_wp, -0.1808457594_wp, &
   & -0.8326568437_wp, -1.0427423173_wp, -0.5044710982_wp, -1.0130189035_wp, &
   & -0.4131311328_wp, -0.4506016597_wp, -0.2284851810_wp, &
   & -1.1213880475_wp, -0.8816147764_wp, -0.6606172280_wp, -1.0457428658_wp, &
   & -0.5671204579_wp, -0.5290477083_wp, -0.2925695748_wp, &
   & -1.1624744929_wp, -0.8265469002_wp, -0.6339889275_wp, -1.2260016097_wp, &
   & -0.5996224128_wp, -0.4917502562_wp, -0.2017161393_wp, &
   & -0.8319192664_wp, -0.2866494899_wp, -0.5302601739_wp, -0.2250576877_wp, &
   & -0.3253674343_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.8021919509_wp, -0.2804540439_wp, -0.5404163972_wp, -0.2448479144_wp, &
   & -0.3548741782_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9180089825_wp, -0.3532088962_wp, -0.6037696295_wp, -0.3530635640_wp, &
   & -0.4175830301_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9542134152_wp, -0.3337707119_wp, -0.5460098027_wp, -0.4013685440_wp, &
   & -0.4777177289_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.1336977934_wp, -0.4016680950_wp, -0.7198647717_wp, -0.4278049855_wp, &
   & -0.5410913893_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.3738105939_wp, -0.4182845275_wp, -0.5547010003_wp, -0.3316712543_wp, &
   & -0.2816751566_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.8007726542_wp, -0.6959596906_wp, -0.6439488598_wp, -0.8612713928_wp, &
   & -0.5347256618_wp, -0.5584407105_wp,  0.0000000000_wp, &
   & -0.6444635073_wp, -0.5958776334_wp, -0.7115551915_wp, -0.8911317231_wp, &
   & -0.5805164839_wp, -0.4646598781_wp,  0.0000000000_wp, &
   & -1.1325691157_wp, -0.4570199466_wp, -0.4697565197_wp, -0.7919091106_wp, &
   & -0.3903117348_wp, -0.5390255175_wp, -0.3370615194_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.8158862939_wp, -0.9025052225_wp, -0.6559785138_wp, -0.7253801018_wp, &
   & -0.2828995201_wp, -0.5089220315_wp, -0.2874149983_wp, &
   & -0.7160500119_wp, -0.8999404001_wp, -0.6246615911_wp, -0.9865762227_wp, &
   & -0.3713120197_wp, -0.4924537786_wp, -0.2366844717_wp, &
   & -0.7356563155_wp, -1.0852931925_wp, -0.6109849652_wp, -0.9494777469_wp, &
   & -0.3487274756_wp, -0.4972536549_wp, -0.2070688732_wp, &
   & -0.8698504635_wp, -0.8554817186_wp, -0.4682093120_wp, -1.2841902024_wp, &
   & -0.3847007012_wp, -0.4524096182_wp, -0.1892879601_wp, &
   & -0.7174308055_wp, -1.0751187382_wp, -0.5603204735_wp, -0.9272240714_wp, &
   & -0.3596171345_wp, -0.4421736194_wp, -0.2662438374_wp, &
   & -0.7109752776_wp, -1.0712466740_wp, -0.6351953751_wp, -0.8164383851_wp, &
   & -0.3175225709_wp, -0.4049844732_wp, -0.1930171019_wp, &
   & -0.9214370978_wp, -0.8348486661_wp, -0.5339212628_wp, -0.9621323836_wp, &
   & -0.4603914563_wp, -0.4557583720_wp, -0.2661913781_wp, &
   & -0.9296158074_wp, -1.1525282684_wp, -0.6363526117_wp, -1.1292048925_wp, &
   & -0.4933966462_wp, -0.4921372117_wp, -0.2957609261_wp, &
   & -0.8411432240_wp, -1.0658559860_wp, -0.5728818235_wp, -1.1044432120_wp, &
   & -0.5508446524_wp, -0.5067098981_wp, -0.2640009880_wp, &
   & -0.8928414589_wp, -0.2057103591_wp, -0.5683778831_wp, -0.3104379963_wp, &
   & -0.2262691068_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.8624419058_wp, -0.2970209466_wp, -0.5254301143_wp, -0.2271571872_wp, &
   & -0.2999310258_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.7499314401_wp, -0.4439088742_wp, -0.5704214138_wp, -0.3191904385_wp, &
   & -0.4341267285_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.7760604098_wp, -0.6900272120_wp, -0.5746255068_wp, -0.4738798235_wp, &
   & -0.5441724090_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -1.0978130236_wp, -0.3645358838_wp, -0.6362465468_wp, -0.5026698653_wp, &
   & -0.5909072447_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.9277788935_wp, -0.4706581291_wp, -0.6080781705_wp, -0.3916995044_wp, &
   & -0.3775660646_wp,  0.0000000000_wp,  0.0000000000_wp], shape(hla))

   !> CN dependence in H0
   !> see Eq. 10 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: klh(max_shell, max_elem) = reshape([&
   &  0.0798017435_wp,  0.0024847884_wp,  0.0075830283_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.8177539033_wp, -0.0000309929_wp, -0.0184556453_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.1480264523_wp, -0.0018958714_wp, -0.0003957978_wp, -0.0001748561_wp, &
   & -0.1746188172_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0073370719_wp,  0.1962234377_wp,  0.0001695612_wp,  0.0058374378_wp, &
   & -0.0692931361_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0030269102_wp,  0.0002823733_wp, -0.0663479533_wp, -0.0053338427_wp, &
   & -0.0330089284_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0097565700_wp,  0.0015142079_wp, -0.0707256409_wp,  0.0010582322_wp, &
   & -0.0031495896_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0215463846_wp,  0.0040695772_wp, -0.0971957668_wp, -0.0394882038_wp, &
   & -0.0018488690_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.1392781521_wp, -0.0319860712_wp, -0.0809331023_wp, -0.0617466410_wp, &
   & -0.0377157488_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0004308290_wp,  0.0068111300_wp, -0.0519981619_wp, -0.0734998360_wp, &
   & -0.0004113586_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0000642853_wp,  0.0026690007_wp, -0.0006940183_wp, -0.0015637912_wp, &
   &  0.1664801731_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0089138548_wp, -0.0767480664_wp, -0.0002456847_wp, -0.0041871159_wp, &
   &  0.0009747205_wp, -0.0067784482_wp,  0.0000000000_wp, &
   & -0.0049925564_wp, -0.0492636218_wp, -0.0000141074_wp, -0.0830208948_wp, &
   &  0.0010075556_wp, -0.2240195516_wp,  0.0000000000_wp, &
   &  0.0739082533_wp, -0.0552072738_wp,  0.0022915673_wp, -0.0125633164_wp, &
   & -0.0383809868_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0054223899_wp,  0.0007542326_wp,  0.0002919618_wp, -0.0453760688_wp, &
   & -0.0056083586_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0079442793_wp,  0.0005323892_wp, -0.0471075045_wp, -0.0181311234_wp, &
   & -0.0120313194_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000635795_wp,  0.0030517181_wp, -0.0373745520_wp, -0.0170088965_wp, &
   & -0.0025690089_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0013150245_wp,  0.0526378615_wp, -0.0341374143_wp, -0.0343687394_wp, &
   & -0.0046766666_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0023412960_wp, -0.0016819443_wp, -0.0362203655_wp, -0.0006020014_wp, &
   &  0.0471541272_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0087911539_wp, -0.0661770061_wp, -0.0001629337_wp, -0.0036932950_wp, &
   &  0.0008133984_wp, -0.0025510274_wp,  0.0000000000_wp, &
   & -0.0025475501_wp, -0.0229747956_wp,  0.0121828945_wp, -0.0045556642_wp, &
   &  0.0036786751_wp, -0.0339894127_wp,  0.0000000000_wp, &
   &  0.0843997090_wp, -0.0057461733_wp,  0.0151668773_wp, -0.0003793421_wp, &
   &  0.0059435157_wp, -0.0010138910_wp, -0.0388343287_wp, &
   &  0.0621563006_wp, -0.0023053686_wp,  0.0262987365_wp, -0.0042915478_wp, &
   &  0.0040447073_wp, -0.0014243231_wp, -0.0039741740_wp, &
   &  0.1416221457_wp, -0.0022307574_wp,  0.0224382861_wp, -0.0014820416_wp, &
   &  0.0040081944_wp, -0.0016524285_wp, -0.0196012720_wp, &
   &  0.0608033267_wp, -0.0113115297_wp,  0.0124212748_wp, -0.0048797464_wp, &
   &  0.0027493569_wp, -0.0170356895_wp, -0.0787488735_wp, &
   &  0.0578151755_wp, -0.0073255162_wp,  0.0339996979_wp, -0.0027986192_wp, &
   &  0.0015897866_wp, -0.0307677732_wp, -0.0761858330_wp, &
   &  0.0287868713_wp, -0.0038948934_wp,  0.2508470593_wp, -0.0123465367_wp, &
   &  0.0016764925_wp, -0.0256518789_wp, -0.0634474424_wp, &
   &  0.0195693716_wp, -0.0060212058_wp,  0.0408715600_wp, -0.0105993990_wp, &
   &  0.0028871588_wp, -0.0376566258_wp, -0.0521130074_wp, &
   &  0.0073531660_wp, -0.0368737932_wp,  0.0491294079_wp, -0.0355477102_wp, &
   &  0.0043864898_wp, -0.0888228540_wp, -0.1042086416_wp, &
   &  0.0250929772_wp, -0.0105103871_wp,  0.0478857173_wp, -0.0147340962_wp, &
   &  0.0041413314_wp, -0.1183516604_wp, -0.0931038104_wp, &
   &  0.0110753799_wp, -0.0114799680_wp,  0.0316731097_wp, -0.0102764156_wp, &
   &  0.0393882978_wp, -0.1233832162_wp, -0.0876507305_wp, &
   &  0.0500195465_wp, -0.0192291561_wp, -0.0000012024_wp, -0.0302287486_wp, &
   & -0.0057099414_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0051848778_wp,  0.0023189218_wp,  0.0007106613_wp, -0.0126841543_wp, &
   & -0.0005895725_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0142938726_wp,  0.0041781210_wp, -0.0306389410_wp, -0.0100277451_wp, &
   & -0.0014701184_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0000088977_wp, -0.0062039043_wp, -0.0312101439_wp, -0.0150641405_wp, &
   &  0.0153703314_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0004738837_wp,  0.0034704887_wp, -0.0220913643_wp, -0.0011238396_wp, &
   & -0.0024779298_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0183087398_wp,  0.0134177578_wp, -0.0034261023_wp, -0.0000743115_wp, &
   &  0.0015832961_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0103284163_wp, -0.0407794967_wp, -0.0004201828_wp, -0.0027023162_wp, &
   &  0.0025994405_wp, -0.0021101360_wp,  0.0000000000_wp, &
   & -0.0038332205_wp, -0.0364714622_wp,  0.0172713102_wp, -0.0083384222_wp, &
   &  0.0020683607_wp, -0.0172649810_wp,  0.0000000000_wp, &
   &  0.0627426092_wp, -0.0034877952_wp,  0.0106400823_wp, -0.0023054451_wp, &
   &  0.0018631897_wp, -0.0018699088_wp, -0.0059391038_wp, &
   &  0.0029502274_wp, -0.0068083609_wp,  0.3880467952_wp, -0.0151848134_wp, &
   &  0.0072770599_wp, -0.0093933940_wp, -0.0057657529_wp, &
   &  0.0053152659_wp, -0.0124180304_wp,  0.4144551949_wp, -0.0176081359_wp, &
   &  0.0032751498_wp, -0.0594722811_wp, -0.0051244459_wp, &
   &  0.0147308741_wp, -0.0174836400_wp,  0.0928991927_wp, -0.0055560832_wp, &
   &  0.0005619444_wp, -0.0304343854_wp, -0.0082853668_wp, &
   &  0.0013442067_wp, -0.0772757458_wp,  0.0326091835_wp, -0.0107554715_wp, &
   &  0.0083480057_wp, -0.0440922808_wp, -0.0028330180_wp, &
   &  0.0088103070_wp, -0.0071768768_wp,  0.1912508604_wp, -0.0130982153_wp, &
   &  0.0046527572_wp, -0.0361986768_wp, -0.0038708709_wp, &
   &  0.0149721194_wp, -0.0115227009_wp,  0.2038656554_wp, -0.0100160204_wp, &
   &  0.0361137407_wp, -0.0142191791_wp, -0.0018231590_wp, &
   &  0.0036278714_wp, -0.0064015555_wp,  0.1039112809_wp, -0.0019528453_wp, &
   &  0.0043032593_wp, -0.0667607904_wp, -0.0012737889_wp, &
   &  0.0181115892_wp,  0.0015936314_wp,  0.0507636810_wp,  0.0021480309_wp, &
   &  0.0012352932_wp, -0.0568390007_wp, -0.0309687064_wp, &
   & -0.0089417471_wp, -0.0018902305_wp,  0.0163479800_wp, -0.0058118879_wp, &
   &  0.1126790103_wp, -0.0484813913_wp, -0.0210897225_wp, &
   &  0.0080307484_wp, -0.0261697440_wp,  0.0005138299_wp, -0.0277017249_wp, &
   & -0.0135793214_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0073204391_wp,  0.0023528891_wp,  0.0018401283_wp, -0.0095364348_wp, &
   & -0.0011274543_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0070228179_wp,  0.0190242778_wp,  0.0006492129_wp,  0.0004458931_wp, &
   &  0.0059698270_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000506828_wp,  0.0013626684_wp, -0.0006424915_wp, -0.0132949571_wp, &
   &  0.0103918716_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0079936336_wp,  0.0010284846_wp, -0.0208210805_wp, -0.0176519627_wp, &
   & -0.0033969313_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0964953056_wp,  0.0028226422_wp,  0.0008363555_wp, -0.0067946086_wp, &
   &  0.0021721828_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0230065164_wp, -0.0293616511_wp, -0.0001147519_wp, -0.0126689793_wp, &
   &  0.0013852772_wp, -0.0023328985_wp,  0.0000000000_wp, &
   & -0.0023911250_wp, -0.0221656932_wp,  0.0349158532_wp, -0.0035898907_wp, &
   &  0.0012755884_wp, -0.0122741343_wp,  0.0000000000_wp, &
   &  0.0056084576_wp, -0.0101326439_wp,  0.2998641443_wp, -0.1328015762_wp, &
   &  0.0072489792_wp, -0.0024263980_wp, -0.0115226093_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  0.0024874401_wp, -0.0034030746_wp,  0.3442679260_wp, -0.0141134542_wp, &
   &  0.0062899271_wp, -0.0223722214_wp, -0.0056201484_wp, &
   &  0.0019054419_wp, -0.0018263062_wp,  0.3405305808_wp, -0.1844990269_wp, &
   &  0.0034050297_wp, -0.0490216851_wp, -0.0026656087_wp, &
   &  0.0000105708_wp, -0.0009300320_wp,  0.3917015367_wp, -0.0421555261_wp, &
   &  0.0005664914_wp, -0.0121513249_wp, -0.0008711138_wp, &
   &  0.0129148692_wp, -0.0294897548_wp,  0.0106724866_wp, -0.0077930940_wp, &
   &  0.0016845693_wp, -0.0485806060_wp, -0.0029698396_wp, &
   &  0.0060433937_wp, -0.0170829645_wp,  0.1636192981_wp, -0.0043645013_wp, &
   &  0.0009310616_wp, -0.0160575212_wp, -0.0096424086_wp, &
   &  0.0097368019_wp, -0.0142526087_wp,  0.2370300980_wp, -0.0092008530_wp, &
   &  0.0044645293_wp, -0.0374425618_wp, -0.0059881267_wp, &
   &  0.0040343212_wp,  0.0009683496_wp,  0.1205514685_wp, -0.0130536110_wp, &
   &  0.0597859129_wp, -0.0458629722_wp, -0.0002454508_wp, &
   &  0.0022987265_wp, -0.0066961508_wp,  0.0297785961_wp, -0.0043319230_wp, &
   &  0.0096100180_wp, -0.0978127533_wp, -0.0206523347_wp, &
   &  0.0017042004_wp, -0.0062315471_wp,  0.0042432757_wp,  0.0000213066_wp, &
   &  0.0484131993_wp, -0.1240528881_wp, -0.2116772233_wp, &
   &  0.0016563874_wp, -0.0376454593_wp,  0.0004909722_wp, -0.0416787009_wp, &
   &  0.0034235003_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0006936659_wp,  0.0035466959_wp,  0.0059396277_wp, -0.0171345872_wp, &
   & -0.0004197173_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0028153673_wp,  0.0028317139_wp,  0.0017520841_wp,  0.0022600199_wp, &
   &  0.0251340864_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0005129660_wp,  0.0006561777_wp, -0.0002336562_wp, -0.0026069069_wp, &
   &  0.0011152398_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0725594483_wp,  0.0155926909_wp, -0.0218709544_wp, -0.0176706796_wp, &
   & -0.0002493202_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000004988_wp,  0.0002517408_wp,  0.0000031741_wp, -0.0077919538_wp, &
   &  0.0162494489_wp,  0.0000000000_wp,  0.0000000000_wp], shape(klh))

   !> K_CN' - scales the CN' in H0
   !> see Eq. 10 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kcnstar(max_elem) = [&
   & -0.5846979150_wp,  0.0140884357_wp,  0.0266390147_wp,  0.0758494067_wp,  0.1309832428_wp, &
   & -0.0161789724_wp, -0.0057925311_wp,  0.4608250620_wp,  0.0951339460_wp,  0.2397739102_wp, &
   &  0.0090344647_wp,  0.0720324870_wp,  0.1668571747_wp,  0.0235901339_wp,  0.1310445989_wp, &
   &  0.2279536781_wp,  0.0077589008_wp,  0.0004622898_wp,  0.0063863069_wp,  0.0801725301_wp, &
   &  0.0814284267_wp,  0.1656840772_wp,  0.2881061695_wp,  0.0950782940_wp,  0.0901344118_wp, &
   &  0.0402889003_wp,  0.1144649602_wp,  0.0007898026_wp,  0.0189746230_wp,  0.0048229970_wp, &
   &  0.0035287736_wp,  0.0086579052_wp,  0.1384930710_wp,  0.1634566951_wp,  0.0091218173_wp, &
   &  0.0105756936_wp,  0.0062657794_wp,  0.0787611723_wp,  0.0282459028_wp,  0.0039724087_wp, &
   &  0.0029470579_wp,  0.0045425044_wp,  0.0011030895_wp,  0.0182719411_wp,  0.0068486541_wp, &
   &  0.0185641682_wp,  0.0014226915_wp,  0.0044033323_wp,  0.2187029026_wp,  0.1086625359_wp, &
   &  0.0139090388_wp,  0.2473910779_wp,  0.2800286603_wp,  0.4440556184_wp,  0.0163578289_wp, &
   &  0.0196091374_wp,  0.0051973246_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  0.0031799462_wp,  0.0041782784_wp,  0.0007620812_wp, &
   &  0.0042018158_wp,  0.0104282632_wp,  0.0245244905_wp,  0.0020251933_wp,  0.0073816786_wp, &
   &  0.0040800417_wp,  0.0081351053_wp,  0.0424322042_wp,  0.2997259823_wp,  0.2317415080_wp, &
   &  0.0283644207_wp,  0.8697289777_wp]

   !> k_shift - scales the additional CN' dependence in H0
   !> see Eq. 10 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kshift(max_elem) = [&
   & -0.0092695988_wp, -0.0265952987_wp,  0.0026109185_wp, -0.0014531059_wp,  0.0004100136_wp, &
   & -0.0060810427_wp, -0.0092582164_wp,  0.0154629366_wp,  0.0001201342_wp, -0.0138199419_wp, &
   &  0.0002737244_wp,  0.0001499417_wp,  0.0022276101_wp, -0.0034527645_wp, -0.0007854979_wp, &
   & -0.0007598140_wp, -0.0016172901_wp, -0.0046847853_wp,  0.0026302108_wp,  0.0000791939_wp, &
   & -0.0000187160_wp, -0.0000793887_wp, -0.0000402804_wp, -0.0000454801_wp, -0.0000262468_wp, &
   & -0.0000160972_wp, -0.0020294249_wp, -0.0007407228_wp, -0.0005209347_wp, -0.0008794727_wp, &
   &  0.0027761326_wp, -0.0009486624_wp, -0.0013291058_wp, -0.0003779178_wp, -0.0016388308_wp, &
   & -0.0079435368_wp,  0.0031674138_wp,  0.0003053313_wp, -0.0000369132_wp, -0.0012247384_wp, &
   & -0.0049783361_wp, -0.0064985110_wp, -0.0058680122_wp, -0.0058872121_wp, -0.0079338367_wp, &
   & -0.0090979509_wp, -0.0035541788_wp, -0.0042180144_wp,  0.0019597862_wp,  0.0009397925_wp, &
   & -0.0000616172_wp, -0.0049840688_wp,  0.0024255447_wp, -0.0032341882_wp,  0.0035837857_wp, &
   &  0.0001700973_wp, -0.0000855103_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.0006985861_wp, -0.0069941771_wp, -0.0040236980_wp, &
   & -0.0075983831_wp, -0.0080693199_wp, -0.0119056331_wp, -0.0084181213_wp, -0.0038357307_wp, &
   & -0.0034352663_wp,  0.0002038086_wp,  0.0002126157_wp, -0.0023192173_wp, -0.0057812322_wp, &
   &  0.0014796832_wp,  0.0000835650_wp]

   !> Atomic radii r_f for CN calculation
   !> see Eq. 11 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: rf(max_elem) = [&
   &  0.1159873118_wp, -0.0000262447_wp, -0.1979977296_wp,  0.7164493573_wp,  0.6828926032_wp, &
   &  0.9229984410_wp,  1.0526756706_wp,  1.0180198722_wp,  0.9549483164_wp,  0.5944195588_wp, &
   &  1.3748594260_wp,  1.5010279743_wp,  1.0107903073_wp,  1.5866269160_wp,  1.4736201698_wp, &
   &  1.5859247918_wp,  1.7401477571_wp,  1.3712836980_wp,  1.6885454030_wp,  1.8280128441_wp, &
   &  0.1617153290_wp,  1.2415896731_wp,  1.4392931243_wp,  1.5338158754_wp,  1.5495861879_wp, &
   &  1.2695521423_wp,  1.0880205219_wp,  1.5344831847_wp,  1.3547454124_wp,  1.5706687871_wp, &
   &  1.3722915645_wp,  1.5806711133_wp,  1.7506328102_wp,  1.7692068370_wp,  1.9777908274_wp, &
   &  1.9151108914_wp,  1.3605128698_wp,  1.9089344402_wp,  0.1311875584_wp,  1.3851546094_wp, &
   &  1.2794422889_wp,  1.7909318877_wp,  1.6334745329_wp,  1.4889419250_wp,  1.4300146933_wp, &
   &  1.5831922375_wp,  1.9639562123_wp,  1.8374005508_wp,  1.6398784145_wp,  1.7611194463_wp, &
   &  1.8489418774_wp,  2.0994236797_wp,  2.2631721343_wp,  2.0657566420_wp,  2.3575179939_wp, &
   &  2.4373594210_wp,  1.2985020319_wp,  &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  1.2811514222_wp,  1.3780340723_wp,  1.4116159367_wp, &
   &  1.8590793929_wp,  1.6980905622_wp,  1.5516554838_wp,  1.6173276189_wp,  1.7044761260_wp, &
   &  0.2015771174_wp,  1.8059465386_wp,  2.0374935375_wp,  1.9190988590_wp,  2.3066017233_wp, &
   &  2.4663689708_wp,  2.3458830442_wp]

   !----------- APPROXIMATED VXC SCALING --------------

   !> kxc1 - scales the GGA exchange-correlation potential in iteration 1
   real(wp), parameter :: kxc1(max_elem) = [&
   &  0.4737591670_wp,  3.4791079720_wp,  0.9425101911_wp,  1.4088951834_wp,  0.9533042982_wp, &
   &  1.1824710671_wp,  0.8999267113_wp,  0.8184983656_wp,  0.6732945612_wp,  0.3207262862_wp, &
   &  0.2192192026_wp,  0.5101373141_wp,  1.5567149210_wp,  0.5757036996_wp,  1.3521795960_wp, &
   &  1.2226599764_wp,  0.6051719427_wp,  0.6625342395_wp,  0.2575373095_wp,  1.9897626363_wp, &
   &  1.8826157632_wp,  1.7225392249_wp,  1.0948426789_wp,  0.9373961899_wp,  1.0745699539_wp, &
   &  0.6212502877_wp,  0.9128062191_wp,  0.8101561244_wp,  1.1774843519_wp,  0.8850779906_wp, &
   &  0.8724385760_wp,  1.2800147818_wp,  0.9520500371_wp,  1.0547767037_wp,  0.5414424877_wp, &
   &  0.9912911706_wp,  0.0894190026_wp,  1.6349056213_wp,  1.2403574811_wp,  1.9797326328_wp, &
   &  1.7876155913_wp,  1.3139022258_wp,  1.4152067053_wp,  1.3035116411_wp,  1.2859731055_wp, &
   &  1.6321616879_wp,  1.1289676956_wp,  1.8837287841_wp,  0.7430207975_wp,  1.1836329122_wp, &
   &  0.9990654158_wp,  1.0078451344_wp,  0.5487303166_wp,  1.1163279461_wp,  0.0282217764_wp, &
   &  1.3543078545_wp,  1.1047495111_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  2.0556992184_wp,  1.7402283299_wp,  1.5870371187_wp, &
   &  1.5075309369_wp,  1.5288412809_wp,  1.4998029568_wp,  1.5447026677_wp,  1.1577357102_wp, &
   &  1.7720285364_wp,  0.7361912692_wp,  1.3527621566_wp,  1.0907517306_wp,  0.9262367196_wp, &
   &  0.6138425637_wp,  0.8546304985_wp]


   !> kxc2l - scales the ECP-like exchange-correlation potential in iteration 2
   !> see Eq. 12 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kxc2l(max_shell, max_elem) = reshape([&
   &  0.0224227250_wp,  2.2534752584_wp,  5.2849740124_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2787165702_wp,  0.7653176998_wp,  0.0039186389_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  2.3065572211_wp,  0.2068150546_wp,  0.1120789521_wp,  0.3306647715_wp, &
   &  0.8961452077_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  2.6968601218_wp,  0.1484230693_wp,  0.7102013662_wp,  0.1475469587_wp, &
   &  0.7510687739_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0543116801_wp,  1.7346302467_wp,  0.4096824177_wp,  0.0546027373_wp, &
   &  0.9279083375_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.3860463725_wp,  0.2349692745_wp,  0.5627485328_wp,  0.5176681001_wp, &
   &  1.6475528368_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.2821994814_wp,  1.6260059763_wp,  0.3530496726_wp,  0.4025354578_wp, &
   &  0.6809421720_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.4042731062_wp,  0.0983977638_wp,  0.3218586069_wp,  0.3845599151_wp, &
   &  0.0279839596_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.4182731280_wp,  0.0005694815_wp,  0.3387845244_wp,  0.5900743992_wp, &
   & -0.0022111486_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.3461064713_wp,  0.0267012966_wp,  0.4417152799_wp,  0.0966673630_wp, &
   &  7.3018447520_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.7428493182_wp,  0.1191414792_wp,  0.0925016691_wp,  0.5134781881_wp, &
   &  0.2576090515_wp,  0.2046206405_wp,  0.0000000000_wp, &
   &  1.0192082118_wp,  0.0522868824_wp,  0.5213134501_wp,  0.1141267902_wp, &
   &  0.2600850191_wp,  1.0555335395_wp,  0.0000000000_wp, &
   &  0.1792196129_wp,  2.8412847832_wp,  0.5791385366_wp,  0.0031722731_wp, &
   & -0.0269646785_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0382306475_wp,  3.8886469342_wp,  0.0880161972_wp,  1.8795563839_wp, &
   &  0.7945524541_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.2391300598_wp,  1.2647610773_wp,  0.4606969163_wp,  0.1711215211_wp, &
   &  0.0566304072_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.3610369929_wp,  0.1511571602_wp,  0.4464069730_wp,  0.2895583131_wp, &
   &  0.1222453224_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.3568188549_wp,  0.0498437188_wp,  0.2097049601_wp,  0.4533714758_wp, &
   &  0.0013475762_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.8824016740_wp,  0.0009293885_wp,  0.6250641424_wp,  0.0083900769_wp, &
   &  1.2867552330_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.6486840650_wp,  0.0513355229_wp,  0.4329965062_wp,  0.6491687261_wp, &
   &  0.3434791277_wp,  0.1718855383_wp,  0.0000000000_wp, &
   &  2.4257176531_wp,  0.0666714936_wp,  0.4223083908_wp,  0.5310410159_wp, &
   &  0.2782431699_wp,  0.2509397463_wp,  0.0000000000_wp, &
   &  2.2383703946_wp,  1.0948645303_wp,  0.3475109881_wp,  2.4568335063_wp, &
   &  0.1316996730_wp,  0.5216799356_wp,  0.3039486072_wp, &
   &  1.8383557841_wp,  1.6864278843_wp,  0.0192626131_wp,  2.1273751407_wp, &
   &  0.0554238047_wp,  1.2892096676_wp,  0.0387357875_wp, &
   &  2.8816167052_wp,  1.5382886287_wp,  0.0102185454_wp,  1.6081421616_wp, &
   &  0.0242147089_wp,  1.0066923287_wp,  0.0124336179_wp, &
   &  2.1823376371_wp,  1.3386293514_wp,  0.0064096368_wp,  0.5338972071_wp, &
   &  0.1148440212_wp,  0.4666232632_wp,  0.1967628977_wp, &
   &  1.8916648724_wp,  2.2186347187_wp,  0.0119449887_wp,  0.6169592368_wp, &
   &  0.0018635048_wp,  0.6338121295_wp, -0.0007404744_wp, &
   &  1.6338537275_wp,  2.6074474559_wp,  0.0091520515_wp,  0.2037085419_wp, &
   &  0.0329293454_wp,  0.5628665721_wp,  0.0113393325_wp, &
   &  2.8553166880_wp,  1.3442974381_wp,  0.0205642543_wp,  0.3517195506_wp, &
   &  0.0329859698_wp,  0.6876859019_wp,  0.0286918062_wp, &
   &  2.8636467782_wp,  1.6363227562_wp,  0.0337937544_wp,  0.4262056804_wp, &
   &  0.0730439851_wp,  0.7336133105_wp,  0.2410334268_wp, &
   &  2.3602609873_wp,  2.4293310036_wp,  0.3000349224_wp,  0.4637934464_wp, &
   &  0.0591212142_wp,  0.7839180440_wp,  0.2874745295_wp, &
   &  2.5683303706_wp,  2.3086868840_wp,  0.1754033547_wp,  0.6331042867_wp, &
   &  0.0317043603_wp,  0.8527095839_wp,  0.2729317183_wp, &
   &  0.3760203398_wp,  0.6203504360_wp,  0.2726323448_wp,  1.0161700496_wp, &
   &  0.6084972513_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0444028116_wp,  1.0143323976_wp,  0.5866380979_wp,  0.9278439274_wp, &
   &  1.5136001388_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1120516275_wp,  1.3172008845_wp,  0.2546476043_wp,  0.3912488528_wp, &
   &  0.2611525538_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.2970309230_wp,  1.2368834512_wp,  0.3151071806_wp,  0.6294098083_wp, &
   &  0.0416657046_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.2960709523_wp,  0.3316158598_wp,  0.1728295793_wp,  0.2954667690_wp, &
   &  0.0060785380_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1161977536_wp,  0.0025455771_wp,  0.6375109163_wp,  0.4139810837_wp, &
   &  0.0008568333_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.5681811642_wp,  0.2245615867_wp,  0.0931728024_wp,  0.4417897426_wp, &
   &  0.1581752355_wp,  0.1317249828_wp,  0.0000000000_wp, &
   &  2.9029495699_wp,  0.0503868238_wp,  0.4238530808_wp,  0.4042559615_wp, &
   &  0.3953394340_wp,  0.2039123622_wp,  0.0000000000_wp, &
   &  1.8255559319_wp,  1.2238096208_wp,  0.0244260705_wp,  2.1610911557_wp, &
   &  0.1803576219_wp,  0.6614411968_wp,  0.0190537972_wp, &
   &  2.3017805440_wp,  1.4432497231_wp,  0.1196363382_wp,  0.8910336136_wp, &
   &  0.0957326204_wp,  0.8973789972_wp,  0.0019782614_wp, &
   &  1.4638628400_wp,  1.9270170987_wp,  0.0745210605_wp,  0.5568460542_wp, &
   &  0.1735513002_wp,  1.1508991453_wp,  0.0047692119_wp, &
   &  0.4304851210_wp,  2.2247931523_wp,  0.0292278037_wp,  0.7659121657_wp, &
   &  0.1096434021_wp,  0.8700880510_wp,  0.0030342211_wp, &
   &  1.4346220752_wp,  1.0247291145_wp,  0.0252343518_wp,  0.8901229375_wp, &
   &  0.0307919800_wp,  0.9628904494_wp,  0.0020424939_wp, &
   &  1.5866463716_wp,  0.7029843167_wp,  0.0215094464_wp,  0.4661070133_wp, &
   &  0.1871880567_wp,  0.8552049963_wp,  0.0057605752_wp, &
   &  1.6580790207_wp,  0.5627127550_wp,  0.0107126255_wp,  0.8764219298_wp, &
   &  0.0202339201_wp,  0.9590943587_wp,  0.0066949568_wp, &
   &  2.8542200630_wp,  2.2365398085_wp,  0.0038301878_wp,  1.0114024167_wp, &
   &  0.0268973542_wp,  1.1069171585_wp,  0.0024467155_wp, &
   &  1.7941134710_wp,  2.7213383167_wp,  0.4074432399_wp,  1.0378735473_wp, &
   &  0.0103095136_wp,  0.7780913748_wp,  0.0099231968_wp, &
   &  2.3909634047_wp,  1.6478827277_wp,  0.6061219424_wp,  0.9581416855_wp, &
   &  0.0174256553_wp,  1.3996053101_wp,  0.0317683183_wp, &
   &  0.2041306100_wp,  0.0785831610_wp,  0.1802579382_wp,  0.0074994839_wp, &
   &  0.6832077389_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.3357366814_wp,  0.1369115480_wp,  0.5293962175_wp,  0.2405063771_wp, &
   &  1.2700030372_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.2417168721_wp,  1.2218737467_wp,  0.4164153383_wp,  0.1434576997_wp, &
   &  0.4756306184_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.3209912118_wp,  0.4003806893_wp,  0.5837355442_wp,  0.0066909325_wp, &
   &  0.0139956399_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1356838324_wp,  0.3129733044_wp,  0.1026552644_wp,  0.6868410449_wp, &
   &  0.0275079355_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0102741415_wp,  0.0011954084_wp,  0.8500133459_wp,  0.0221989973_wp, &
   &  0.0304070726_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.1789689917_wp,  0.3549003022_wp,  0.3131192676_wp,  0.4762286333_wp, &
   &  0.1969649087_wp,  0.3181332000_wp,  0.0000000000_wp, &
   &  1.9441744418_wp,  0.0604782324_wp,  0.1013873795_wp,  0.3924960590_wp, &
   &  0.7990870998_wp,  0.0867067664_wp,  0.0000000000_wp, &
   &  0.6784549063_wp,  0.8765431700_wp,  0.8696572019_wp,  0.6378785114_wp, &
   &  0.0286632964_wp,  0.2906475027_wp,  0.0015253807_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  1.3423389642_wp,  3.3458263665_wp,  1.6635441472_wp,  0.9983780447_wp, &
   &  0.0695066926_wp,  0.8606384553_wp,  0.0036698644_wp, &
   &  1.8771541839_wp,  2.7399733812_wp,  0.1754534970_wp,  0.5842180582_wp, &
   &  0.0573441221_wp,  0.8135744314_wp,  0.0043409381_wp, &
   &  2.1453890160_wp,  2.3271194903_wp,  0.2424129414_wp,  0.8035785757_wp, &
   &  0.0027049195_wp,  1.0368032177_wp,  0.0132875825_wp, &
   &  1.9520418171_wp,  2.8084021693_wp,  0.0148244788_wp,  0.3520508253_wp, &
   &  0.0440472143_wp,  1.0508288206_wp,  0.0066945339_wp, &
   &  3.1981977694_wp,  1.7511424543_wp,  0.0204927707_wp,  0.9282416122_wp, &
   &  0.1357639845_wp,  1.1073918383_wp,  0.0059191208_wp, &
   &  3.1478575757_wp,  2.3954238108_wp,  0.0300003005_wp,  0.9386320565_wp, &
   &  0.0706883974_wp,  1.2103131060_wp,  0.0041820784_wp, &
   &  3.4368935151_wp,  0.2673498242_wp,  0.0069960235_wp,  1.2095405269_wp, &
   & -0.0057037103_wp,  1.1296433668_wp, -0.0007871494_wp, &
   &  2.4712459916_wp,  2.7878929478_wp,  0.5032340528_wp,  0.8673376952_wp, &
   &  0.0484212970_wp,  0.8522243322_wp,  0.1794038294_wp, &
   &  4.1293198799_wp,  2.9921583956_wp,  0.3954750899_wp,  1.1633381607_wp, &
   &  0.0441985924_wp,  0.9263382804_wp,  0.0600090916_wp, &
   &  0.3105032274_wp,  0.0223599310_wp,  0.1084513852_wp,  0.1673335403_wp, &
   &  0.4356921178_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0777562257_wp,  1.3682178161_wp,  0.5751087805_wp,  0.7307337456_wp, &
   &  2.4807120121_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0944454217_wp,  1.4344554737_wp,  0.4842552792_wp,  0.1911481808_wp, &
   &  0.9258516779_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.2089777100_wp,  1.5104191956_wp,  0.5093107447_wp,  0.1085288319_wp, &
   &  0.0154522052_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1489863910_wp,  0.1876215309_wp,  0.0940982510_wp,  0.8767737708_wp, &
   &  0.0121057572_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.6116006730_wp,  0.0067027208_wp,  0.3464479343_wp,  0.7688877377_wp, &
   & -0.0000004187_wp,  0.0000000000_wp,  0.0000000000_wp], shape(kxc2l))

   !----------- BASIS EXP SCALING DATA --------------

   !> Basic exponent scaling parameter for the modified overlap matrix
   !> see Eq. 13 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kalphah0l(max_shell, max_elem) = reshape([&
   &  1.2148553604_wp,  0.8784379294_wp,  0.9292666347_wp,  1.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.7289129128_wp,  0.8175679692_wp,  0.9608607570_wp,  1.0000000000_wp, &
   &  1.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  2.6639346346_wp,  0.9708466297_wp,  1.1841758984_wp,  1.1149738042_wp, &
   &  1.0316043556_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.7214870876_wp,  1.2573963914_wp,  1.0715615523_wp,  1.1082825071_wp, &
   &  1.0897499607_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.3146436629_wp,  0.8383558799_wp,  1.1834964487_wp,  0.9189377779_wp, &
   &  1.0205336689_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.4818942947_wp,  0.8676739352_wp,  1.0668595005_wp,  0.9899725408_wp, &
   &  1.0253468290_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.6752684026_wp,  0.8304154248_wp,  0.9928181340_wp,  1.0176426602_wp, &
   &  1.1344728972_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.6990540626_wp,  0.8081647748_wp,  0.9304494749_wp,  0.9932590534_wp, &
   &  1.1796305414_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.3017137247_wp,  0.9308948985_wp,  0.8131078701_wp,  0.9461529729_wp, &
   &  1.1342440835_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  2.4307859598_wp,  1.2330693618_wp,  0.9216829820_wp,  1.2576263704_wp, &
   &  0.9499810880_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.3564213766_wp,  1.0914388995_wp,  1.1479482878_wp,  1.2677480997_wp, &
   &  1.1173090720_wp,  1.1036198821_wp,  0.0000000000_wp, &
   &  1.1646965853_wp,  1.5760763300_wp,  1.3672963746_wp,  0.9821036438_wp, &
   &  1.0764858130_wp,  1.0497344848_wp,  0.0000000000_wp, &
   &  1.1977740912_wp,  0.9137019055_wp,  1.0934164321_wp,  0.9650727506_wp, &
   &  1.0949515930_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.1571863468_wp,  0.9267732136_wp,  1.1948702860_wp,  0.9404004029_wp, &
   &  0.9986995612_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.1008826807_wp,  1.0610935833_wp,  1.0692948287_wp,  1.0095078064_wp, &
   &  0.9895740320_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2567153868_wp,  0.9395809732_wp,  1.0055017761_wp,  0.9660884612_wp, &
   &  1.0274692089_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2807959898_wp,  1.1080370577_wp,  0.9941646779_wp,  0.9612946850_wp, &
   &  1.0608247454_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.6436934517_wp,  1.1064695157_wp,  1.2668678442_wp,  0.9823492048_wp, &
   &  1.0638634976_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2413269885_wp,  1.2587213867_wp,  1.1826488071_wp,  1.2149593609_wp, &
   &  0.9629357105_wp,  1.0650966497_wp,  0.0000000000_wp, &
   &  1.2130711417_wp,  0.3809002946_wp,  1.0789378157_wp,  1.3983714292_wp, &
   &  0.9890542819_wp,  0.9537860705_wp,  0.0000000000_wp, &
   &  1.4740250796_wp,  1.5211415757_wp,  1.3178338754_wp,  1.4747516279_wp, &
   &  1.0746999681_wp,  0.8056364870_wp,  1.1109527815_wp, &
   &  1.6400139188_wp,  1.6207651039_wp,  1.6612728440_wp,  1.2256986335_wp, &
   &  1.1934149485_wp,  0.7134780398_wp,  1.2068936622_wp, &
   &  1.4952632411_wp,  1.5831031375_wp,  1.3247212307_wp,  1.0895806981_wp, &
   &  1.2397844646_wp,  0.3684300602_wp,  1.3042247857_wp, &
   &  1.1331586934_wp,  1.3908403293_wp,  1.7699587040_wp,  1.0140974315_wp, &
   &  1.2300400394_wp,  0.3522774764_wp,  1.0447727776_wp, &
   &  2.8267228951_wp,  2.5526224173_wp,  1.5965087637_wp,  1.2020864483_wp, &
   &  1.0907216999_wp,  0.3631502151_wp,  1.2006627437_wp, &
   &  1.1961408054_wp,  1.2245061149_wp,  1.1869897714_wp,  1.1688842922_wp, &
   &  1.0275998470_wp,  0.2584563279_wp,  1.2513876205_wp, &
   &  1.1797049138_wp,  1.3320188925_wp,  1.5101215826_wp,  1.3195962337_wp, &
   &  1.0862180531_wp,  0.2161015065_wp,  1.1116709662_wp, &
   &  1.4099594960_wp,  1.5439571802_wp,  0.9517287366_wp,  1.0372664290_wp, &
   &  0.8545713390_wp,  0.1608107751_wp,  1.0608595804_wp, &
   &  1.1154247044_wp,  1.3118368788_wp,  1.1724958784_wp,  1.1343042477_wp, &
   &  0.8938708937_wp,  0.1906903670_wp,  1.0134047413_wp, &
   &  1.5243685184_wp,  1.8982844710_wp,  1.0765944473_wp,  1.3213008816_wp, &
   &  1.0470901269_wp,  0.2153372363_wp,  1.1851871318_wp, &
   &  1.2576608921_wp,  0.8780707056_wp,  1.1290376974_wp,  0.8407584760_wp, &
   &  1.0128980487_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.1877590316_wp,  0.8858879068_wp,  1.0709331972_wp,  0.9774101663_wp, &
   &  1.0084817083_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.1409264479_wp,  1.0490334711_wp,  1.0982728999_wp,  1.0254258673_wp, &
   &  1.0010113747_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2972577179_wp,  1.0687306658_wp,  1.0670472797_wp,  0.9780791385_wp, &
   &  1.0527285026_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2153881607_wp,  0.9849458791_wp,  1.0391088743_wp,  0.9482287295_wp, &
   &  1.0496419474_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.6232396761_wp,  1.1079701776_wp,  0.9490007892_wp,  0.9650831713_wp, &
   &  1.0827145876_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2470259999_wp,  1.2102323302_wp,  1.0093579086_wp,  1.2256114648_wp, &
   &  0.9896216648_wp,  1.0046144815_wp,  0.0000000000_wp, &
   &  0.9428253769_wp,  0.8784890515_wp,  1.0383543268_wp,  1.1539840026_wp, &
   &  1.0352550746_wp,  0.9588958227_wp,  0.0000000000_wp, &
   &  1.3038834618_wp,  1.2537110563_wp,  1.2282035770_wp,  1.3701235315_wp, &
   &  1.0890454867_wp,  0.9385265840_wp,  1.0177731028_wp, &
   &  1.4653164298_wp,  1.5301393093_wp,  1.2949842432_wp,  1.4842735504_wp, &
   &  1.0349376714_wp,  0.9960479044_wp,  0.8545498429_wp, &
   &  1.4606228127_wp,  1.5294614320_wp,  1.6443648890_wp,  1.4650981917_wp, &
   &  1.1239858283_wp,  0.9767117116_wp,  0.9269288652_wp, &
   &  1.6406055498_wp,  1.6862774067_wp,  1.8868468537_wp,  1.4419192681_wp, &
   &  1.1288956256_wp,  1.0172185169_wp,  0.9605230354_wp, &
   &  2.1864133565_wp,  1.9936188267_wp,  1.4524779237_wp,  1.7375866607_wp, &
   &  1.0006937361_wp,  0.9957027347_wp,  1.0361467811_wp, &
   &  1.8887846027_wp,  1.7319260733_wp,  1.3807039549_wp,  1.5496442815_wp, &
   &  1.1743790182_wp,  0.8475413444_wp,  0.9551871368_wp, &
   &  1.4231720785_wp,  1.5238894624_wp,  1.5360323890_wp,  1.3981601944_wp, &
   &  1.2315931011_wp,  0.8661649269_wp,  0.9698237664_wp, &
   &  1.5997764878_wp,  1.5003579972_wp,  0.9590262720_wp,  1.3687556832_wp, &
   &  1.0903983473_wp,  0.7327361459_wp,  0.9248389052_wp, &
   &  1.3083552690_wp,  1.3217603839_wp,  1.0799500831_wp,  1.2557699152_wp, &
   &  1.0060963935_wp,  0.7426789685_wp,  0.8647114496_wp, &
   &  1.6986243888_wp,  1.8151129514_wp,  1.0634003360_wp,  1.3691903902_wp, &
   &  1.1225896192_wp,  0.6382350177_wp,  0.8310163716_wp, &
   &  1.2539898330_wp,  0.9050163618_wp,  1.0293161396_wp,  0.8560018905_wp, &
   &  1.1453856571_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.1012556372_wp,  0.8985138262_wp,  1.0536182456_wp,  1.0204090085_wp, &
   &  0.9503012870_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.0566442748_wp,  1.0953828039_wp,  1.0628372314_wp,  1.0352081452_wp, &
   &  1.0431288056_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.1926739368_wp,  1.1420047808_wp,  1.0728950334_wp,  0.9468027268_wp, &
   &  1.2560299005_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.3880925543_wp,  1.2916999711_wp,  1.0738801270_wp,  0.9263262904_wp, &
   &  1.0364700684_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.4365201590_wp,  0.8376518552_wp,  1.0332209285_wp,  0.9947738081_wp, &
   &  1.0304395687_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2432597438_wp,  1.3439134633_wp,  0.9122186319_wp,  1.2561227631_wp, &
   &  1.0343769478_wp,  0.8217281421_wp,  0.0000000000_wp, &
   &  1.0671151116_wp,  1.0078072387_wp,  1.1626398831_wp,  1.1007005279_wp, &
   &  0.9808051164_wp,  0.9980540445_wp,  0.0000000000_wp, &
   &  1.9522093286_wp,  1.7241128101_wp,  1.0765032077_wp,  1.5160353826_wp, &
   &  1.0923290767_wp,  0.9971417716_wp,  0.8769573457_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  1.0461748122_wp,  1.1139857911_wp,  1.4273297320_wp,  1.6685617559_wp, &
   &  1.2739320383_wp,  1.0305753096_wp,  1.0671922584_wp, &
   &  1.4497044467_wp,  1.9531782596_wp,  1.3862920947_wp,  1.5862128712_wp, &
   &  1.2275718001_wp,  0.9968777885_wp,  1.0936965348_wp, &
   &  1.5288628997_wp,  2.0534005244_wp,  1.4676215190_wp,  1.4311946656_wp, &
   &  1.0330485168_wp,  1.0006904265_wp,  0.9327152667_wp, &
   &  1.7335479563_wp,  2.0700796155_wp,  1.5312710882_wp,  1.2933123385_wp, &
   &  0.9739854037_wp,  1.0053812462_wp,  1.0307487458_wp, &
   &  1.1772143782_wp,  1.8788006478_wp,  1.6162610683_wp,  1.2684848123_wp, &
   &  0.9723102573_wp,  0.9585057447_wp,  0.9142559538_wp, &
   &  1.2602942370_wp,  1.7421725494_wp,  1.2079668299_wp,  1.2356462778_wp, &
   &  1.2138323004_wp,  1.1045414341_wp,  1.0456742905_wp, &
   &  1.7670014146_wp,  1.7785102330_wp,  0.9806335264_wp,  1.2036884930_wp, &
   &  0.9613266285_wp,  1.0294798634_wp,  0.9828904139_wp, &
   &  1.0948845899_wp,  1.2666968935_wp,  1.1614474671_wp,  1.2608688266_wp, &
   &  1.0882035676_wp,  1.0732359842_wp,  0.9169528363_wp, &
   &  1.2656344227_wp,  0.8876671233_wp,  1.0732642825_wp,  1.3490465266_wp, &
   &  1.1093079799_wp,  0.9897963369_wp,  0.8366778019_wp, &
   &  1.2053682398_wp,  0.9886730789_wp,  1.0424110028_wp,  0.9840254776_wp, &
   &  1.1937450259_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.0734922213_wp,  1.0322051081_wp,  1.0242428688_wp,  0.9754891093_wp, &
   &  0.9626550345_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.0370176197_wp,  1.1958320903_wp,  1.0090644240_wp,  1.0292624987_wp, &
   &  1.0578754229_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2182895055_wp,  1.2852946574_wp,  1.0365340723_wp,  1.0318446323_wp, &
   &  1.1314451909_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.8797129719_wp,  1.0175814571_wp,  1.0401813758_wp,  1.0235764213_wp, &
   &  1.0550594360_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2275561991_wp,  0.9667985310_wp,  0.9873716626_wp,  0.9668379080_wp, &
   &  1.0012652889_wp,  0.0000000000_wp,  0.0000000000_wp], shape(kalphah0l))

   !> Exponent scaling for overlap integrals
   !> see Eq. 14 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: klalphaxc(max_shell, max_elem) = reshape([&
   &  1.4260939011_wp, 1.1354702797_wp, 1.1469455922_wp, 1.0000000000_wp, &
   &  1.0000000000_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.8042973327_wp, 1.0065148665_wp, 0.8510026526_wp, 1.0000000000_wp, &
   &  1.0000000000_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.7477989788_wp, 1.0125665854_wp, 0.8102984364_wp, 1.0061278923_wp, &
   &  1.1253246769_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.9707969232_wp, 0.9122684392_wp, 1.0197270176_wp, 1.1928690370_wp, &
   &  0.7906491144_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.1979048155_wp, 1.2466925592_wp, 1.0662891137_wp, 1.4932968030_wp, &
   &  0.6935563015_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.8523567063_wp, 1.0945916760_wp, 0.9441704117_wp, 1.0818007603_wp, &
   &  0.9268429239_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.7301767862_wp, 1.2770717941_wp, 1.0409735644_wp, 1.1057189871_wp, &
   &  0.8367404219_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.6446861483_wp, 1.3828857673_wp, 0.9987900390_wp, 1.0177700438_wp, &
   &  0.6847918497_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.6503325510_wp, 1.2230668963_wp, 0.9992380098_wp, 0.9979590280_wp, &
   &  0.8068626905_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.4131243089_wp, 2.2705369003_wp, 0.7648826025_wp, 1.1762411740_wp, &
   &  0.9283492390_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.9655434458_wp, 0.9631082896_wp, 0.7408454315_wp, 0.9929477518_wp, &
   &  0.9759644534_wp, 0.9892233582_wp, 1.0000000000_wp, &
   &  1.0974214107_wp, 1.0802561243_wp, 0.7669449348_wp, 1.2412272403_wp, &
   &  0.8358787511_wp, 0.9524117160_wp, 1.0000000000_wp, &
   &  2.3415233426_wp, 0.9207632311_wp, 1.5242402849_wp, 1.1367830704_wp, &
   &  0.8119740535_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  3.8324243277_wp, 0.9800307886_wp, 1.0450874835_wp, 1.4524922550_wp, &
   &  0.7244909860_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.1557962223_wp, 1.0068524367_wp, 1.6615661095_wp, 1.2840632598_wp, &
   &  0.8416045413_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.7583062269_wp, 1.3270488340_wp, 1.6221643950_wp, 1.2595756283_wp, &
   &  0.8322137542_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.8216405818_wp, 0.9295644504_wp, 1.1816290155_wp, 1.2387875356_wp, &
   &  0.6340826633_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.6310305156_wp, 0.2837319539_wp, 0.8076760233_wp, 1.1816508981_wp, &
   &  0.6631491612_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.4867142723_wp, 0.8374226511_wp, 0.7555388339_wp, 0.9737457311_wp, &
   &  0.9123855728_wp, 1.0346525989_wp, 1.0000000000_wp, &
   &  1.2813202379_wp, 1.1391684135_wp, 1.0035863069_wp, 0.9230927254_wp, &
   &  0.9234595625_wp, 1.1390315546_wp, 1.0000000000_wp, &
   &  0.9230499807_wp, 1.2152800211_wp, 1.0033403307_wp, 0.8726930136_wp, &
   &  0.9393930095_wp, 0.9555933653_wp, 1.5824796093_wp, &
   &  0.8737072767_wp, 1.4263347753_wp, 0.9750741619_wp, 1.0748942417_wp, &
   &  0.9638449675_wp, 0.8708829637_wp, 1.1372241215_wp, &
   &  1.2173466996_wp, 1.3569259308_wp, 1.2719250744_wp, 1.0759759767_wp, &
   &  0.9217788931_wp, 0.9032036960_wp, 0.7378955204_wp, &
   &  1.7051132133_wp, 1.6844081136_wp, 1.5007591003_wp, 0.9199572035_wp, &
   &  1.2420084428_wp, 0.9278243954_wp, 1.6390586239_wp, &
   &  1.1389696769_wp, 2.0394016722_wp, 1.3400940915_wp, 0.9238699466_wp, &
   &  0.8223071061_wp, 0.8246916999_wp, 0.6171856195_wp, &
   &  1.5633989939_wp, 1.8825867980_wp, 1.7776199076_wp, 0.7704361738_wp, &
   &  0.8722798158_wp, 0.7557801370_wp, 0.8936294415_wp, &
   &  1.3846460727_wp, 1.4544605944_wp, 1.1789432766_wp, 0.6975380021_wp, &
   &  0.7509856731_wp, 1.2186423974_wp, 1.0493277094_wp, &
   &  1.1256527549_wp, 1.2227451581_wp, 1.2937611350_wp, 0.9874127380_wp, &
   &  0.9657291691_wp, 1.0420886354_wp, 0.9702705236_wp, &
   &  1.2146052109_wp, 1.2064593975_wp, 1.0988569568_wp, 0.9965870108_wp, &
   &  1.0059076307_wp, 0.9588071017_wp, 0.9965780231_wp, &
   &  1.1521156201_wp, 1.0122147628_wp, 1.0666185294_wp, 1.0001032241_wp, &
   &  1.0051247354_wp, 0.9936097297_wp, 0.9943576758_wp, &
   &  1.3687880831_wp, 0.9806700681_wp, 1.5770629179_wp, 1.2297406897_wp, &
   &  0.7889913292_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.4697514025_wp, 1.0427028876_wp, 1.5099855012_wp, 1.3608890078_wp, &
   &  0.8180754643_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.8805830015_wp, 0.8484419509_wp, 1.1919831947_wp, 1.1788000870_wp, &
   &  1.0889001380_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.8907657428_wp, 1.0628451016_wp, 1.4904188353_wp, 1.1562397251_wp, &
   &  0.9038584470_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.4740368632_wp, 0.7275625867_wp, 1.2932544605_wp, 1.4848578729_wp, &
   &  0.7876882233_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.5883619927_wp, 1.6854351199_wp, 1.8821883786_wp, 1.3236191320_wp, &
   &  1.3434723054_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.6119420376_wp, 0.7722648484_wp, 0.9991628899_wp, 0.9919219281_wp, &
   &  0.7696797189_wp, 0.9693981618_wp, 1.0000000000_wp, &
   &  1.3835734679_wp, 1.1131384315_wp, 1.0985394501_wp, 1.0086661889_wp, &
   &  0.7812147822_wp, 1.0575507314_wp, 1.0000000000_wp, &
   &  0.9863376721_wp, 1.1812174143_wp, 0.9601308347_wp, 1.0410816291_wp, &
   &  0.8817889309_wp, 1.5826459067_wp, 1.3970017206_wp, &
   &  1.0474572636_wp, 1.1422337751_wp, 1.2414185791_wp, 0.8433329802_wp, &
   &  0.8878936620_wp, 1.4086379850_wp, 1.6200424448_wp, &
   &  1.1971519271_wp, 1.4300721230_wp, 0.9305127959_wp, 0.8731770959_wp, &
   &  0.7981106860_wp, 1.2990635453_wp, 1.3741831056_wp, &
   &  0.8259706718_wp, 1.4789495960_wp, 1.0316194577_wp, 0.9886561357_wp, &
   &  0.9032263777_wp, 1.0730637255_wp, 1.3925113204_wp, &
   &  1.6556005304_wp, 2.0360020739_wp, 2.0815039618_wp, 1.1364980864_wp, &
   &  0.5910664937_wp, 1.0708496529_wp, 1.9167176815_wp, &
   &  1.0739570777_wp, 1.2750413207_wp, 1.5161885103_wp, 0.8669189783_wp, &
   &  0.8942067536_wp, 1.3681701390_wp, 0.7336195707_wp, &
   &  1.1321900897_wp, 1.3279917652_wp, 1.1455207695_wp, 0.9797826116_wp, &
   &  0.8704078014_wp, 0.9292183745_wp, 0.8447828610_wp, &
   &  1.0758040056_wp, 1.3234550550_wp, 1.3973952948_wp, 0.9896784263_wp, &
   &  0.9704126486_wp, 1.0000307063_wp, 0.8459770776_wp, &
   &  1.1072012886_wp, 0.9311363322_wp, 1.1824685574_wp, 0.9761779967_wp, &
   &  1.0330887229_wp, 1.0667466073_wp, 1.0089439715_wp, &
   &  1.1656669258_wp, 1.0044816277_wp, 1.0011204702_wp, 0.9888954566_wp, &
   &  1.0060247840_wp, 1.0070566173_wp, 0.9926047400_wp, &
   &  1.3027423650_wp, 0.9769888023_wp, 1.3860063172_wp, 1.1761249064_wp, &
   &  0.8101245315_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.1514231988_wp, 0.9102901348_wp, 1.7568743474_wp, 0.9481112763_wp, &
   &  0.8973872754_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.1669096135_wp, 0.8313982366_wp, 1.3339187115_wp, 1.0374364918_wp, &
   &  0.6569785860_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.8083340865_wp, 1.1423751348_wp, 1.3029488128_wp, 2.0127459544_wp, &
   &  0.6743854166_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.6441791974_wp, 0.6555088210_wp, 1.4368418782_wp, 1.5750692535_wp, &
   &  0.7632986315_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.7314903634_wp, 2.3512940299_wp, 1.3119065255_wp, 1.0829904081_wp, &
   &  1.1694413950_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.0365977614_wp, 0.7591345063_wp, 1.3370941469_wp, 1.2032106207_wp, &
   &  0.8651925068_wp, 0.9381814379_wp, 1.0000000000_wp, &
   &  1.0889658133_wp, 0.8872637260_wp, 0.8027186167_wp, 1.0289394345_wp, &
   &  0.9457908735_wp, 0.9268551531_wp, 1.0000000000_wp, &
   &  0.7025220491_wp, 0.6346584025_wp, 1.0632898912_wp, 0.7801606150_wp, &
   &  0.8790518702_wp, 1.1109821592_wp, 1.4395023424_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  1.3221464435_wp, 1.7373781191_wp, 1.0900771199_wp, 0.7880820923_wp, &
   &  0.9388070107_wp, 1.4430309462_wp, 1.1327647957_wp, &
   &  1.5223484312_wp, 1.2114450022_wp, 1.3571702832_wp, 0.8029418690_wp, &
   &  0.7341648890_wp, 1.1656833843_wp, 0.8398270487_wp, &
   &  1.7639202346_wp, 1.3117308450_wp, 1.4600995990_wp, 0.9510382897_wp, &
   &  0.9887332853_wp, 1.3368299529_wp, 1.0917307970_wp, &
   &  1.4305331016_wp, 1.4189327931_wp, 1.0451577271_wp, 0.9566665177_wp, &
   &  0.8514184272_wp, 1.1448851450_wp, 1.2696891578_wp, &
   &  2.3008553427_wp, 1.1196563238_wp, 1.6582364032_wp, 0.9847066515_wp, &
   &  0.9557357850_wp, 1.0094323796_wp, 0.8444359324_wp, &
   &  1.3706229129_wp, 1.4538883015_wp, 1.3733070569_wp, 1.0664310633_wp, &
   &  0.8476265334_wp, 1.1087872178_wp, 0.8719878732_wp, &
   &  1.1405229422_wp, 1.1047297827_wp, 1.4358820967_wp, 0.9806124973_wp, &
   &  0.9934236785_wp, 1.0194587486_wp, 0.7965699965_wp, &
   &  1.4419559040_wp, 1.3169448725_wp, 1.5119733220_wp, 1.0046384995_wp, &
   &  1.0365777744_wp, 1.0407900101_wp, 1.0286995137_wp, &
   &  1.0269624037_wp, 1.0191466331_wp, 1.3182476831_wp, 0.9751151086_wp, &
   &  1.0174390457_wp, 1.0595489034_wp, 1.0797206366_wp, &
   &  0.9845772861_wp, 0.8542150777_wp, 1.3112793077_wp, 1.0450515672_wp, &
   &  0.6923964859_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.1173430963_wp, 0.7137809130_wp, 1.3201347375_wp, 1.1222693217_wp, &
   &  0.9857320342_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.9073549835_wp, 0.8933110632_wp, 1.2329866381_wp, 1.0659442561_wp, &
   &  0.8625528331_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.5253117114_wp, 1.2392715332_wp, 1.1870991268_wp, 1.6731175896_wp, &
   &  0.7638138520_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  1.5496506389_wp, 0.6922890920_wp, 1.4896232165_wp, 1.7410711546_wp, &
   &  0.8429537943_wp, 1.0000000000_wp, 1.0000000000_wp, &
   &  0.5816696364_wp, 0.8970780139_wp, 1.0526327927_wp, 1.2545845255_wp, &
   &  1.2551010522_wp, 1.0000000000_wp, 1.0000000000_wp], shape(klalphaxc))

   !> Iteration- and charge-dependent exponent scaling for modified overlap
   !> see Eq. 13 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kits0(max_elem) = [&
   &  0.0345563416_wp,  0.2490759409_wp,  0.0096921344_wp,  0.0052688847_wp,  0.0000927567_wp, &
   &  0.0049265913_wp,  0.0060303400_wp,  0.0044399052_wp,  0.0013813016_wp,  0.0002090231_wp, &
   &  0.0059579091_wp,  0.0049482940_wp,  0.0038703596_wp,  0.0031487403_wp,  0.0040479582_wp, &
   &  0.0012162410_wp,  0.0015421546_wp,  0.0186254938_wp,  0.0052962151_wp,  0.0044250152_wp, &
   &  0.0145289517_wp,  0.0054043812_wp,  0.0038465218_wp,  0.0144389526_wp,  0.0262104865_wp, &
   &  0.0513178560_wp,  0.1462222214_wp,  0.0278473160_wp,  0.0059094129_wp,  0.0056102427_wp, &
   &  0.0221848096_wp,  0.0065438478_wp,  0.0055430431_wp,  0.0258123946_wp,  0.0053341567_wp, &
   &  0.0003417099_wp,  0.0039265306_wp,  0.0066469869_wp,  0.0060781303_wp,  0.0038661558_wp, &
   &  0.0071985771_wp,  0.0063698424_wp,  0.0039024078_wp,  0.0046089460_wp,  0.0392553118_wp, &
   &  0.0219909268_wp,  0.0124399388_wp,  0.0081809371_wp,  0.0071670994_wp,  0.0086881972_wp, &
   &  0.0041826842_wp,  0.0162183342_wp,  0.0020432538_wp,  0.0006627796_wp,  0.0048416544_wp, &
   &  0.0085306980_wp,  0.0034488541_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  0.0044646465_wp,  0.0037643178_wp,  0.0164657429_wp, &
   &  0.0108526576_wp,  0.0189154795_wp,  0.0179109470_wp,  0.0606772576_wp,  0.0095351673_wp, &
   &  0.0347138541_wp,  0.1124703099_wp,  0.0158899766_wp,  0.0219319307_wp,  0.0319808290_wp, &
   &  0.0027623516_wp,  0.0051545980_wp]

   !-------------- APPROXIMATION FOR EFFECTIVE CORE POTENTIAL --------------

   !> Scaling of effective core potential epsilon
   !> see Eq. 15 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kecpepsilon(max_elem) = [&
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.1889593705_wp, &
   &  0.6715207621_wp,  0.7318094493_wp,  0.7908518047_wp,  0.0830118354_wp,  0.1765793578_wp, &
   &  0.0178867991_wp,  0.0675824505_wp,  0.2850009706_wp,  0.8784424412_wp,  0.2696488475_wp, &
   &  0.2739346272_wp,  0.1126010579_wp,  0.4571596922_wp,  0.0229558466_wp,  0.3162238982_wp, &
   &  0.0860799120_wp,  0.1050153013_wp,  0.1048805943_wp,  0.0999470140_wp,  0.0945367009_wp, &
   &  0.1161020662_wp,  0.1055399893_wp,  0.0958179107_wp,  0.0973418300_wp,  0.0988140175_wp, &
   &  0.3702459412_wp,  0.4281314268_wp,  0.1559189248_wp,  0.4741206151_wp,  0.4694724080_wp, &
   &  0.0454814023_wp,  0.0126828320_wp,  0.0334199228_wp,  0.0771917240_wp,  0.0987571825_wp, &
   &  0.1113723507_wp,  0.0962772906_wp,  0.0583363543_wp,  0.0611463030_wp,  0.1016590341_wp, &
   &  0.0546720741_wp,  0.0708508817_wp,  0.1072172445_wp,  0.2157289541_wp,  0.2940096811_wp, &
   &  0.2096555169_wp,  0.1889885116_wp,  0.5170256448_wp,  0.4316453946_wp,  0.0172221193_wp, &
   &  0.0363472013_wp,  0.1581451773_wp,  &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  0.1140491174_wp,  0.1216496513_wp,  0.1243518580_wp, &
   &  0.1024630959_wp,  0.1013971612_wp,  0.0731344063_wp,  0.1067889091_wp,  0.0463633976_wp, &
   &  0.0459298324_wp,  0.4442052314_wp,  0.2779168214_wp,  0.0141363077_wp,  0.2370368904_wp, &
   &  0.1613809443_wp,  0.2026459983_wp]

   !> Maximum core shell supported by PTB
   integer, parameter :: max_core_shell = 12

   integer, parameter :: max_core_prim = 6

   integer, parameter :: cbas_nshell(max_elem) = [ &
   & 0,  0,  0,  0,  1,  1,  1,  1,  1,  1, &
   & 1,  1,  3,  3,  3,  3,  3,  3,  3,  3, &
   & 3,  3,  3,  3,  3,  3,  3,  3,  3,  3, &
   & 6,  6,  6,  6,  6,  6,  6,  6,  6,  6, &
   & 6,  6,  6,  6,  6,  6,  6,  6,  9,  9, &
   & 9,  9,  9,  9,  9,  9,  9,  9,  9,  9, &
   & 9,  9,  9,  9,  9,  9,  9,  9,  9,  9, &
   & 9,  9,  9,  9,  9,  9,  9,  9,  9,  9, &
   &12, 12, 12, 12, 12, 12 ]

   !> Core basis Slater exponents
   !> Clementi's SZ Slater exponents from JCP 38 (1963), 2686 ibid 47 (1967), 1300.
   real(wp), protected :: cbas_sl_exp(max_core_shell, max_elem) = 0.0_wp

   !> Core basis principal quantum numbers
   integer, protected :: cbas_pqn(max_core_shell, max_elem) = 0

   !> Core basis angular momentum quantum numbers
   integer, protected :: cbas_angshell(max_core_shell, max_elem) = 0

   !> HF levels for core basis
   real(wp), protected :: cbas_hflev(max_core_shell, max_elem) = 0.0_wp

   !---------- ELECTROSTATICS ------------------------------

   !> Alpha parameter for EEQ model
   real(wp), parameter :: alpeeq(max_elem) = [&
   &  0.5657758564_wp,  0.5264324358_wp,  0.4076876973_wp,  0.8430348778_wp,  1.8417990011_wp, &
   &  1.2356870476_wp,  1.8227023952_wp,  0.9910065813_wp,  1.2171879303_wp,  0.9864571581_wp, &
   &  2.6376549487_wp,  0.7032651749_wp,  0.9495021748_wp,  1.0750637030_wp,  1.4156643264_wp, &
   &  1.7582971814_wp,  1.1989902540_wp,  1.5049984972_wp,  1.7709935834_wp,  0.8512715114_wp, &
   &  1.9341122876_wp,  1.4094226788_wp,  1.5883585458_wp,  1.7375612123_wp,  0.7725823620_wp, &
   &  2.0607630928_wp,  2.0145857488_wp,  1.3639272707_wp,  2.4052425125_wp,  1.6022619873_wp, &
   &  2.4240746035_wp,  2.2285197912_wp,  2.4273351003_wp,  1.0374977341_wp,  2.1844686980_wp, &
   &  0.9355412455_wp,  2.4946012118_wp,  1.6912703122_wp,  0.1961108119_wp,  2.2072579086_wp, &
   &  1.9839767620_wp,  3.0690541890_wp,  0.9187635425_wp,  1.7053269529_wp,  2.2186131385_wp, &
   &  1.0716920086_wp,  2.4049369507_wp,  1.5925371886_wp,  1.5489352558_wp,  1.6117292390_wp, &
   &  2.0611152059_wp,  5.5148052215_wp,  1.5823885590_wp,  1.0108324609_wp,  1.7459896596_wp, &
   &  0.8108752071_wp,  0.9684037673_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  2.8282251474_wp,  1.8390281498_wp,  1.8561417231_wp,  0.8343866989_wp, &
   &  1.9218876925_wp,  1.8270772624_wp,  1.3828025559_wp,  2.3643408340_wp,  4.5940836203_wp, &
   &  0.5168661972_wp,  2.8813043257_wp,  1.9567823098_wp,  1.9017746826_wp,  1.3816326453_wp, &
   &  1.3184427268_wp ]

   !> EN parameter for EEQ model
   real(wp), parameter :: chieeq(max_elem) = [&
   &  1.3324387940_wp,  0.8733576310_wp,  1.3505444112_wp,  1.2895941036_wp,  1.4259709044_wp, &
   &  1.6158437446_wp,  1.5781155782_wp,  1.5741332611_wp,  1.4957861964_wp,  0.6316393544_wp, &
   &  1.0172830975_wp,  0.6629374146_wp,  1.4025565964_wp,  1.0300183391_wp,  1.5220526900_wp, &
   &  1.6390906469_wp,  1.4184237620_wp,  0.9386379250_wp,  0.8026482983_wp,  0.2812122495_wp, &
   &  1.0138227413_wp,  1.2824910133_wp,  1.1980043463_wp,  1.3331083293_wp,  1.1365585040_wp, &
   &  1.3045704526_wp,  1.2448612188_wp,  1.0488575735_wp,  1.2382170932_wp,  1.0087621656_wp, &
   &  1.2864844599_wp,  0.9810489146_wp,  1.5029035345_wp,  1.6161788724_wp,  1.3145005745_wp, &
   &  0.7760299625_wp,  0.8856649366_wp,  0.5892001338_wp,  0.4068665973_wp,  1.2168285292_wp, &
   &  1.1983641772_wp,  1.1527436843_wp,  1.3946254815_wp,  1.3220703747_wp,  1.2143909182_wp, &
   &  1.7193841024_wp,  1.2984816745_wp,  0.9981662989_wp,  1.0994342198_wp,  1.0233563180_wp, &
   &  1.3442711029_wp,  1.5752786688_wp,  1.4252075161_wp,  0.8106394460_wp,  0.8474675731_wp, &
   &  0.1866630546_wp,  0.9234913939_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  1.2740602787_wp,  1.2542683508_wp,  1.1749624550_wp,  1.7534136603_wp, &
   &  1.1681614209_wp,  1.1087966201_wp,  1.6125467809_wp,  1.2683903082_wp,  1.2081751664_wp, &
   &  1.2450303937_wp,  1.1050549434_wp,  1.2630906941_wp,  1.5221797847_wp,  1.3701971305_wp, &
   &  0.9614477363_wp ]

   !> CN dependency parameter of EEQ model
   real(wp), parameter :: cnfeeq(max_elem) = [&
   &  0.0429433473_wp,  0.5224987110_wp, -0.0401414718_wp, -0.0280908302_wp, -0.0015137481_wp, &
   &  0.1408744650_wp,  0.1457845869_wp,  0.1690574187_wp,  0.1545931807_wp,  0.0612532325_wp, &
   & -0.1107215184_wp, -0.0942861034_wp, -0.0119730354_wp, -0.0081478459_wp,  0.0652698836_wp, &
   &  0.2197938426_wp,  0.2322622198_wp,  0.0211313957_wp, -0.0985759904_wp, -0.0902303411_wp, &
   & -0.0478550508_wp, -0.0078449301_wp, -0.0137691607_wp, -0.0088355875_wp, -0.0510844429_wp, &
   & -0.0042296887_wp, -0.0024004217_wp, -0.0265589752_wp, -0.0622286837_wp,  0.0193235579_wp, &
   & -0.0021121579_wp, -0.1034004076_wp,  0.0708928780_wp,  0.1357962494_wp,  0.1287464494_wp, &
   &  0.0311410920_wp, -0.1023026788_wp, -0.0748570226_wp, -0.0248558198_wp, -0.0217723692_wp, &
   & -0.0301884990_wp, -0.0451498002_wp, -0.0570976140_wp, -0.0213128508_wp, -0.0044448524_wp, &
   &  0.1690593112_wp,  0.0377573856_wp,  0.0191294735_wp, -0.1230880659_wp, -0.0439299152_wp, &
   &  0.0127229348_wp,  0.2009586869_wp,  0.1483677945_wp,  0.0058665809_wp, -0.0258738571_wp, &
   & -0.0606399879_wp, -0.0341091361_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp, -0.0104825200_wp, -0.0216005701_wp, -0.0606778940_wp, -0.0162314255_wp, &
   & -0.0302225239_wp, -0.0083748437_wp,  0.1666971438_wp,  0.0324108360_wp,  0.0154321648_wp, &
   & -0.0027540995_wp, -0.0115695788_wp, -0.0067927563_wp,  0.1355973623_wp,  0.1222063089_wp, &
   &  0.0035852141_wp ]

   !> Chemical hardness parameters for EEQ model
   real(wp), parameter :: gameeq(max_elem) = [&
   & -0.3501586000_wp,  0.1636673223_wp,  0.1237051131_wp,  0.0344412672_wp,  0.1874706313_wp, &
   &  0.2245967420_wp,  0.0884788849_wp,  0.1041840532_wp,  0.0292877327_wp,  0.3015023459_wp, &
   &  0.1971100591_wp,  0.0371707619_wp, -0.0407255207_wp,  0.1701912888_wp,  0.0248530164_wp, &
   &  0.4753433091_wp,  0.0073548274_wp,  0.2667053429_wp,  0.2107558349_wp,  0.0204207531_wp, &
   &  0.0513660842_wp,  0.0086189159_wp,  0.0053906791_wp,  0.0168014501_wp,  0.0034607133_wp, &
   &  0.0053183085_wp,  0.0120237498_wp,  0.0234311863_wp,  0.0129509725_wp,  0.0284309478_wp, &
   &  0.0008560387_wp,  0.0554049971_wp,  0.3157799670_wp,  0.2534773010_wp,  0.1164313649_wp, &
   &  0.3244809198_wp,  0.2983403735_wp,  0.0201305538_wp,  0.1169606011_wp,  0.0265334419_wp, &
   &  0.0115725394_wp,  0.0231932735_wp,  0.0030903705_wp,  0.0138615893_wp,  0.0035596456_wp, &
   &  0.0202668390_wp,  0.0402859632_wp,  0.0428340245_wp,  0.1129549211_wp,  0.1706769524_wp, &
   &  0.0269297390_wp,  0.3835139467_wp,  0.1474788245_wp,  0.4105328337_wp,  0.1721268161_wp, &
   &  0.0541951305_wp,  0.0561531002_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0189580980_wp,  0.0662819765_wp,  0.0025139491_wp,  0.0321981375_wp, &
   &  0.0249202623_wp,  0.0320425374_wp,  0.0097875822_wp,  0.0250017464_wp,  0.1680643036_wp, &
   &  0.1649480383_wp,  0.0257445649_wp,  0.0261153886_wp,  0.0104571301_wp,  0.0425577867_wp, &
   &  0.6923881184_wp ]

   !> Scaling of third-order electrostatics contribution
   !> see Eq. 18 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kto(max_elem) = [&
   &  0.1345540888_wp, -0.1721884044_wp, -0.0028282186_wp,  0.0170199168_wp, -0.0061401571_wp, &
   &  0.0089395630_wp,  0.0052029478_wp,  0.0181154137_wp,  0.0007501486_wp, -0.0785632658_wp, &
   & -0.0003227058_wp, -0.0003250842_wp, -0.0016913248_wp,  0.0291527052_wp,  0.0122897182_wp, &
   &  0.0017882021_wp, -0.0000271936_wp, -0.0000710653_wp, -0.0003737146_wp, -0.0006868439_wp, &
   & -0.0003298299_wp, -0.0033368241_wp, -0.0001090113_wp, -0.0000636885_wp, -0.0001581403_wp, &
   & -0.0000196986_wp, -0.0031528428_wp, -0.0199449346_wp, -0.0012578466_wp, -0.0013523753_wp, &
   & -0.0230726831_wp,  0.0088080696_wp,  0.0131819645_wp,  0.0022695898_wp, -0.0120801287_wp, &
   &  0.0033133056_wp, -0.0005828168_wp, -0.0009391686_wp,  0.0008538315_wp,  0.0006329497_wp, &
   &  0.0154027689_wp, -0.0001447286_wp, -0.0009705924_wp, -0.0001561802_wp, -0.0001575413_wp, &
   & -0.0067959011_wp, -0.0002423289_wp, -0.0002044170_wp, -0.0003342786_wp,  0.0090486660_wp, &
   &  0.0074338696_wp,  0.0101486139_wp, -0.0121249041_wp,  0.0080215995_wp, -0.0004453500_wp, &
   & -0.0035653432_wp,  0.0032479881_wp,  &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  0.0012706304_wp,  0.0057480337_wp,  0.0083385740_wp, &
   & -0.0024500790_wp, -0.0000672572_wp, -0.0007936986_wp, -0.0059408381_wp, -0.0004484650_wp, &
   & -0.0007491817_wp, -0.0000083894_wp,  0.0015549207_wp,  0.0096017175_wp,  0.0057597105_wp, &
   & -0.0069100744_wp,  0.0101950525_wp]

   !> Scaling of chemical hardness parameter eta from EEQ model in iteration 1
   !> see Eq. 19 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: keta1(max_shell, max_elem) = reshape([&
   &  1.4462087478_wp,  0.0278008485_wp,  2.9829828470_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.5705246669_wp,  2.1566715380_wp,  0.4846193957_wp,  1.0010000000_wp, &
   &  1.0000000000_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  3.3558785320_wp, 14.0376417112_wp,  0.0333659538_wp,  1.0719426415_wp, &
   &  2.8785795421_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  6.1670683552_wp,  2.2773901983_wp,  0.0269884422_wp,  3.1265256013_wp, &
   &  0.7877272414_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.1439112980_wp,  0.0245743396_wp,  2.6561624083_wp,  0.0887147293_wp, &
   &  4.7237010008_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.2267691121_wp,  0.6537699887_wp,  1.8449810489_wp,  0.4607044246_wp, &
   &  2.4513533518_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.3946974682_wp,  0.5232935157_wp,  1.2328973181_wp,  0.5657349727_wp, &
   &  1.6133370762_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.4902376917_wp,  0.8540951774_wp,  1.2847757741_wp,  0.9796594290_wp, &
   &  2.8211036111_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.3373877187_wp,  5.8200904301_wp,  0.9594024486_wp,  1.2942638413_wp, &
   &  2.3544778932_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.7394250511_wp,  1.2417034988_wp,  0.9783859247_wp,  0.6299014100_wp, &
   &  2.6960330353_wp,  0.0000000000_wp,  1.0000000000_wp, &
   &  3.0412244087_wp,  0.9295389325_wp,  1.4777030414_wp,  2.1160852872_wp, &
   &  0.7454149915_wp,  0.3809802533_wp,  1.0000000000_wp, &
   &  1.0891254719_wp, 32.2588153273_wp,  2.2757981242_wp,  4.5705694988_wp, &
   &  0.4505081921_wp,  2.4889249333_wp,  1.0000000000_wp, &
   & 14.7521922435_wp,  0.0299525071_wp,  0.8838129544_wp,  0.3368093205_wp, &
   &  4.1407610924_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  6.5821354295_wp,  0.3581149168_wp,  1.3769604035_wp,  0.4102120674_wp, &
   &  0.8181205629_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.5392898682_wp,  0.3806746294_wp,  3.4196393435_wp,  0.2951580628_wp, &
   &  0.6848981398_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.8881748796_wp,  0.9907160230_wp,  2.5126357646_wp,  0.3573484277_wp, &
   &  0.2260512096_wp,  0.0000000000_wp,  1.0000000000_wp, &
   & 23.1899519846_wp,  0.0229772746_wp,  0.8580729093_wp,  0.6719007344_wp, &
   &  0.1507692794_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  5.7818489374_wp,  0.0011738551_wp,  1.0883272545_wp,  0.3644718652_wp, &
   &  0.9977888706_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  2.7627183968_wp,  1.6486413559_wp,  0.7491716686_wp,  1.0093278693_wp, &
   &  2.4329063883_wp,  2.2371383163_wp,  1.0000000000_wp, &
   &  2.3885594736_wp,  0.2210269198_wp,  0.7315876356_wp,  1.8180546517_wp, &
   &  0.1583640462_wp,  2.7956965534_wp,  1.0000000000_wp, &
   &  1.0056325604_wp,  5.3471410568_wp,  0.8517956185_wp,  1.4889961617_wp, &
   &  1.3681756248_wp,  1.6250345826_wp,  0.5294214016_wp, &
   &  0.3980532840_wp,  4.1703446757_wp,  0.2476380787_wp,  0.8006942538_wp, &
   &  0.6726312690_wp,  6.6979180676_wp,  0.2735176853_wp, &
   &  2.8112970470_wp,  0.2314702186_wp,  0.0671255620_wp,  0.1285217467_wp, &
   &  1.2369564581_wp,  1.1798667662_wp,  1.2096406175_wp, &
   &  0.6419340742_wp,  0.1299287416_wp,  0.0587779015_wp,  0.1678253838_wp, &
   &  1.4307563616_wp,  4.7134609888_wp,  0.4488245750_wp, &
   &  1.6559757194_wp,  0.3737361599_wp,  0.0202182730_wp,  4.1014273691_wp, &
   &  0.9825546406_wp,  1.6347405690_wp,  2.5245930179_wp, &
   &  0.2190564150_wp,  0.0714193378_wp,  0.7872009508_wp,  0.7678276390_wp, &
   &  0.2203785895_wp,  3.3506468989_wp,  1.2029601369_wp, &
   &  0.0934150030_wp,  1.2944192847_wp,  0.8423077830_wp,  1.4396007342_wp, &
   &  1.5763869415_wp,  1.2262394587_wp,  0.7779862238_wp, &
   &  1.5695369548_wp,  0.5820916359_wp,  0.1359013162_wp,  2.8548289234_wp, &
   &  0.2742269396_wp,  1.1196252534_wp,  0.7001075218_wp, &
   &  0.4832757190_wp,  2.8997747678_wp,  0.2575278780_wp,  0.0636085712_wp, &
   &  2.1048238444_wp,  0.8524031522_wp,  0.9767783108_wp, &
   &  1.1163899741_wp,  1.4659382429_wp,  0.8334866514_wp,  0.0036671601_wp, &
   &  1.0904645324_wp,  2.4848317825_wp,  1.6666305448_wp, &
   &  0.1318292692_wp,  6.7836732040_wp,  1.7822999438_wp,  0.4059009108_wp, &
   &  0.9587847400_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.3629045535_wp,  0.1747129812_wp,  0.5033605453_wp,  0.0813595948_wp, &
   &  2.7058640809_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.1587533448_wp,  0.7653909614_wp,  3.6420760043_wp,  0.1537970158_wp, &
   &  0.7514530073_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.1151054728_wp,  0.0858739966_wp,  3.3132651804_wp,  0.1979701344_wp, &
   &  0.4506879427_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  6.3094379290_wp,  0.1319223725_wp,  1.0120090733_wp,  0.5373636794_wp, &
   &  0.1317371997_wp,  1.0000000000_wp,  1.0000000000_wp, &
   & 12.7199900338_wp,  0.0091480059_wp,  1.4459428777_wp,  0.1493581111_wp, &
   &  2.4282342388_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  2.0766422094_wp,  2.1072408832_wp,  0.8926376569_wp,  1.8565197298_wp, &
   &  0.5085483088_wp,  1.4605750219_wp,  1.0000000000_wp, &
   &  0.6611910736_wp,  2.4719219476_wp,  0.6948878559_wp,  1.5135654402_wp, &
   &  0.1574093385_wp,  1.6622375860_wp,  1.0000000000_wp, &
   &  0.9409812203_wp,  0.8006522957_wp,  1.1444783179_wp,  2.6948241459_wp, &
   &  3.7709789168_wp,  3.1060198231_wp,  2.0055430527_wp, &
   &  0.9607606185_wp,  1.2312408149_wp,  0.7710879230_wp,  0.2744892744_wp, &
   &  0.9069330084_wp,  2.2730197290_wp,  0.4633366155_wp, &
   &  0.7428661565_wp,  1.2731043698_wp,  0.8241989547_wp,  0.6445587999_wp, &
   &  1.5434545839_wp,  2.4176767070_wp,  0.3797055814_wp, &
   &  0.0949454434_wp,  1.5982663971_wp,  0.2503199915_wp,  0.1617666248_wp, &
   &  0.2074973174_wp,  2.0221784953_wp,  0.2367231842_wp, &
   &  2.3415856448_wp,  2.1259981439_wp, -0.0232671406_wp, -0.0490127286_wp, &
   & -0.6793032305_wp,  6.7837324298_wp,  0.5568943145_wp, &
   &  1.4406240826_wp,  1.1407988848_wp,  0.1850281574_wp,  0.1116386820_wp, &
   &  0.4607304356_wp,  4.2056112584_wp,  1.5891846459_wp, &
   &  2.2636053868_wp,  1.6268925109_wp,  0.4574710403_wp,  0.4072525670_wp, &
   &  1.2919050685_wp,  1.2920315870_wp,  0.7909958698_wp, &
   &  1.6683086587_wp,  2.7293498532_wp,  1.2824380478_wp, -0.0000025435_wp, &
   &  0.0259414355_wp,  2.9704408737_wp, 19.1391711482_wp, &
   &  1.2245690355_wp,  0.8006118590_wp,  0.2659067140_wp,  0.7319695853_wp, &
   &  0.1984557065_wp,  1.1212989293_wp,  0.9430600969_wp, &
   &  1.1254550510_wp,  1.5148287558_wp,  0.5249439316_wp,  0.0054979514_wp, &
   &  1.0875922341_wp,  3.2416299012_wp,  0.8704281052_wp, &
   &  1.0220004391_wp,  0.0540550581_wp,  0.3208238418_wp,  4.8143399344_wp, &
   &  3.7684649525_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.4241087894_wp,  0.2534955317_wp,  2.4906915165_wp,  0.1180784592_wp, &
   &  0.4110928486_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.0658654563_wp,  0.1253909767_wp,  2.1997532325_wp,  0.2412202681_wp, &
   &  1.3652168190_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.1095790261_wp,  0.2526366234_wp,  2.1258921840_wp,  0.8282396549_wp, &
   &  0.2568944388_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  2.4270127498_wp,  0.5931303229_wp,  1.1577348494_wp,  0.4616155405_wp, &
   &  0.9789150888_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  8.0673022489_wp,  1.0694881676_wp,  1.5655207383_wp,  0.0055660640_wp, &
   &  2.6134119639_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.4976229619_wp,  1.4111150908_wp,  1.6702236791_wp,  0.8504162901_wp, &
   &  0.7100447250_wp,  0.3944044645_wp,  1.0000000000_wp, &
   &  3.2897744595_wp,  0.2736086939_wp,  0.3263782257_wp,  5.4504051074_wp, &
   &  3.7030533494_wp,  8.0966335518_wp,  1.0000000000_wp, &
   &  2.5061273989_wp,  0.9491822713_wp,  0.5012002199_wp,  7.5691248795_wp, &
   &  6.1960404858_wp,  1.3416376535_wp,  0.9443191875_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  0.4554209325_wp,  3.3871210031_wp,  0.5861532582_wp,  0.1516538166_wp, &
   &  0.5393643856_wp,  1.2042759080_wp,  0.1362583365_wp, &
   &  2.0835414458_wp,  1.5876671078_wp,  1.9968779423_wp,  1.2592149061_wp, &
   &  0.4381333488_wp,  0.9747862561_wp,  0.2247960376_wp, &
   & 29.5589772231_wp,  0.0430027331_wp,  0.3763079093_wp,  0.0042356243_wp, &
   &  0.0831987337_wp,  1.4919704917_wp,  0.0657244671_wp, &
   &  1.9471145352_wp,  1.2469768497_wp,  0.0594923683_wp,  0.1518905160_wp, &
   &  0.6622389657_wp,  1.2262013932_wp,  0.1178612320_wp, &
   &  1.3879967135_wp,  0.7150944696_wp,  0.5851155790_wp,  0.9431034584_wp, &
   &  0.7858900198_wp,  0.7498064629_wp,  1.7801577783_wp, &
   &  2.1913506085_wp,  0.6253058448_wp,  0.2362026374_wp,  0.5095649890_wp, &
   &  0.5954756467_wp,  0.7856544208_wp,  0.9236305768_wp, &
   &  1.1878809826_wp,  1.2625877588_wp,  1.0669552116_wp,  0.0187064141_wp, &
   &  0.0201192778_wp,  1.0894313979_wp,  2.5329852364_wp, &
   &  0.7267686228_wp,  1.0513962180_wp,  0.2062503231_wp,  0.2637012389_wp, &
   &  0.4105786306_wp,  0.7426119242_wp,  1.0600138172_wp, &
   &  0.7846742967_wp,  0.8501083359_wp,  0.3884255325_wp,  0.0339109674_wp, &
   &  0.1024344876_wp,  0.4515095463_wp,  1.0611365566_wp, &
   &  3.1911536564_wp,  0.3476926392_wp,  5.9369245041_wp,  0.1081423490_wp, &
   &  0.9553746260_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.4824480029_wp,  0.1483877914_wp,  0.3554672156_wp,  1.4986103845_wp, &
   & 10.1991996423_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.1359369054_wp,  1.3631225895_wp,  4.9858067222_wp,  0.5211704434_wp, &
   &  1.4337500131_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.8891687363_wp,  1.6792113421_wp,  3.1992294249_wp,  1.5483961995_wp, &
   &  0.5585617314_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  5.2596093635_wp,  0.9454174807_wp,  1.8267386740_wp,  0.8683253252_wp, &
   &  0.2606352224_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  8.6768558722_wp,  1.6520576507_wp,  2.2983214543_wp,  1.0209253822_wp, &
   & 25.9089271453_wp,  1.0000000000_wp,  1.0000000000_wp], shape(keta1))

   !> Scaling of chemical hardness parameter eta from EEQ model in iteration 2
   !> see Eq. 21 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: keta2(max_shell, max_elem) = reshape([&
   &  1.2679904344_wp,  0.7250220171_wp,  2.4223440988_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.9073009109_wp,  0.3580079079_wp,  1.2926506971_wp,  1.0010000000_wp, &
   &  1.0000000000_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  4.5010038095_wp,  3.8610819367_wp,  0.0814654893_wp,  3.0751458228_wp, &
   &  0.9216084723_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  2.6525136603_wp,  2.2501272326_wp,  0.0297831466_wp,  2.3686720925_wp, &
   &  0.8161698193_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.7847837599_wp,  0.4529262820_wp,  2.0499795979_wp,  0.4189131772_wp, &
   &  0.7978213626_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.9155652999_wp,  0.5451823466_wp,  1.8371188929_wp,  0.4776941509_wp, &
   &  0.5745523991_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.3210264737_wp,  0.3458762130_wp,  1.4701925796_wp,  0.4069978728_wp, &
   &  0.5407927965_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.1850095988_wp,  0.2802859773_wp,  1.3307671313_wp,  0.3946097037_wp, &
   &  0.4246419201_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.9997845831_wp,  0.2283613864_wp,  1.1645729329_wp,  0.3687956735_wp, &
   &  0.4800574036_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.6189950190_wp,  0.4289870647_wp,  1.2827579182_wp,  0.5772881170_wp, &
   &  0.3406334009_wp,  0.0000000000_wp,  1.0000000000_wp, &
   &  3.3473281476_wp,  2.2170773927_wp,  0.5763998192_wp,  2.3351288567_wp, &
   &  1.4585600407_wp,  0.8099651396_wp,  1.0000000000_wp, &
   &  3.2730263484_wp,  3.2551655752_wp,  1.3757452961_wp,  0.8640246503_wp, &
   &  0.9988359924_wp,  3.7626874552_wp,  1.0000000000_wp, &
   &  1.9641666601_wp,  0.6165346624_wp,  2.1550889597_wp,  0.4023236500_wp, &
   &  1.3093258728_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.1141197009_wp,  0.3062120933_wp,  1.3811715205_wp,  0.4464487793_wp, &
   &  0.9850177675_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  1.2589424300_wp,  0.3672133873_wp,  1.7316390401_wp,  0.4517340132_wp, &
   &  0.8944511511_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.9086682621_wp,  0.6334961214_wp,  1.6121138969_wp,  0.4201688940_wp, &
   &  0.6986205862_wp,  0.0000000000_wp,  1.0000000000_wp, &
   &  0.7172954163_wp,  0.8917938028_wp,  1.3419627254_wp,  0.3026760282_wp, &
   &  0.5757222691_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.7219147956_wp,  0.1906986044_wp,  0.9711868389_wp,  0.9814947295_wp, &
   &  0.5046686664_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  4.6733491653_wp,  2.9175654790_wp,  0.6807498818_wp,  2.4586726113_wp, &
   &  0.8815371268_wp,  2.8145813698_wp,  1.0000000000_wp, &
   &  7.0487624642_wp,  0.1795271534_wp,  0.3649763005_wp,  1.0011628263_wp, &
   &  1.5692259688_wp,  3.0282239813_wp,  1.0000000000_wp, &
   &  0.9207001074_wp,  1.2870741625_wp,  0.6200166243_wp,  0.7953375042_wp, &
   &  1.1570994255_wp,  3.0889709143_wp,  0.2621785601_wp, &
   &  1.1536702335_wp,  0.7387806746_wp,  0.2087433077_wp,  0.7626549219_wp, &
   &  0.8397760868_wp,  2.6165771119_wp,  0.9266971384_wp, &
   &  0.9466491824_wp,  3.3353676649_wp,  0.0543000378_wp,  0.4264730727_wp, &
   &  0.8752336713_wp,  2.3021340862_wp,  0.9551098641_wp, &
   &  0.0541963086_wp,  0.2652068330_wp,  0.4993207147_wp,  0.6628219422_wp, &
   &  1.6983296367_wp,  2.0847551015_wp,  0.5951346911_wp, &
   &  0.2127112660_wp,  0.4320318773_wp,  0.3006023356_wp,  0.5657471137_wp, &
   &  1.9300816266_wp,  1.9087806244_wp,  0.8543820980_wp, &
   &  0.3431366249_wp,  0.6630992697_wp,  0.3460091637_wp,  0.5198955267_wp, &
   &  1.2257875794_wp,  1.5541741654_wp,  1.8464904200_wp, &
   &  0.3597041922_wp,  0.6423642312_wp,  0.3628101369_wp,  0.3780462227_wp, &
   &  1.0182781770_wp,  2.0015918838_wp,  0.8826033117_wp, &
   &  0.3181686169_wp,  0.9562993043_wp,  0.4700798953_wp,  0.1879133688_wp, &
   &  0.9175777345_wp,  2.0854493313_wp,  1.4191506479_wp, &
   &  1.0203111941_wp,  1.0783377765_wp,  0.5393274652_wp,  0.0148035804_wp, &
   &  0.6042632126_wp,  2.4482271950_wp,  1.4917571380_wp, &
   &  0.7250538996_wp,  2.1761621255_wp,  0.8126239355_wp,  0.0025379784_wp, &
   &  0.8801186836_wp,  2.1530635019_wp,  1.7429703716_wp, &
   &  1.7112745921_wp,  0.3298321588_wp,  1.8488583184_wp,  0.2880043006_wp, &
   &  0.5525688884_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.4300314553_wp,  0.3706881688_wp,  1.6685220359_wp,  0.3501796288_wp, &
   &  0.4732111443_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.9580771230_wp,  0.4667076808_wp,  1.1996969326_wp,  0.5217064563_wp, &
   &  0.6134744932_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.7628108101_wp,  0.5134198288_wp,  1.3843263701_wp,  0.4479725887_wp, &
   &  0.6097947342_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.1675469345_wp,  1.2808783118_wp,  1.2164207125_wp,  0.3845641128_wp, &
   &  0.4290611639_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.5624279574_wp,  0.3013214869_wp,  1.3049265059_wp,  0.4757420993_wp, &
   &  0.4018434346_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  3.3058594305_wp,  0.9927664195_wp,  0.6949802067_wp,  3.1414565403_wp, &
   &  0.7180216680_wp,  1.2237054285_wp,  1.0000000000_wp, &
   &  2.1395688676_wp,  2.7054808127_wp,  0.3959225733_wp,  0.8682401955_wp, &
   &  1.5400477641_wp,  2.7197443601_wp,  1.0000000000_wp, &
   &  0.4555620001_wp,  0.3772164848_wp,  1.3972275217_wp,  0.9673637179_wp, &
   &  2.6084773643_wp,  2.4065443006_wp,  0.4094093749_wp, &
   &  2.2561972499_wp,  1.9389093123_wp,  0.5583177842_wp,  1.0658809973_wp, &
   &  1.6324660114_wp,  2.2490122720_wp,  0.4830255048_wp, &
   &  0.9180943947_wp,  0.5263742282_wp,  0.5247239106_wp,  0.6369935884_wp, &
   &  1.3042247699_wp,  1.9643699739_wp,  0.4595834182_wp, &
   &  1.1672074984_wp,  0.2796528853_wp,  0.0331320047_wp,  1.0272359004_wp, &
   &  1.5873538115_wp,  2.2318823440_wp,  0.3405906842_wp, &
   &  1.5815129138_wp,  1.4811199464_wp,  0.2628572019_wp,  1.8079818812_wp, &
   &  1.0656816869_wp,  2.0275114064_wp,  1.0257070399_wp, &
   &  6.1295648023_wp,  2.2612494410_wp,  0.4366236173_wp,  0.3720291408_wp, &
   &  1.2565926591_wp,  2.0613912012_wp,  0.7447281892_wp, &
   &  3.6822958840_wp,  2.1894008346_wp,  0.1983098308_wp,  0.4153906728_wp, &
   &  0.8196883480_wp,  2.0619178475_wp,  0.9189301722_wp, &
   &  2.8481102254_wp,  2.5224782746_wp,  0.4392883712_wp, -0.0000877885_wp, &
   &  0.9184731029_wp,  2.2106621529_wp,  0.8246580863_wp, &
   &  2.4543878070_wp,  1.3131737256_wp,  0.6281428174_wp,  0.0626284492_wp, &
   &  0.4132121581_wp,  2.3467296108_wp,  0.9417825526_wp, &
   &  0.7735572825_wp,  0.9335625499_wp,  1.7138597552_wp,  0.0039439887_wp, &
   &  1.1425344139_wp,  1.5849043249_wp,  0.9245985686_wp, &
   &  0.7484208562_wp,  0.3111160573_wp,  1.4768275736_wp,  0.1854194366_wp, &
   &  0.8494411703_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.6157490386_wp,  0.3068900055_wp,  1.3670334115_wp,  0.4579752863_wp, &
   &  0.6650939909_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.7955610287_wp,  1.0495504783_wp,  1.3947303139_wp,  0.3341873231_wp, &
   &  0.4516566789_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  0.6307738755_wp,  0.4372143832_wp,  0.9010091881_wp,  1.0075269085_wp, &
   &  0.3761894253_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.5960292182_wp,  0.7190026920_wp,  1.1398738704_wp,  0.4129860929_wp, &
   &  0.7394842985_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.4675876412_wp,  0.5408446262_wp,  1.0812088386_wp,  0.3534701067_wp, &
   &  0.3181111120_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  2.2509429861_wp,  1.7323907156_wp,  2.1354722978_wp,  4.5835456064_wp, &
   &  1.2502794208_wp,  0.8641416219_wp,  1.0000000000_wp, &
   &  1.8157156370_wp,  3.2684684200_wp,  1.8484029695_wp,  3.2982233260_wp, &
   &  2.8948117713_wp,  7.9317203222_wp,  1.0000000000_wp, &
   &  1.1202897357_wp,  0.1683711368_wp,  1.5990212310_wp,  2.7766038933_wp, &
   &  3.6551661036_wp,  3.6343566294_wp,  0.5087178333_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  1.0701721181_wp,  0.9316872643_wp,  0.3237489507_wp,  1.1610562170_wp, &
   &  0.5806598571_wp,  1.2656879790_wp,  0.2887280283_wp, &
   &  1.3255588071_wp,  1.1392264714_wp,  0.4819066850_wp,  0.9330352166_wp, &
   &  0.9004975435_wp,  1.1343124694_wp,  0.3310013788_wp, &
   &  1.0817428904_wp,  0.7691220970_wp,  0.1860597178_wp,  0.2517779405_wp, &
   &  0.3896075279_wp,  1.1542870101_wp,  0.3543769394_wp, &
   &  2.6166714516_wp,  1.3870163411_wp,  0.1905130801_wp,  0.2686596761_wp, &
   &  0.7639709884_wp,  1.0782452556_wp,  0.5438612115_wp, &
   &  4.4269303457_wp,  2.6724041387_wp,  0.2844698146_wp,  0.2077461954_wp, &
   &  0.5715105141_wp,  1.1489703072_wp,  0.4205195661_wp, &
   &  4.5521787228_wp,  3.3492429043_wp,  0.2083020511_wp,  0.3079121794_wp, &
   &  0.4401958070_wp,  1.1162660321_wp,  0.4378290025_wp, &
   &  0.6036702266_wp,  0.8065598546_wp,  0.2518253497_wp,  0.1900906619_wp, &
   &  0.7791202796_wp,  1.0266216676_wp,  0.3689632272_wp, &
   &  1.7933472564_wp,  1.2992833058_wp,  0.3939083895_wp,  0.1987690350_wp, &
   &  0.4553941403_wp,  1.2105409882_wp,  0.4539892096_wp, &
   &  0.8995550855_wp,  2.6385957136_wp,  0.5136215069_wp,  0.0515726310_wp, &
   &  0.4807633472_wp,  1.5610437482_wp,  0.8416639521_wp, &
   &  1.6651006285_wp,  0.5462200844_wp,  2.9256166422_wp,  1.8665164360_wp, &
   &  1.3566540292_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  2.9572689542_wp,  1.6372317678_wp,  3.4991355190_wp,  0.4096685382_wp, &
   &  1.4166310383_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  2.2094311761_wp,  1.4595477455_wp,  2.8688050082_wp,  0.9630908063_wp, &
   &  0.7837391955_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.4527536356_wp,  1.0372273080_wp,  1.8808278000_wp,  2.0284210209_wp, &
   &  0.7273412844_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  3.2719949995_wp,  1.2822566886_wp,  2.3324655617_wp,  0.7288328864_wp, &
   &  1.3082748795_wp,  1.0000000000_wp,  1.0000000000_wp, &
   &  1.7881995935_wp,  0.8472278311_wp,  1.6783782922_wp,  0.8398805554_wp, &
   &  0.8846459541_wp,  1.0000000000_wp,  1.0000000000_wp], shape(keta2))

   !----------------------- +U contribution ------------------------------

   !> Base +U Hamiltonian elements
   !> see Eq. 22 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: cueffl(max_shell, max_elem) = reshape([&
   &  0.0022354720_wp, -0.3140773698_wp,  0.0264925450_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0241174031_wp, -0.0003922589_wp,  0.0132660808_wp,  0.0000000000_wp, &
   &  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0010597290_wp, -0.0048966568_wp, -0.0395002945_wp, -0.0252943259_wp, &
   &  0.0205195083_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0006248940_wp,  0.0029552824_wp, -0.2566974136_wp, -0.1857615445_wp, &
   &  0.5908323446_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0004542519_wp,  0.0000012166_wp, -0.0003411786_wp, -0.5250956140_wp, &
   &  0.0245904216_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1724139218_wp, -0.4854379663_wp, -0.1199096540_wp, -0.6378899673_wp, &
   &  0.0803598862_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0663857861_wp, -0.3332674365_wp, -0.0786642581_wp, -0.5119933588_wp, &
   &  0.2431464682_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0235017196_wp, -0.0447051814_wp, -0.1069337208_wp, -0.4619750546_wp, &
   & -0.0020689369_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0000396739_wp, -0.4372407508_wp, -0.0931165088_wp, -0.1925616058_wp, &
   &  0.1474867867_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1219632729_wp,  0.0216320304_wp, -0.0995896696_wp, -0.0161239052_wp, &
   &  0.0150675758_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0178400335_wp, -0.0051179345_wp, -0.0006891975_wp,  0.0120862599_wp, &
   &  0.2002178566_wp,  0.1814667847_wp,  0.0000000000_wp, &
   &  0.0079140176_wp,  0.0000019619_wp,  0.0003076426_wp, -0.0030577442_wp, &
   &  0.0497363993_wp,  0.0402914756_wp,  0.0000000000_wp, &
   & -0.0006294543_wp, -0.2974642255_wp, -0.0009492341_wp, -0.8627973633_wp, &
   &  0.1105011516_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0127035205_wp, -0.0016453735_wp, -0.1197517584_wp, -0.0321352870_wp, &
   &  0.0312504630_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0981642458_wp, -0.0042721476_wp, -0.1439044540_wp, -0.3872166626_wp, &
   &  0.0001741347_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0577692002_wp, -0.0127883566_wp, -0.1251025402_wp, -0.3825895844_wp, &
   &  0.2281788874_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0004385521_wp, -0.0033516468_wp, -0.1057322222_wp, -0.2753955192_wp, &
   &  0.2267181671_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0009722027_wp, -0.0456836977_wp, -0.1489854759_wp,  0.0779396556_wp, &
   &  0.1924821482_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1232734527_wp, -0.0074676127_wp, -0.0022209319_wp,  0.0096716293_wp, &
   &  0.1960232088_wp,  0.1925534108_wp,  0.0000000000_wp, &
   &  0.0042123798_wp,  0.0011649863_wp,  0.0000760797_wp, -0.0053628559_wp, &
   &  0.0156380626_wp,  0.0435914760_wp,  0.0000000000_wp, &
   & -0.0000024014_wp,  0.0005002094_wp,  0.0213794687_wp, -0.1619909073_wp, &
   &  0.0017848288_wp,  0.0006369561_wp,  0.0002758760_wp, &
   &  0.0005973521_wp,  0.0009026787_wp,  0.0075866966_wp, -0.2319648904_wp, &
   &  0.0009815453_wp,  0.0004519391_wp,  0.0007198260_wp, &
   &  0.0002373872_wp,  0.0010335991_wp,  0.0154381911_wp, -0.1677508341_wp, &
   &  0.0011182345_wp,  0.0002499857_wp,  0.0003323357_wp, &
   &  0.0080612166_wp,  0.0015184264_wp,  0.0256824027_wp, -0.2096626283_wp, &
   &  0.0018391383_wp,  0.0002019845_wp,  0.0009597973_wp, &
   &  0.0000223732_wp,  0.0026525075_wp,  0.2378359893_wp, -0.0857282931_wp, &
   &  0.0007816857_wp,  0.0000631025_wp,  0.0002174501_wp, &
   &  0.0005064406_wp,  0.0019397049_wp,  0.1405905055_wp, -0.1945160489_wp, &
   &  0.0002551197_wp,  0.0000866718_wp,  0.0010976223_wp, &
   &  0.0031893149_wp,  0.0028682296_wp,  0.1521212999_wp, -0.0266365975_wp, &
   &  0.0006732942_wp,  0.0001231917_wp,  0.0003396705_wp, &
   &  0.0004542960_wp,  0.0129793663_wp,  0.0627913169_wp, -0.0278104760_wp, &
   &  0.0002047806_wp,  0.0000604974_wp,  0.0003802923_wp, &
   &  0.0062472037_wp,  0.0034834556_wp,  0.0003511845_wp, -0.0681740017_wp, &
   &  0.0027794734_wp,  0.0024637510_wp,  0.0143957006_wp, &
   &  0.0081655035_wp,  0.0087461070_wp,  0.0025363447_wp, -0.0586186764_wp, &
   &  0.0008283908_wp,  0.0020646190_wp,  0.0070479352_wp, &
   & -0.0001488690_wp, -0.0812323587_wp,  0.0000000024_wp, -0.1863325525_wp, &
   &  0.4507429260_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0553726457_wp, -0.0234514437_wp, -0.1043060702_wp, -0.3637263355_wp, &
   &  0.2183964199_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0955822348_wp, -0.0344446371_wp, -0.1280980746_wp, -0.3882609865_wp, &
   &  0.3031415680_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0680882645_wp, -0.0258264468_wp, -0.1302435619_wp, -0.4325377444_wp, &
   &  0.2064721950_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0001115549_wp, -0.0360258867_wp, -0.1204290640_wp, -0.2263714267_wp, &
   &  0.3124265364_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0001009724_wp, -0.0013855768_wp, -0.1568778185_wp,  0.0060078210_wp, &
   & -0.0042673068_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0573048570_wp, -0.0022551369_wp,  0.0026777366_wp,  0.0280167645_wp, &
   &  0.1503156774_wp,  0.0892845885_wp,  0.0000000000_wp, &
   &  0.0035778507_wp,  0.0024229153_wp,  0.0000691736_wp, -0.0009562721_wp, &
   &  0.0133671096_wp,  0.0497980208_wp,  0.0000000000_wp, &
   & -0.0000001432_wp,  0.0006314171_wp,  0.0051473819_wp, -0.0404387349_wp, &
   &  0.0029235514_wp,  0.0002640353_wp,  0.0010526560_wp, &
   & -0.0000714649_wp,  0.0039390759_wp,  0.0020953244_wp, -0.0307681000_wp, &
   &  0.0000127094_wp,  0.0000890466_wp,  0.0030183021_wp, &
   & -0.0000447074_wp,  0.0023597023_wp,  0.0037137689_wp, -0.0343243516_wp, &
   &  0.0000463605_wp,  0.0000727605_wp,  0.0074822710_wp, &
   &  0.0005470807_wp,  0.0007435155_wp,  0.0001453632_wp, -0.1121810749_wp, &
   &  0.0002570898_wp,  0.0000164127_wp,  0.0006624186_wp, &
   &  0.0002435176_wp,  0.0006192512_wp,  0.0065092328_wp, -0.0404739356_wp, &
   &  0.0001701202_wp,  0.0000153001_wp,  0.0019550969_wp, &
   &  0.0016496200_wp,  0.0044364738_wp,  0.0100030836_wp, -0.0882806170_wp, &
   &  0.0005523874_wp,  0.0001538482_wp,  0.0003007974_wp, &
   &  0.0010693554_wp,  0.0304290523_wp,  0.0042821230_wp, -0.1087017565_wp, &
   &  0.0024047054_wp,  0.0000807961_wp,  0.0001463374_wp, &
   &  0.0009139682_wp,  0.0009645170_wp,  0.0038312153_wp, -0.0991054435_wp, &
   &  0.0000142659_wp,  0.0003972889_wp,  0.0002356786_wp, &
   &  0.0134876750_wp,  0.0033797760_wp, -0.0004338300_wp, -0.1110756926_wp, &
   &  0.0001964700_wp,  0.0007241140_wp,  0.0361017174_wp, &
   &  0.0073350735_wp,  0.0064372418_wp,  0.0016056365_wp, -0.0546847384_wp, &
   &  0.0005621783_wp,  0.0067892738_wp,  0.0146776633_wp, &
   & -0.1731133644_wp, -0.0217059370_wp, -0.0008386624_wp, -0.1183455572_wp, &
   &  0.3340514248_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0049326261_wp, -0.0093559641_wp, -0.1127655270_wp, -0.7687617529_wp, &
   &  0.3072217148_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1529597558_wp, -0.1641032911_wp, -0.1656219502_wp, -0.4322188698_wp, &
   &  0.3234187295_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.2221696570_wp, -0.3134002636_wp, -0.2257437261_wp, -0.3181231658_wp, &
   &  0.2795029663_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0004886236_wp, -0.0380577004_wp, -0.1232648159_wp, -0.1876352998_wp, &
   &  0.2999882241_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0013840086_wp, -0.3828500620_wp, -0.0906578012_wp,  0.0397152386_wp, &
   &  0.0004691989_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0107419050_wp, -0.0105184587_wp,  0.0023132542_wp,  0.0565740301_wp, &
   &  0.2445914036_wp,  0.1216898853_wp,  0.0000000000_wp, &
   &  0.0026887117_wp,  0.0009700779_wp,  0.0000986672_wp, -0.0005040440_wp, &
   &  0.0147920920_wp,  0.0216760255_wp,  0.0000000000_wp, &
   & -0.0000674062_wp,  0.0007041815_wp,  0.0015102983_wp, -0.0045844674_wp, &
   &  0.0000480548_wp,  0.0001628999_wp,  0.0044445604_wp, &
! Include 14 f-element dummy parameters initially set to zero
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.0001096703_wp,  0.0027864963_wp,  0.0015300439_wp, -0.0153080520_wp, &
   &  0.0000518048_wp,  0.0001070960_wp,  0.0145902613_wp, &
   & -0.0000978001_wp,  0.0009792851_wp,  0.0017648840_wp, -0.0115089519_wp, &
   &  0.0000231125_wp,  0.0001014295_wp,  0.0029702456_wp, &
   &  0.0007024603_wp,  0.0006967300_wp,  0.0008153787_wp, -0.0563550406_wp, &
   & -0.0000019109_wp,  0.0000000857_wp,  0.0004693446_wp, &
   &  0.0009179622_wp,  0.0042093756_wp,  0.0095584362_wp, -0.0790285522_wp, &
   &  0.0001141194_wp,  0.0001085448_wp,  0.0014726441_wp, &
   &  0.0005437839_wp,  0.0056330312_wp,  0.0083444949_wp, -0.0891650070_wp, &
   &  0.0001382869_wp,  0.0001050664_wp,  0.0014306302_wp, &
   &  0.0005933770_wp,  0.0020515982_wp,  0.0105612607_wp, -0.2226883250_wp, &
   &  0.0002055435_wp,  0.0001174981_wp,  0.0014286732_wp, &
   &  0.0006706482_wp,  0.0066521480_wp,  0.0002674554_wp, -0.0894934279_wp, &
   &  0.0005415626_wp, -0.0011573888_wp,  0.0246365483_wp, &
   &  0.0004604128_wp,  0.0002961477_wp,  0.0001107710_wp, -0.0966493777_wp, &
   &  0.0000033932_wp,  0.0000408161_wp,  0.0415665955_wp, &
   &  0.0042169158_wp,  0.0031481274_wp,  0.0011405079_wp, -0.0317609717_wp, &
   &  0.0000423061_wp,  0.0018947286_wp,  0.0891341129_wp, &
   & -0.1523419047_wp, -0.0247923308_wp, -0.0000000105_wp, -1.5697160443_wp, &
   &  0.4772821847_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.0049125832_wp, -0.0946158786_wp, -0.0123018490_wp, -1.0055019231_wp, &
   &  0.0267643868_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1779784524_wp, -0.1821349091_wp, -0.1756033489_wp, -0.7177377231_wp, &
   &  0.5458751284_wp,  0.0000000000_wp,  0.0000000000_wp, &
   &  0.1743560211_wp, -0.1223193570_wp, -0.2135273804_wp, -0.4229375933_wp, &
   &  0.3544934113_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0001327417_wp, -0.0392141753_wp, -0.1156436602_wp, -0.0299791302_wp, &
   &  0.1176766639_wp,  0.0000000000_wp,  0.0000000000_wp, &
   & -0.0201496372_wp, -0.0038121526_wp, -0.1738976984_wp,  0.0016586205_wp, &
   & -0.0005980076_wp,  0.0000000000_wp,  0.0000000000_wp], shape(cueffl))

   !> Hubbard parameter scaling for DFT+U in the diagonal case
   !> see Eq. 20 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: cud(max_elem) = [&
   &  4.1408024976_wp,  0.9632071978_wp,  5.0337790754_wp,  0.4098174275_wp,  0.6321492446_wp, &
   &  1.1129161397_wp,  1.2921841802_wp,  1.0043813929_wp,  1.0172130414_wp,  1.1780459263_wp, &
   &  0.6784257431_wp,  1.8966763783_wp,  2.0539776469_wp,  0.5526824853_wp,  0.7154926953_wp, &
   &  1.1183021259_wp,  0.9515597639_wp,  0.6089321705_wp,  2.2123686957_wp,  0.3598157976_wp, &
   &  1.6645745177_wp,  1.1530356169_wp,  1.1560237181_wp,  0.7772495334_wp,  0.5450613212_wp, &
   &  1.1515426255_wp,  3.0819400057_wp,  0.7242678813_wp,  0.9885649089_wp,  0.9697497781_wp, &
   & 14.0615958013_wp,  0.9167316951_wp,  0.7797680287_wp,  0.9304113443_wp,  1.0635155533_wp, &
   &  0.9080705218_wp,  0.9957804286_wp,  0.8070050240_wp,  2.0056459596_wp,  1.1973747896_wp, &
   &  0.7878700087_wp,  0.7035454297_wp,  0.7022623235_wp,  0.9929987312_wp,  1.2880425301_wp, &
   &  0.7137564734_wp,  0.8562015148_wp,  1.2879234257_wp,  1.5175456778_wp,  0.6267048993_wp, &
   &  0.5595165476_wp,  0.7760237810_wp,  0.9533027314_wp,  0.9114613099_wp,  0.8283753666_wp, &
   &  0.2600316100_wp,  1.3984524512_wp,  &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & 2.7097909414_wp,  1.1328336426_wp,  0.9291182466_wp, &
   &  0.9499797986_wp,  1.1694960997_wp,  1.0212383713_wp,  0.9409221598_wp,  1.0978063927_wp, &
   &  1.4550295431_wp,  1.5141693612_wp,  0.0745271245_wp,  0.6507507243_wp,  0.8358711571_wp, &
   &  0.9038563558_wp,  0.7034149219_wp]

   !> First-order charge dependence of +U Hamiltonian elements
   !> see Eq. 22 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: cu1(max_elem) = [&
   &  1.0562096820_wp,  4.2513918919_wp, -0.0529592544_wp,  0.8721574145_wp,  0.0000042235_wp, &
   &  0.0564914874_wp,  0.1831156438_wp,  0.2439417025_wp,  0.0156850844_wp, -0.5178970697_wp, &
   & -0.0454996959_wp, -0.0082847432_wp,  0.5281089113_wp, -0.0061179561_wp,  0.5097593117_wp, &
   &  0.1219967004_wp,  0.0020283995_wp, -0.9500715494_wp, -0.0455862229_wp, -0.0107318015_wp, &
   & -0.0490641750_wp, -0.3105822607_wp, -0.0364428056_wp, -0.5217684776_wp, -3.6148291723_wp, &
   & -0.4989880543_wp, -0.0149288776_wp, -5.3948319318_wp, -0.6614350498_wp, -0.1340979987_wp, &
   &  0.5261738950_wp, -0.0047835966_wp,  0.2907624627_wp,  0.1803930090_wp,  0.0013160007_wp, &
   & -0.4447881408_wp, -0.1091288188_wp, -0.0188656461_wp, -0.0826929287_wp, -1.0163584281_wp, &
   & -0.6853934346_wp, -0.4085149931_wp, -0.7086627956_wp, -0.5818689437_wp, -0.3655194756_wp, &
   & -0.6918075919_wp, -0.4677287155_wp, -0.4350180507_wp,  0.1012902432_wp, -0.0006361940_wp, &
   &  0.2909432503_wp,  0.0030234671_wp,  0.0056991136_wp, -0.3211722022_wp, -0.0645497300_wp, &
   & -0.0081843947_wp, -0.6960291229_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.4264869764_wp, -0.4321439426_wp, -1.2689276017_wp, &
   & -0.7972574276_wp, -0.3615372974_wp, -0.1642177576_wp, -1.0071974105_wp, -0.3430066881_wp, &
   & -0.1470750192_wp,  0.0072039642_wp, -0.0659094369_wp,  0.1678933538_wp,  0.0015669005_wp, &
   &  0.0040973880_wp, -0.4824895186_wp]

   !> Second-order charge dependence of +U Hamiltonian elements
   !> see Eq. 22 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: cu2(max_elem) = [&
   &  1.3525995992_wp, -1.4516871715_wp, -0.0000572628_wp,  0.4032354601_wp, -0.8336564414_wp, &
   &  0.0243755779_wp,  0.0689852936_wp,  0.0204538409_wp,  0.0687599339_wp, -0.0394658916_wp, &
   & -0.0018148200_wp, -0.0071978001_wp,  0.0001015111_wp, -0.0055569698_wp,  0.0086391927_wp, &
   &  0.0668651210_wp,  0.0272529641_wp, -0.0004965216_wp, -0.0014578692_wp, -0.0012161128_wp, &
   & -0.0002074882_wp, -0.0001123772_wp, -0.0001531210_wp, -0.0001594202_wp,  0.0002974601_wp, &
   & -0.0011942472_wp, -0.0006284381_wp, -0.0313211899_wp, -0.0000542139_wp, -0.0002675519_wp, &
   &  0.6015750312_wp, -0.0054223243_wp,  0.0056859169_wp,  0.0564772026_wp,  0.0000101064_wp, &
   & -0.0003786663_wp, -0.0008412240_wp, -0.0069494414_wp, -0.0006534067_wp, -0.0051021551_wp, &
   & -0.0023209872_wp, -0.0016927794_wp, -0.0009194111_wp, -0.0001951513_wp, -0.0015952347_wp, &
   & -0.0000687023_wp, -0.0010489656_wp, -0.0000797378_wp,  0.0007172705_wp, -0.0073589222_wp, &
   &  0.0359551718_wp,  0.1029920152_wp,  0.0015932618_wp, -0.0002693104_wp, -0.0029468278_wp, &
   & -0.0039873904_wp, -0.0035546161_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.0015774239_wp, -0.0041795453_wp, -0.0001403746_wp, &
   & -0.0015572206_wp, -0.0059294513_wp, -0.0005959109_wp, -0.0015384083_wp, -0.0011762264_wp, &
   & -0.0031683283_wp,  0.0023191960_wp, -0.0061046325_wp,  0.0452005557_wp,  0.1036140559_wp, &
   &  0.0000835835_wp,  0.0000423466_wp]

   !> Empirical atomic radius for +U damping function
   !> see Eq. 24 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: ar(max_elem) = [&
   &  3.3392528850_wp,  1.9259592984_wp,  7.3381422285_wp,  4.3135659457_wp,  7.9691261380_wp, &
   &  4.5207384982_wp,  2.6850382750_wp,  3.6785810016_wp,  2.5133987301_wp,  0.9873346860_wp, &
   & 15.6283943037_wp,  9.5944382403_wp,  5.1906593322_wp,  9.9161767845_wp,  8.6374217428_wp, &
   &  3.8820455596_wp,  5.2017674757_wp,  2.3700112620_wp,  8.4738904338_wp,  7.7449345260_wp, &
   & 42.1449944745_wp, 28.1686011406_wp, 61.6561643353_wp, 86.3416352738_wp, 75.7274848192_wp, &
   & 61.7766482605_wp, 57.8798540385_wp, 19.8451484534_wp, 17.9113876136_wp,  7.0726997652_wp, &
   &  5.5677947085_wp,  6.3764221739_wp,  9.0557558953_wp,  3.4941349202_wp,  6.1640731231_wp, &
   & 12.1836147301_wp,  6.0512942169_wp,  8.8584830606_wp,  5.9316464472_wp, 11.5888263536_wp, &
   & 31.8083067896_wp, 50.2208510079_wp, 33.0801765841_wp, 25.6532049528_wp, 61.3010412688_wp, &
   &  5.7495276519_wp, 12.2729775054_wp,  3.7298227520_wp,  2.6465941371_wp, 14.4460862230_wp, &
   & 15.9077245664_wp,  5.2833647048_wp,  4.8966627601_wp, 36.3955533747_wp,  5.2367728795_wp, &
   &  6.5273540069_wp, 48.0409613981_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & 21.1794965214_wp, 27.9695843046_wp, 94.6563340482_wp, &
   & 21.4021775847_wp, 35.9747595340_wp, 18.6664940573_wp,  3.9244695587_wp,  7.6812651217_wp, &
   &  4.3085818701_wp,  2.5876026826_wp, 37.9330867506_wp, 16.0518902390_wp,  6.2698130157_wp, &
   &  5.0363953399_wp,  7.4636790547_wp]

   !> Scaling of CN correction for +U damping function
   !> see Eq. 24 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: arcn(max_elem) = [&
   & -0.4639203632_wp, -3.4485126405_wp,  0.5112055932_wp,  0.1277231370_wp, -2.5836929728_wp, &
   & -0.1117214812_wp, -1.8967433399_wp, -0.6057453510_wp, -0.0785691898_wp,  0.0001592952_wp, &
   &  0.5353420432_wp, -0.0506697501_wp, -0.0000000031_wp, -2.9876209160_wp, -0.8531838111_wp, &
   & -0.3246119129_wp, -0.7897163220_wp,  0.0022398424_wp,  0.0770367975_wp, -0.0001855429_wp, &
   & -0.0000178246_wp,  0.0994615781_wp, -0.5628876249_wp,  0.0414251666_wp,  0.0022655293_wp, &
   & -0.0492708583_wp,  0.0001670017_wp,  0.0154850738_wp,  0.0000244119_wp,  0.0000244392_wp, &
   &  0.0029085874_wp, -0.1742838604_wp, -0.0796864919_wp, -0.0010465841_wp, -2.4017731773_wp, &
   &  0.7840787360_wp,  0.7004096969_wp, -0.0059617169_wp, -0.8005785181_wp, -0.7770360929_wp, &
   & -0.5286568971_wp, -0.0971743419_wp, -0.4598496944_wp, -0.3921881790_wp, -0.0206325270_wp, &
   & -0.0083881414_wp, -0.0056511104_wp, -0.0116402628_wp, -0.0821478451_wp, -0.0076578281_wp, &
   & -0.0204643672_wp, -0.0017757924_wp, -0.0305601568_wp,  0.0000609901_wp,  0.2118632988_wp, &
   & -0.0036854964_wp, -0.5329920266_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -0.8633899208_wp, -0.2120500541_wp, -0.0387423523_wp, &
   & -0.2050355038_wp, -0.4904078341_wp, -0.1978617712_wp, -0.0220961123_wp, -0.0056804623_wp, &
   & -0.0213573127_wp,  0.0000000014_wp, -0.0001092631_wp, -0.0013482778_wp, -0.0153551917_wp, &
   & -0.0347689705_wp,  0.1053436128_wp]

   !> Average coordination number for +U damping function
   !> see Eq. 24 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: avcn(max_elem) = [ &
   & 0.8571_wp, 0.5576_wp, 2.5636_wp, 2.2666_wp, 2.7596_wp, &
   & 3.1576_wp, 2.6221_wp, 1.5218_wp, 1.1580_wp, 0.6325_wp, &
   & 2.4115_wp, 2.8134_wp, 3.4564_wp, 3.1467_wp, 3.4848_wp, &
   & 2.2996_wp, 1.6193_wp, 0.9971_wp, 2.7819_wp, 4.0839_wp, &
   & 4.6882_wp, 4.1372_wp, 5.2103_wp, 4.8825_wp, 5.1648_wp, &
   & 5.4032_wp, 4.6666_wp, 3.3283_wp, 2.3970_wp, 2.4055_wp, &
   & 3.3721_wp, 3.1015_wp, 3.1793_wp, 2.2906_wp, 1.7517_wp, &
   & 1.2780_wp, 3.0927_wp, 2.8011_wp, 5.3701_wp, 4.4574_wp, &
   & 5.2953_wp, 5.7108_wp, 5.1423_wp, 5.5080_wp, 5.1645_wp, &
   & 3.3691_wp, 2.5553_wp, 2.5331_wp, 3.4365_wp, 3.3329_wp, &
   & 3.1065_wp, 2.3380_wp, 1.9027_wp, 1.4142_wp, 3.2766_wp, &
   & 2.5869_wp, 5.7731_wp, &
   ! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   ! ----------------------------------------------------------
   & 4.3281_wp, 5.1258_wp, 5.4683_wp, 5.1728_wp, &
   & 5.5049_wp, 4.9224_wp, 3.5058_wp, 2.4969_wp, 2.5980_wp, &
   & 3.2478_wp, 3.0994_wp, 3.1497_wp, 2.3199_wp, 1.6944_wp, &
   & 1.4813_wp ]


   !------------- RESPONSE APPROXIMATION -----------------------------------

   !> Wolfsberg parameter used in one-scf for response approximation
   real(wp), parameter :: kares(max_elem) = [&
   &  2.2706175180_wp,  2.1471484161_wp,  2.1770808307_wp,  1.7239136638_wp,  1.9637516953_wp, &
   &  2.4591302400_wp,  2.2762208854_wp,  2.1511713154_wp,  2.1202424807_wp,  3.2382477560_wp, &
   &  1.3450415358_wp,  1.2424371573_wp,  2.1730307885_wp,  2.1591046594_wp,  1.7866494668_wp, &
   &  1.8145439092_wp,  1.9139721755_wp,  2.3903290968_wp,  1.3524020800_wp,  1.1158002422_wp, &
   &  1.5329437010_wp,  2.7243224212_wp,  2.4482875013_wp,  2.5259712425_wp,  2.6949180987_wp, &
   &  2.7219715149_wp,  2.8034011628_wp,  2.4897090880_wp,  1.3836761133_wp,  1.0908059423_wp, &
   &  1.8191422070_wp,  2.0644144766_wp,  1.7978503648_wp,  1.9060704221_wp,  1.7520468379_wp, &
   &  1.7437772219_wp,  1.1857874518_wp,  1.2691750352_wp,  1.8880084591_wp,  2.2911017244_wp, &
   &  2.0770604008_wp,  2.8999021649_wp,  2.6852607145_wp,  2.6964566810_wp,  3.5889533132_wp, &
   &  2.8937325935_wp,  1.3846602233_wp,  1.0120214682_wp,  2.8683213684_wp,  2.1093756588_wp, &
   &  1.8539628369_wp,  2.2517987661_wp,  1.9283461706_wp,  1.5653991371_wp,  1.0444749711_wp, &
   &  1.2258351499_wp,  2.0071713899_wp, &
   ! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   ! ----------------------------------------------------------
   &  2.1417149499_wp,  2.2330253487_wp,  2.4494994673_wp, &
   &  2.7186315705_wp,  2.2191516142_wp,  3.2350098880_wp,  3.5012478083_wp,  2.7369955778_wp, &
   &  1.1205992535_wp,  2.0573095629_wp,  2.1930284001_wp,  1.7781733406_wp,  1.8635169960_wp, &
   &  2.1183134150_wp,  1.5226795093_wp]

   !> Scaling of +U Hamiltonian contributions in the response approximation
   !> see Eq. 27 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: kueffres(max_elem) = [&
   &  1.0863599879_wp, 16.9812685688_wp,  8.6420038476_wp,  0.5484484631_wp,  0.2577436503_wp, &
   &  0.3210333975_wp,  0.5562396695_wp,  0.7742792877_wp,  1.3002008951_wp,  2.3568951691_wp, &
   &  2.8372990464_wp, -3.6933029996_wp,  1.7678107153_wp,  0.0000363495_wp, -0.0000157508_wp, &
   &  0.4400781915_wp,  0.9049487228_wp, -0.2895605038_wp, -0.4310711823_wp,-20.8795500364_wp, &
   & -0.0072266540_wp,  0.0124089334_wp,  1.7078138466_wp,  2.0879345709_wp,  0.6946791125_wp, &
   &  3.2587396726_wp, -1.3306854739_wp,  1.8424262449_wp, 10.5946412950_wp,  9.7311656284_wp, &
   & -5.0162050309_wp,  0.1202810640_wp,  0.0307732364_wp,  0.4843232728_wp,  0.6720438652_wp, &
   & -0.2315060175_wp, -5.3770836652_wp,-10.0115980952_wp, -0.0476733347_wp, -0.7042412501_wp, &
   & -0.0275494919_wp,  0.3586611657_wp,  1.8448256211_wp,  1.9790773962_wp,  0.0963647731_wp, &
   &  0.9239095116_wp,  6.1466747919_wp,  8.3907255338_wp,  0.7006214982_wp,  0.2531610204_wp, &
   &  0.1594402342_wp,  0.5591352707_wp,  0.7950425432_wp, -0.2839227549_wp, -3.6263466270_wp, &
   &-12.9287793419_wp,  2.0101709211_wp, &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   & -1.3624458451_wp, -2.7380964989_wp,  0.5653100295_wp, &
   &  0.8094252656_wp,  0.5944864198_wp,  0.5704853319_wp,  0.5967148897_wp,  2.5726383089_wp, &
   & 12.3862934557_wp,  0.2993283149_wp,  0.2370205943_wp,  0.2185483719_wp,  0.8206054951_wp, &
   &  1.6230558612_wp,  0.6538569265_wp]

   !> Scaling of IES+TO+U elements in the response case
   !> see Eq. 26 in J. Chem. Phys. 158, 124111 (2023)
   real(wp), parameter :: cvesres(max_elem) = [&
   &  0.5699276681_wp, -0.9303039725_wp,  1.1589279019_wp,  0.7451568313_wp,  0.5381649334_wp, &
   &  0.7181178607_wp,  0.6920744852_wp,  0.6742096930_wp,  0.3937865767_wp,  1.5783943459_wp, &
   & 16.7381529578_wp,  1.2041305914_wp,  0.8036230577_wp,  0.7726690167_wp,  0.5671786712_wp, &
   &  0.6548674774_wp,  0.4015788827_wp,  1.2131355661_wp,  6.6968314360_wp,  3.1638225974_wp, &
   &  0.2884619435_wp,  0.6293858213_wp,  0.5348087976_wp,  0.3185501820_wp,  0.8324452351_wp, &
   &  0.9104239212_wp,  0.7697797310_wp,  0.7474487093_wp,  1.3527268874_wp,  0.6439039609_wp, &
   &  1.3844331046_wp,  0.7445930239_wp,  0.7509055007_wp,  0.6147832249_wp,  0.5015363783_wp, &
   &  1.3191503314_wp,  5.6466811094_wp,  2.9169750832_wp,  0.2962968637_wp,  0.6673973765_wp, &
   &  0.3863800136_wp,  0.6242902752_wp,  0.8987235789_wp,  1.1432407530_wp,  0.7577479900_wp, &
   &  0.4622168381_wp,  2.3139272923_wp,  0.3233690869_wp, -0.3309406473_wp,  0.8032604988_wp, &
   &  0.8781942413_wp,  0.7736165549_wp,  0.4021759699_wp,  1.1837916338_wp, 10.0873240838_wp, &
   &  1.8428162439_wp,  0.3768712849_wp,  &
! Include 14 f-element dummy parameters initially set to zero
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
   &  0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
! ----------------------------------------------------------
   &  0.5411751361_wp,  0.5366078353_wp,  0.8028260336_wp, &
   &  0.7828825483_wp,  0.9337863113_wp,  0.6578192696_wp,  0.5305709389_wp,  0.3228077942_wp, &
   &  0.7356040187_wp,  0.6242240249_wp,  0.8636931960_wp,  0.9155901840_wp,  0.8589415679_wp, &
   &  0.7012900230_wp,  0.4805803329_wp]

contains

   subroutine initData(self, num)

      !> Data instance
      type(TPTBData), intent(out) :: self

      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)

      self%name = 'PTB'
      self%doi = '10.1063/5.0137838'
      self%nshell = nshell

      call initPTB(self%corepotential)
      call initPTB(self%eeq, num)
      call initPTB(self%hamiltonian, num)
      call initPTB(self%pauli, num)
      call initPTB(self%coulomb, num)
      call initPTB(self%plusu, num)
      call initPTB(self%response, num)

   end subroutine initData

   subroutine initCorepotential(self)

      !> Data instance
      type(TCorePotentialData), intent(out) :: self

      call set_cbas()
      call init(self, max_core_prim, max_core_shell, &
      & cbas_sl_exp, cbas_nshell, cbas_pqn, cbas_angshell, cbas_hflev, &
      & kecpepsilon)

   end subroutine initCorepotential

   subroutine initEEQ(self, num)

      !> Data instance
      type(TEEQData), intent(out) :: self

      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)

      call init(self, num, alpeeq, chieeq, cnfeeq, gameeq)

   end subroutine initEEQ

   subroutine initPauliXC(self, num)

      !> Data instance
      type(TPauliXCData), intent(out) :: self
      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)

      call init(self, num, nshell, kxc1, kxc2l, klalphaxc)

   end subroutine initPauliXC

   subroutine initHamiltonian(self, num)

      !> Data instance
      type(THamiltonianData), intent(out) :: self
      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)

      call init(self, num, nshell, ang_shell, &
      & hla, klh, kcnstar, kshift, kla, kr, kocod, &
      & ksla, kalphah0l, kits0)

      allocate (self%refocc(max_shell, max_elem))
      call setPTBReferenceOcc(self,num)

   end subroutine initHamiltonian

   subroutine initCoulomb(self, num)

      !> Data instance
      type(TCoulombData), intent(out) :: self
      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)

      call init(self, num, nshell, keta1, keta2, kto)
      self%kQHubbard = 0.25_wp
      self%kOK1 = 1.6_wp
      self%kOK2 = 1.0_wp

   end subroutine initCoulomb

   subroutine initPlusU(self, num)

      !> Data instance
      type(TPlusU), intent(out) :: self
      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)

      call init(self, num, nshell, &
      & avcn, cud, cu1, cu2, cueffl, &
      & ar, arcn)

   end subroutine initPlusU

   subroutine initResponse(self, num)

      !> Data instance
      type(TResponse), intent(out) :: self
      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)

      call init(self, num, nshell, &
      & kares, cvesres, kueffres)
      self%kOK_onescf = 0.75_wp

   end subroutine initResponse

   subroutine setPTBReferenceOcc(self,num)

      !> Data instance
      type(THamiltonianData), intent(inout) :: self
      !> Atomic numbers for unique elements
      integer, intent(in) :: num(:)
      real(wp), allocatable :: refocc(:)

      integer :: isp, izp

      self%refocc(:, :) = 0.0_wp

      do isp = 1, size(num)
         izp = num(isp)
         call shellocc_ref(izp, nshell(izp), refocc)
         self%refocc(1:nshell(izp), isp) = refocc(1:nshell(izp))
         deallocate(refocc)
      end do

   end subroutine setPTBReferenceOcc

   subroutine set_cbas()

      !> Slater exponents
      cbas_sl_exp(1,1) =   1.0000_wp
      cbas_sl_exp(1,2) =   1.6250_wp
      cbas_sl_exp(1,3) =   2.6906_wp
      cbas_sl_exp(1,4) =   3.6848_wp
      cbas_sl_exp(1,5) =   4.6795_wp
      cbas_sl_exp(1,6) =   5.6727_wp
      cbas_sl_exp(1,7) =   6.6651_wp
      cbas_sl_exp(1,8) =   7.6579_wp
      cbas_sl_exp(1,9) =   8.6501_wp
      cbas_sl_exp(1,10)=   9.6421_wp
      cbas_sl_exp(1,11)=  10.6259_wp
      cbas_sl_exp(2,11)=   3.2857_wp
      cbas_sl_exp(3,11)=   3.4409_wp
      cbas_sl_exp(1,12)=  11.6089_wp
      cbas_sl_exp(2,12)=   3.6960_wp
      cbas_sl_exp(3,12)=   3.9129_wp
      cbas_sl_exp(1,13)=  12.5910_wp
      cbas_sl_exp(2,13)=   4.1068_wp
      cbas_sl_exp(3,13)=   4.4817_wp
      cbas_sl_exp(1,14)=  13.5745_wp
      cbas_sl_exp(2,14)=   5.5100_wp
      cbas_sl_exp(3,14)=   4.9725_wp
      cbas_sl_exp(1,15)=  14.5578_wp
      cbas_sl_exp(2,15)=   4.9125_wp
      cbas_sl_exp(3,15)=   5.4806_wp
      cbas_sl_exp(1,16)=  15.5409_wp
      cbas_sl_exp(2,16)=   5.3144_wp
      cbas_sl_exp(3,16)=   5.9855_wp
      cbas_sl_exp(1,17)=  16.5239_wp
      cbas_sl_exp(2,17)=   5.7152_wp
      cbas_sl_exp(3,17)=   6.4966_wp
      cbas_sl_exp(1,18)=  17.5075_wp
      cbas_sl_exp(2,18)=   6.1152_wp
      cbas_sl_exp(3,18)=   7.0041_wp
      cbas_sl_exp(1,19)=  18.4895_wp
      cbas_sl_exp(2,19)=   6.5031_wp
      cbas_sl_exp(3,19)=   7.5136_wp
      cbas_sl_exp(4,19)=   2.8933_wp
      cbas_sl_exp(5,19)=   2.5752_wp
      cbas_sl_exp(1,20)=  19.4730_wp
      cbas_sl_exp(2,20)=   6.8882_wp
      cbas_sl_exp(3,20)=   8.0207_wp
      cbas_sl_exp(4,20)=   3.2005_wp
      cbas_sl_exp(5,20)=   2.8861_wp
      cbas_sl_exp(1,21)=  20.4566_wp
      cbas_sl_exp(2,21)=   7.2868_wp
      cbas_sl_exp(3,21)=   8.5273_wp
      cbas_sl_exp(1,22)=  21.4409_wp
      cbas_sl_exp(2,22)=   7.6883_wp
      cbas_sl_exp(3,22)=   9.0324_wp
      cbas_sl_exp(1,23)=  22.4256_wp
      cbas_sl_exp(2,23)=   8.0907_wp
      cbas_sl_exp(3,23)=   9.5364_wp
      cbas_sl_exp(1,24)=  23.4138_wp
      cbas_sl_exp(2,24)=   8.4919_wp
      cbas_sl_exp(3,24)=  10.0376_wp
      cbas_sl_exp(1,25)=  24.3957_wp
      cbas_sl_exp(2,25)=   8.8969_wp
      cbas_sl_exp(3,25)=  10.5420_wp
      cbas_sl_exp(1,26)=  25.3810_wp
      cbas_sl_exp(2,26)=   9.2995_wp
      cbas_sl_exp(3,26)=  11.0444_wp
      cbas_sl_exp(1,27)=  26.3668_wp
      cbas_sl_exp(2,27)=   9.7025_wp
      cbas_sl_exp(3,27)=  11.5462_wp
      cbas_sl_exp(1,28)=  27.3526_wp
      cbas_sl_exp(2,28)=  10.1063_wp
      cbas_sl_exp(3,28)=  12.0476_wp
      cbas_sl_exp(1,29)=  28.3386_wp
      cbas_sl_exp(2,29)=  10.5099_wp
      cbas_sl_exp(3,29)=  12.5485_wp
      cbas_sl_exp(1,30)=  29.3245_wp
      cbas_sl_exp(2,30)=  10.9140_wp
      cbas_sl_exp(3,30)=  13.0490_wp
      cbas_sl_exp(1,31)=  30.3094_wp
      cbas_sl_exp(2,31)=  11.2995_wp
      cbas_sl_exp(3,31)=  13.5454_wp
      cbas_sl_exp(4,31)=   5.6654_wp
      cbas_sl_exp(5,31)=   5.4012_wp
      cbas_sl_exp(6,31)=   5.0311_wp
      cbas_sl_exp(1,32)=  31.2937_wp
      cbas_sl_exp(2,32)=  11.6824_wp
      cbas_sl_exp(3,32)=  14.0411_wp
      cbas_sl_exp(4,32)=   5.9299_wp
      cbas_sl_exp(5,32)=   5.6712_wp
      cbas_sl_exp(6,32)=   5.4171_wp
      cbas_sl_exp(1,33)=  32.2783_wp
      cbas_sl_exp(2,33)=  12.0635_wp
      cbas_sl_exp(3,33)=  14.5368_wp
      cbas_sl_exp(4,33)=   6.1985_wp
      cbas_sl_exp(5,33)=   5.9499_wp
      cbas_sl_exp(6,33)=   5.7928_wp
      cbas_sl_exp(1,34)=  33.2622_wp
      cbas_sl_exp(2,34)=  12.4442_wp
      cbas_sl_exp(3,34)=  15.0326_wp
      cbas_sl_exp(4,34)=   6.4678_wp
      cbas_sl_exp(5,34)=   6.2350_wp
      cbas_sl_exp(6,34)=   6.1590_wp
      cbas_sl_exp(1,35)=  34.2471_wp
      cbas_sl_exp(2,35)=  12.8217_wp
      cbas_sl_exp(3,35)=  15.5282_wp
      cbas_sl_exp(4,35)=   6.7395_wp
      cbas_sl_exp(5,35)=   6.5236_wp
      cbas_sl_exp(6,35)=   6.5197_wp
      cbas_sl_exp(1,36)=  35.2316_wp
      cbas_sl_exp(2,36)=  13.1990_wp
      cbas_sl_exp(3,36)=  16.0235_wp
      cbas_sl_exp(4,36)=   7.0109_wp
      cbas_sl_exp(5,36)=   6.8114_wp
      cbas_sl_exp(6,36)=   6.8753_wp
      cbas_sl_exp(1,37)=  36.2078_wp
      cbas_sl_exp(2,37)=  13.5784_wp
      cbas_sl_exp(3,37)=  16.5194_wp
      cbas_sl_exp(4,37)=   7.2809_wp
      cbas_sl_exp(5,37)=   7.1011_wp
      cbas_sl_exp(6,37)=   7.2264_wp
      cbas_sl_exp(7,37)=   3.0970_wp
      cbas_sl_exp(8,37)=   2.7202_wp
      cbas_sl_exp(1,38)=  37.1911_wp
      cbas_sl_exp(2,38)=  13.9509_wp
      cbas_sl_exp(3,38)=  17.0152_wp
      cbas_sl_exp(4,38)=   7.5546_wp
      cbas_sl_exp(5,38)=   7.3892_wp
      cbas_sl_exp(6,38)=   7.5754_wp
      cbas_sl_exp(7,38)=   3.3611_wp
      cbas_sl_exp(8,38)=   2.9830_wp
      cbas_sl_exp(1,39)=  38.1756_wp
      cbas_sl_exp(2,39)=  14.3111_wp
      cbas_sl_exp(3,39)=  17.5016_wp
      cbas_sl_exp(4,39)=   7.8505_wp
      cbas_sl_exp(5,39)=   7.6975_wp
      cbas_sl_exp(6,39)=   8.4657_wp
      cbas_sl_exp(1,40)=  39.1590_wp
      cbas_sl_exp(2,40)=  14.6869_wp
      cbas_sl_exp(3,40)=  17.9964_wp
      cbas_sl_exp(4,40)=   8.1205_wp
      cbas_sl_exp(5,40)=   7.9485_wp
      cbas_sl_exp(6,40)=   8.5223_wp
      cbas_sl_exp(1,41)=  40.1423_wp
      cbas_sl_exp(2,41)=  15.0626_wp
      cbas_sl_exp(3,41)=  18.4911_wp
      cbas_sl_exp(4,41)=   8.3905_wp
      cbas_sl_exp(5,41)=   8.2184_wp
      cbas_sl_exp(6,41)=   8.7847_wp
      cbas_sl_exp(1,42)=  41.1256_wp
      cbas_sl_exp(2,42)=  15.4384_wp
      cbas_sl_exp(3,42)=  18.9859_wp
      cbas_sl_exp(4,42)=   8.6605_wp
      cbas_sl_exp(5,42)=   8.4912_wp
      cbas_sl_exp(6,42)=   9.0761_wp
      cbas_sl_exp(1,43)=  42.1090_wp
      cbas_sl_exp(2,43)=  15.8141_wp
      cbas_sl_exp(3,43)=  19.4704_wp
      cbas_sl_exp(4,43)=   8.9304_wp
      cbas_sl_exp(5,43)=   8.7947_wp
      cbas_sl_exp(6,43)=   9.4510_wp
      cbas_sl_exp(1,44)=  43.0923_wp
      cbas_sl_exp(2,44)=  16.1899_wp
      cbas_sl_exp(3,44)=  19.9754_wp
      cbas_sl_exp(4,44)=   9.2004_wp
      cbas_sl_exp(5,44)=   9.0844_wp
      cbas_sl_exp(6,44)=   9.7981_wp
      cbas_sl_exp(1,45)=  44.0756_wp
      cbas_sl_exp(2,45)=  16.5656_wp
      cbas_sl_exp(3,45)=  20.4702_wp
      cbas_sl_exp(4,45)=   9.4704_wp
      cbas_sl_exp(5,45)=   9.3724_wp
      cbas_sl_exp(6,45)=  10.1478_wp
      cbas_sl_exp(1,46)=  45.0589_wp
      cbas_sl_exp(2,46)=  16.9414_wp
      cbas_sl_exp(3,46)=  20.9650_wp
      cbas_sl_exp(4,46)=   9.7404_wp
      cbas_sl_exp(5,46)=   9.6616_wp
      cbas_sl_exp(6,46)=  10.4989_wp
      cbas_sl_exp(1,47)=  46.0423_wp
      cbas_sl_exp(2,47)=  17.3171_wp
      cbas_sl_exp(3,47)=  21.4597_wp
      cbas_sl_exp(4,47)=  10.0104_wp
      cbas_sl_exp(5,47)=   9.9476_wp
      cbas_sl_exp(6,47)=  10.8503_wp
      cbas_sl_exp(1,48)=  47.0256_wp
      cbas_sl_exp(2,48)=  17.6929_wp
      cbas_sl_exp(3,48)=  21.9545_wp
      cbas_sl_exp(4,48)=  10.2804_wp
      cbas_sl_exp(5,48)=  10.2305_wp
      cbas_sl_exp(6,48)=  11.2023_wp
      cbas_sl_exp(1,49)=  48.0097_wp
      cbas_sl_exp(2,49)=  18.0618_wp
      cbas_sl_exp(3,49)=  22.4490_wp
      cbas_sl_exp(4,49)=  10.5436_wp
      cbas_sl_exp(5,49)=  10.5069_wp
      cbas_sl_exp(6,49)=  11.5594_wp
      cbas_sl_exp(7,49)=   5.4403_wp
      cbas_sl_exp(8,49)=   5.0922_wp
      cbas_sl_exp(9,49)=   4.2354_wp
      cbas_sl_exp(1,50)=  48.9920_wp
      cbas_sl_exp(2,50)=  18.4297_wp
      cbas_sl_exp(3,50)=  22.9427_wp
      cbas_sl_exp(4,50)=  10.8066_wp
      cbas_sl_exp(5,50)=  10.7844_wp
      cbas_sl_exp(6,50)=  11.9139_wp
      cbas_sl_exp(7,50)=   5.6645_wp
      cbas_sl_exp(8,50)=   5.3163_wp
      cbas_sl_exp(9,50)=   4.4925_wp
      cbas_sl_exp(1,51)=  49.9744_wp
      cbas_sl_exp(2,51)=  18.7977_wp
      cbas_sl_exp(3,51)=  23.4363_wp
      cbas_sl_exp(4,51)=  11.0697_wp
      cbas_sl_exp(5,51)=  11.0613_wp
      cbas_sl_exp(6,51)=  12.2666_wp
      cbas_sl_exp(7,51)=   5.8859_wp
      cbas_sl_exp(8,51)=   5.5453_wp
      cbas_sl_exp(9,51)=   4.7436_wp
      cbas_sl_exp(1,52)=  50.9568_wp
      cbas_sl_exp(2,52)=  19.1656_wp
      cbas_sl_exp(3,52)=  23.9300_wp
      cbas_sl_exp(4,52)=  11.3327_wp
      cbas_sl_exp(5,52)=  11.3363_wp
      cbas_sl_exp(6,52)=  12.6131_wp
      cbas_sl_exp(7,52)=   6.1021_wp
      cbas_sl_exp(8,52)=   5.7805_wp
      cbas_sl_exp(9,52)=   4.9900_wp
      cbas_sl_exp(1,53)=  51.9391_wp
      cbas_sl_exp(2,53)=  19.5335_wp
      cbas_sl_exp(3,53)=  24.4237_wp
      cbas_sl_exp(4,53)=  11.5958_wp
      cbas_sl_exp(5,53)=  11.6138_wp
      cbas_sl_exp(6,53)=  12.9669_wp
      cbas_sl_exp(7,53)=   6.3243_wp
      cbas_sl_exp(8,53)=   6.0074_wp
      cbas_sl_exp(9,53)=   5.2335_wp
      cbas_sl_exp(1,54)=  52.9215_wp
      cbas_sl_exp(2,54)=  19.9015_wp
      cbas_sl_exp(3,54)=  24.9173_wp
      cbas_sl_exp(4,54)=  11.8588_wp
      cbas_sl_exp(5,54)=  11.8892_wp
      cbas_sl_exp(6,54)=  13.3156_wp
      cbas_sl_exp(7,54)=   6.5432_wp
      cbas_sl_exp(8,54)=   6.2393_wp
      cbas_sl_exp(9,54)=   5.4733_wp
      cbas_sl_exp(1,55)=  53.9043_wp
      cbas_sl_exp(2,55)=  20.2558_wp
      cbas_sl_exp(3,55)=  25.4098_wp
      cbas_sl_exp(4,55)=  12.1258_wp
      cbas_sl_exp(5,55)=  12.1926_wp
      cbas_sl_exp(6,55)=  13.6602_wp
      cbas_sl_exp(7,55)=   6.7606_wp
      cbas_sl_exp(8,55)=   6.4644_wp
      cbas_sl_exp(9,55)=   5.7096_wp
      cbas_sl_exp(10,55)=  3.0889_wp
      cbas_sl_exp(11,55)=  2.7302_wp
      cbas_sl_exp(1, 56)= 54.8861_wp
      cbas_sl_exp(2, 56)= 20.6234_wp
      cbas_sl_exp(3, 56)= 25.9048_wp
      cbas_sl_exp(4, 56)= 12.3852_wp
      cbas_sl_exp(5, 56)= 12.4388_wp
      cbas_sl_exp(6, 56)= 14.0081_wp
      cbas_sl_exp(7, 56)=  6.9800_wp
      cbas_sl_exp(8, 56)=  6.7008_wp
      cbas_sl_exp(9, 56)=  5.9460_wp
      cbas_sl_exp(10,56)=  3.3239_wp
      cbas_sl_exp(11,56)=  2.9601_wp
      cbas_sl_exp(1, 57)= 55.8683_wp
      cbas_sl_exp(2, 57)= 20.9767_wp
      cbas_sl_exp(3, 57)= 26.3978_wp
      cbas_sl_exp(4, 57)= 12.6477_wp
      cbas_sl_exp(5, 57)= 12.7132_wp
      cbas_sl_exp(6, 57)= 14.3534_wp
      cbas_sl_exp(7, 57)=  7.1991_wp
      cbas_sl_exp(8, 57)=  6.9266_wp
      cbas_sl_exp(9, 57)=  6.1813_wp
      cbas_sl_exp(1, 72)= 70.6016_wp
      cbas_sl_exp(2, 72)= 26.5949_wp
      cbas_sl_exp(3, 72)= 33.7994_wp
      cbas_sl_exp(4, 72)= 16.7705_wp
      cbas_sl_exp(5, 72)= 16.9944_wp
      cbas_sl_exp(6, 72)= 19.4766_wp
      cbas_sl_exp(7, 72)=  9.7443_wp
      cbas_sl_exp(8, 72)=  9.4824_wp
      cbas_sl_exp(9, 72)=  8.8810_wp
      cbas_sl_exp(1, 73)= 71.5837_wp
      cbas_sl_exp(2, 73)= 26.9649_wp
      cbas_sl_exp(3, 73)= 34.2932_wp
      cbas_sl_exp(4, 73)= 17.0305_wp
      cbas_sl_exp(5, 73)= 17.2668_wp
      cbas_sl_exp(6, 73)= 19.8137_wp
      cbas_sl_exp(7, 73)=  9.9397_wp
      cbas_sl_exp(8, 73)=  9.6837_wp
      cbas_sl_exp(9, 73)=  9.0810_wp
      cbas_sl_exp(1, 74)= 72.5657_wp
      cbas_sl_exp(2, 74)= 27.3349_wp
      cbas_sl_exp(3, 74)= 34.7871_wp
      cbas_sl_exp(4, 74)= 17.2900_wp
      cbas_sl_exp(5, 74)= 17.5392_wp
      cbas_sl_exp(6, 74)= 20.1508_wp
      cbas_sl_exp(7, 74)= 10.1397_wp
      cbas_sl_exp(8, 74)=  9.8871_wp
      cbas_sl_exp(9, 74)=  9.2933_wp
      cbas_sl_exp(1, 75)= 73.5478_wp
      cbas_sl_exp(2, 75)= 27.7049_wp
      cbas_sl_exp(3, 75)= 35.2810_wp
      cbas_sl_exp(4, 75)= 17.5495_wp
      cbas_sl_exp(5, 75)= 17.8115_wp
      cbas_sl_exp(6, 75)= 20.4849_wp
      cbas_sl_exp(7, 75)= 10.3391_wp
      cbas_sl_exp(8, 75)= 10.0933_wp
      cbas_sl_exp(9, 75)=  9.5136_wp
      cbas_sl_exp(1, 76)= 74.5299_wp
      cbas_sl_exp(2, 76)= 28.0749_wp
      cbas_sl_exp(3, 76)= 35.7749_wp
      cbas_sl_exp(4, 76)= 17.8091_wp
      cbas_sl_exp(5, 76)= 18.0839_wp
      cbas_sl_exp(6, 76)= 20.8249_wp
      cbas_sl_exp(7, 76)= 10.5238_wp
      cbas_sl_exp(8, 76)= 10.2860_wp
      cbas_sl_exp(9, 76)=  9.7145_wp
      cbas_sl_exp(1, 77)= 75.5119_wp
      cbas_sl_exp(2, 77)= 28.4449_wp
      cbas_sl_exp(3, 77)= 36.2688_wp
      cbas_sl_exp(4, 77)= 18.0686_wp
      cbas_sl_exp(5, 77)= 18.3563_wp
      cbas_sl_exp(6, 77)= 21.1620_wp
      cbas_sl_exp(7, 77)= 10.7120_wp
      cbas_sl_exp(8, 77)= 10.4785_wp
      cbas_sl_exp(9, 77)=  9.9343_wp
      cbas_sl_exp(1, 78)= 76.4940_wp
      cbas_sl_exp(2, 78)= 28.8149_wp
      cbas_sl_exp(3, 78)= 36.7627_wp
      cbas_sl_exp(4, 78)= 18.3281_wp
      cbas_sl_exp(5, 78)= 18.6287_wp
      cbas_sl_exp(6, 78)= 21.4991_wp
      cbas_sl_exp(7, 78)= 10.9097_wp
      cbas_sl_exp(8, 78)= 10.6826_wp
      cbas_sl_exp(9, 78)= 10.1575_wp
      cbas_sl_exp(1, 79)= 77.4761_wp
      cbas_sl_exp(2, 79)= 29.1849_wp
      cbas_sl_exp(3, 79)= 37.2566_wp
      cbas_sl_exp(4, 79)= 18.5876_wp
      cbas_sl_exp(5, 79)= 18.9010_wp
      cbas_sl_exp(6, 79)= 21.8361_wp
      cbas_sl_exp(7, 79)= 11.1033_wp
      cbas_sl_exp(8, 79)= 10.8867_wp
      cbas_sl_exp(9, 79)= 10.3820_wp
      cbas_sl_exp(1, 80)= 78.4581_wp
      cbas_sl_exp(2, 80)= 29.5547_wp
      cbas_sl_exp(3, 80)= 37.7505_wp
      cbas_sl_exp(4, 80)= 18.8471_wp
      cbas_sl_exp(5, 80)= 19.1734_wp
      cbas_sl_exp(6, 80)= 22.1732_wp
      cbas_sl_exp(7, 80)= 11.3112_wp
      cbas_sl_exp(8, 80)= 11.1015_wp
      cbas_sl_exp(9, 80)= 10.6170_wp
      cbas_sl_exp(1, 81)= 79.4409_wp
      cbas_sl_exp(2, 81)= 29.8421_wp
      cbas_sl_exp(3, 81)= 38.2431_wp
      cbas_sl_exp(4, 81)= 19.1397_wp
      cbas_sl_exp(5, 81)= 19.4555_wp
      cbas_sl_exp(6, 81)= 22.5114_wp
      cbas_sl_exp(7, 81)= 11.5197_wp
      cbas_sl_exp(8, 81)= 11.3042_wp
      cbas_sl_exp(9, 81)= 10.8472_wp
      cbas_sl_exp(10,81)=  5.8244_wp
      cbas_sl_exp(11,81)=  5.4177_wp
      cbas_sl_exp(12,81)=  4.4050_wp
      cbas_sl_exp(1, 82)= 80.4195_wp
      cbas_sl_exp(2, 82)= 30.2150_wp
      cbas_sl_exp(3, 82)= 38.7383_wp
      cbas_sl_exp(4, 82)= 19.3841_wp
      cbas_sl_exp(5, 82)= 19.7165_wp
      cbas_sl_exp(6, 82)= 22.8489_wp
      cbas_sl_exp(7, 82)= 11.7232_wp
      cbas_sl_exp(8, 82)= 11.5084_wp
      cbas_sl_exp(9, 82)= 11.0799_wp
      cbas_sl_exp(10,82)=  6.0263_wp
      cbas_sl_exp(11,82)=  5.6060_wp
      cbas_sl_exp(12,82)=  4.6304_wp
      cbas_sl_exp(1, 83)= 81.3982_wp
      cbas_sl_exp(2, 83)= 30.5880_wp
      cbas_sl_exp(3, 83)= 39.2335_wp
      cbas_sl_exp(4, 83)= 19.6285_wp
      cbas_sl_exp(5, 83)= 19.9774_wp
      cbas_sl_exp(6, 83)= 23.1805_wp
      cbas_sl_exp(7, 83)= 11.9268_wp
      cbas_sl_exp(8, 83)= 11.7126_wp
      cbas_sl_exp(9, 83)= 11.3098_wp
      cbas_sl_exp(10,83)=  6.2058_wp
      cbas_sl_exp(11,83)=  5.8042_wp
      cbas_sl_exp(12,83)=  4.8488_wp
      cbas_sl_exp(1, 84)= 82.3768_wp
      cbas_sl_exp(2, 84)= 30.9609_wp
      cbas_sl_exp(3, 84)= 39.7286_wp
      cbas_sl_exp(4, 84)= 19.8729_wp
      cbas_sl_exp(5, 84)= 20.2383_wp
      cbas_sl_exp(6, 84)= 23.5240_wp
      cbas_sl_exp(7, 84)= 12.1304_wp
      cbas_sl_exp(8, 84)= 11.9168_wp
      cbas_sl_exp(9, 84)= 11.9168_wp
      cbas_sl_exp(10,84)=  6.4046_wp
      cbas_sl_exp(11,84)=  6.0049_wp
      cbas_sl_exp(12,84)=  5.0608_wp
      cbas_sl_exp(1, 85)= 83.3554_wp
      cbas_sl_exp(2, 85)= 31.3338_wp
      cbas_sl_exp(3, 85)= 40.2238_wp
      cbas_sl_exp(4, 85)= 20.1173_wp
      cbas_sl_exp(5, 85)= 20.4992_wp
      cbas_sl_exp(6, 85)= 23.8615_wp
      cbas_sl_exp(7, 85)= 12.3339_wp
      cbas_sl_exp(8, 85)= 12.1210_wp
      cbas_sl_exp(9, 85)= 11.7624_wp
      cbas_sl_exp(10,85)=  6.5867_wp
      cbas_sl_exp(11,85)=  6.2080_wp
      cbas_sl_exp(12,85)=  5.2678_wp
      cbas_sl_exp(1,86) = 84.3341_wp
      cbas_sl_exp(2,86) = 31.7068_wp
      cbas_sl_exp(3,86) = 40.7190_wp
      cbas_sl_exp(4,86) = 20.3617_wp
      cbas_sl_exp(5,86) = 20.7602_wp
      cbas_sl_exp(6,86) = 24.1991_wp
      cbas_sl_exp(7,86) = 12.5375_wp
      cbas_sl_exp(8,86) = 12.3253_wp
      cbas_sl_exp(9,86) = 11.9857_wp
      cbas_sl_exp(10,86)=  6.7786_wp
      cbas_sl_exp(11,86)=  6.3942_wp
      cbas_sl_exp(12,86)=  5.4700_wp

      !> Principal quantum numbers
      cbas_pqn(1:cbas_nshell(5), 5) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(6), 6) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(7), 7) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(8), 8) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(9), 9) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(10), 10) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(11), 11) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(12), 12) = [ &
      &  1 ]
      cbas_pqn(1:cbas_nshell(13), 13) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(14), 14) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(15), 15) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(16), 16) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(17), 17) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(18), 18) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(19), 19) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(20), 20) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(21), 21) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(22), 22) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(23), 23) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(24), 24) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(25), 25) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(26), 26) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(27), 27) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(28), 28) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(29), 29) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(30), 30) = [ &
      &  1, 2, 2 ]
      cbas_pqn(1:cbas_nshell(31), 31) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(32), 32) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(33), 33) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(34), 34) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(35), 35) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(36), 36) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(37), 37) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(38), 38) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(39), 39) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(40), 40) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(41), 41) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(42), 42) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(43), 43) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(44), 44) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(45), 45) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(46), 46) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(47), 47) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(48), 48) = [ &
      &  1, 2, 2, 3, 3, 3 ]
      cbas_pqn(1:cbas_nshell(49), 49) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(50), 50) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(51), 51) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(52), 52) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(53), 53) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(54), 54) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(55), 55) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(56), 56) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(57), 57) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(58), 58) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(59), 59) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(60), 60) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(61), 61) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(62), 62) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(63), 63) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(64), 64) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(65), 65) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(66), 66) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(67), 67) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(68), 68) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(69), 69) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(70), 70) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(71), 71) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(72), 72) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(73), 73) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(74), 74) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(75), 75) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(76), 76) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(77), 77) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(78), 78) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(79), 79) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(80), 80) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4 ]
      cbas_pqn(1:cbas_nshell(81), 81) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5 ]
      cbas_pqn(1:cbas_nshell(82), 82) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5 ]
      cbas_pqn(1:cbas_nshell(83), 83) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5 ]
      cbas_pqn(1:cbas_nshell(84), 84) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5 ]
      cbas_pqn(1:cbas_nshell(85), 85) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5 ]
      cbas_pqn(1:cbas_nshell(86), 86) = [ &
      &  1, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5 ]

      !> Angular momentum quantum numbers
      cbas_angshell(1:cbas_nshell(5), 5) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(6), 6) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(7), 7) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(8), 8) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(9), 9) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(10), 10) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(11), 11) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(12), 12) = [ &
      &  0 ]
      cbas_angshell(1:cbas_nshell(13), 13) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(14), 14) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(15), 15) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(16), 16) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(17), 17) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(18), 18) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(19), 19) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(20), 20) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(21), 21) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(22), 22) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(23), 23) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(24), 24) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(25), 25) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(26), 26) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(27), 27) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(28), 28) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(29), 29) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(30), 30) = [ &
      &  0, 0, 1 ]
      cbas_angshell(1:cbas_nshell(31), 31) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(32), 32) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(33), 33) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(34), 34) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(35), 35) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(36), 36) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(37), 37) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(38), 38) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(39), 39) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(40), 40) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(41), 41) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(42), 42) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(43), 43) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(44), 44) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(45), 45) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(46), 46) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(47), 47) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(48), 48) = [ &
      &  0, 0, 1, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(49), 49) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(50), 50) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(51), 51) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(52), 52) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(53), 53) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(54), 54) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(55), 55) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(56), 56) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(57), 57) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(58), 58) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(59), 59) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(60), 60) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(61), 61) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(62), 62) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(63), 63) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(64), 64) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(65), 65) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(66), 66) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(67), 67) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(68), 68) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(69), 69) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(70), 70) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(71), 71) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(72), 72) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(73), 73) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(74), 74) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(75), 75) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(76), 76) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(77), 77) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(78), 78) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(79), 79) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(80), 80) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(81), 81) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(82), 82) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(83), 83) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(84), 84) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(85), 85) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2, 0, 1, 2 ]
      cbas_angshell(1:cbas_nshell(86), 86) = [ &
      &  0, 0, 1, 0, 1, 2, 0, 1, 2, 0, 1, 2 ]

      !> HF levels for core shells
      cbas_hflev(1, 1) =  -0.5000000_wp
      cbas_hflev(1, 2) =  -0.9179508_wp
      cbas_hflev(1, 3) =  -2.4777262_wp
      cbas_hflev(1, 4) =  -4.7863203_wp
      cbas_hflev(1, 5) =  -7.6940338_wp
      cbas_hflev(1, 6) = -11.3255170_wp
      cbas_hflev(1, 7) = -15.6290579_wp
      cbas_hflev(1, 8) = -20.6686540_wp
      cbas_hflev(1, 9) = -26.3827566_wp
      cbas_hflev(1,10) = -32.7724138_wp
      cbas_hflev(1,11) = -40.4782393_wp
      cbas_hflev(2,11) =  -2.7968751_wp
      cbas_hflev(3,11) =  -1.5180133_wp
      cbas_hflev(1,12) = -49.0801076_wp
      cbas_hflev(2,12) =  -3.8127947_wp
      cbas_hflev(3,12) =  -2.3310958_wp
      cbas_hflev(1,13) = -58.5009508_wp
      cbas_hflev(2,13) =  -4.9106092_wp
      cbas_hflev(3,13) =  -3.2182410_wp
      cbas_hflev(1,14) = -68.8123721_wp
      cbas_hflev(2,14) =  -6.1564677_wp
      cbas_hflev(3,14) =  -4.2559826_wp
      cbas_hflev(1,15) = -79.9696269_wp
      cbas_hflev(2,15) =  -7.5110229_wp
      cbas_hflev(3,15) =  -5.4008827_wp
      cbas_hflev(1,16) = -92.0043547_wp
      cbas_hflev(2,16) =  -9.0042114_wp
      cbas_hflev(3,16) =  -6.6824236_wp
      cbas_hflev(1,17) =-104.8843196_wp
      cbas_hflev(2,17) = -10.6073995_wp
      cbas_hflev(3,17) =  -8.0721369_wp
      cbas_hflev(1,18) =-118.6102421_wp
      cbas_hflev(2,18) = -12.3220674_wp
      cbas_hflev(3,18) =  -9.5713692_wp
      cbas_hflev(1,19) =-133.5237406_wp
      cbas_hflev(2,19) = -14.4775370_wp
      cbas_hflev(3,19) = -11.4944223_wp
      cbas_hflev(4,19) =  -1.7405194_wp
      cbas_hflev(5,19) =  -.94265914_wp
      cbas_hflev(1,20) =-149.3613693_wp
      cbas_hflev(2,20) = -16.8210000_wp
      cbas_hflev(3,20) = -13.6277555_wp
      cbas_hflev(4,20) =  -2.2444052_wp
      cbas_hflev(5,20) =  -1.3398417_wp
      cbas_hflev(1,21) =-165.8969001_wp
      cbas_hflev(2,21) = -19.0780965_wp
      cbas_hflev(3,21) = -15.6659817_wp
      cbas_hflev(1,22) =-183.2692614_wp
      cbas_hflev(2,22) = -21.4197345_wp
      cbas_hflev(3,22) = -17.7883022_wp
      cbas_hflev(1,23) =-201.4987403_wp
      cbas_hflev(2,23) = -23.8706785_wp
      cbas_hflev(3,23) = -20.0188512_wp
      cbas_hflev(1,24) =-220.5874573_wp
      cbas_hflev(2,24) = -26.4342375_wp
      cbas_hflev(3,24) = -22.3607530_wp
      cbas_hflev(1,25) =-240.5330490_wp
      cbas_hflev(2,25) = -29.1089634_wp
      cbas_hflev(3,25) = -24.8121004_wp
      cbas_hflev(1,26) =-261.3667440_wp
      cbas_hflev(2,26) = -31.9280372_wp
      cbas_hflev(3,26) = -27.4067383_wp
      cbas_hflev(1,27) =-283.0575937_wp
      cbas_hflev(2,27) = -34.8591867_wp
      cbas_hflev(3,27) = -30.1116132_wp
      cbas_hflev(1,28) =-305.6097805_wp
      cbas_hflev(2,28) = -37.9068679_wp
      cbas_hflev(3,28) = -32.9314426_wp
      cbas_hflev(1,29) =-328.7838501_wp
      cbas_hflev(2,29) = -40.8100313_wp
      cbas_hflev(3,29) = -35.6095546_wp
      cbas_hflev(1,30) =-353.2923720_wp
      cbas_hflev(2,30) = -44.3467866_wp
      cbas_hflev(3,30) = -38.9107605_wp
      cbas_hflev(4,30) =  -5.6215957_wp
      cbas_hflev(5,30) =  -3.8244276_wp
      cbas_hflev(1,31) =-378.8176563_wp
      cbas_hflev(2,31) = -48.1674004_wp
      cbas_hflev(3,31) = -42.4930214_wp
      cbas_hflev(4,31) =  -6.3935791_wp
      cbas_hflev(5,31) =  -4.4813393_wp
      cbas_hflev(6,31) =  -1.1921173_wp
      cbas_hflev(1,32) =-405.2355329_wp
      cbas_hflev(2,32) = -52.1395939_wp
      cbas_hflev(3,32) = -46.2263666_wp
      cbas_hflev(4,32) =  -7.1797990_wp
      cbas_hflev(5,32) =  -5.1513892_wp
      cbas_hflev(6,32) =  -1.6202589_wp
      cbas_hflev(1,33) =-432.5779002_wp
      cbas_hflev(2,33) = -56.2999996_wp
      cbas_hflev(3,33) = -50.1448668_wp
      cbas_hflev(4,33) =  -8.0195796_wp
      cbas_hflev(5,33) =  -5.8715542_wp
      cbas_hflev(6,33) =  -2.0988900_wp
      cbas_hflev(1,34) =-460.8590130_wp
      cbas_hflev(2,34) = -60.6592102_wp
      cbas_hflev(3,34) = -54.2601596_wp
      cbas_hflev(4,34) =  -8.9223639_wp
      cbas_hflev(5,34) =  -6.6526355_wp
      cbas_hflev(6,34) =  -2.6359437_wp
      cbas_hflev(1,35) =-490.0518044_wp
      cbas_hflev(2,35) = -65.1903812_wp
      cbas_hflev(3,35) = -58.5455164_wp
      cbas_hflev(4,35) =  -9.8623357_wp
      cbas_hflev(5,35) =  -7.4694575_wp
      cbas_hflev(6,35) =  -3.2064812_wp
      cbas_hflev(1,36) =-520.1568748_wp
      cbas_hflev(2,36) = -69.8936596_wp
      cbas_hflev(3,36) = -63.0011463_wp
      cbas_hflev(4,36) = -10.8401409_wp
      cbas_hflev(5,36) =  -8.3229134_wp
      cbas_hflev(6,36) =  -3.8115953_wp
      cbas_hflev(1,37) = -547.980732_wp
      cbas_hflev(2,37) =  -74.693343_wp
      cbas_hflev(3,37) =  -67.825593_wp
      cbas_hflev(4,37) =  -12.075449_wp
      cbas_hflev(5,37) =   -9.475439_wp
      cbas_hflev(6,37) =   -4.721253_wp
      cbas_hflev(7,37) =   -1.518441_wp
      cbas_hflev(8,37) =   -0.810443_wp
      cbas_hflev(1,38) = -579.760364_wp
      cbas_hflev(2,38) =  -79.980107_wp
      cbas_hflev(3,38) =  -72.890869_wp
      cbas_hflev(4,38) =  -13.408268_wp
      cbas_hflev(5,38) =  -10.684558_wp
      cbas_hflev(6,38) =   -5.680669_wp
      cbas_hflev(7,38) =   -1.891043_wp
      cbas_hflev(8,38) =   -1.099292_wp
      cbas_hflev(1,39)=  -612.239994_wp
      cbas_hflev(2,39)=   -85.216226_wp
      cbas_hflev(3,39)=   -77.906449_wp
      cbas_hflev(4,39)=   -14.551228_wp
      cbas_hflev(5,39)=   -11.704937_wp
      cbas_hflev(6,39)=    -6.455055_wp
      cbas_hflev(1,40)=  -645.621746_wp
      cbas_hflev(2,40)=   -90.685052_wp
      cbas_hflev(3,40)=   -83.168932_wp
      cbas_hflev(4,40)=   -15.797094_wp
      cbas_hflev(5,40)=   -12.828955_wp
      cbas_hflev(6,40)=    -7.330992_wp
      cbas_hflev(1,41)=  -679.938278_wp
      cbas_hflev(2,41)=   -96.327380_wp
      cbas_hflev(3,41)=   -88.599225_wp
      cbas_hflev(4,41)=   -17.079740_wp
      cbas_hflev(5,41)=   -13.987897_wp
      cbas_hflev(6,41)=    -8.241586_wp
      cbas_hflev(1,42)=   -715.14602_wp
      cbas_hflev(2,42)=   -102.15416_wp
      cbas_hflev(3,42)=   -94.198674_wp
      cbas_hflev(4,42)=   -18.401556_wp
      cbas_hflev(5,42)=   -15.181811_wp
      cbas_hflev(6,42)=    -9.186680_wp
      cbas_hflev(1,43)=  -751.332698_wp
      cbas_hflev(2,43)=  -108.238479_wp
      cbas_hflev(3,43)=  -100.067916_wp
      cbas_hflev(4,43)=   -19.862132_wp
      cbas_hflev(5,43)=   -16.523991_wp
      cbas_hflev(6,43)=   -10.282606_wp
      cbas_hflev(1,44)=  -788.313407_wp
      cbas_hflev(2,44)=  -114.416024_wp
      cbas_hflev(3,44)=  -106.026746_wp
      cbas_hflev(4,44)=   -21.277790_wp
      cbas_hflev(5,44)=   -17.803945_wp
      cbas_hflev(6,44)=   -11.321791_wp
      cbas_hflev(1,45)=  -826.232244_wp
      cbas_hflev(2,45)=  -120.763025_wp
      cbas_hflev(3,45)=  -112.153807_wp
      cbas_hflev(4,45)=   -22.730477_wp
      cbas_hflev(5,45)=   -19.128054_wp
      cbas_hflev(6,45)=   -12.393816_wp
      cbas_hflev(1,46)=  -865.069606_wp
      cbas_hflev(2,46)=  -127.281073_wp
      cbas_hflev(3,46)=  -118.447098_wp
      cbas_hflev(4,46)=   -24.220852_wp
      cbas_hflev(5,46)=   -20.482548_wp
      cbas_hflev(6,46)=   -13.508039_wp
      cbas_hflev(1,47)=  -904.736227_wp
      cbas_hflev(2,47)=  -133.958897_wp
      cbas_hflev(3,47)=  -124.906322_wp
      cbas_hflev(4,47)=   -25.742792_wp
      cbas_hflev(5,47)=   -21.880871_wp
      cbas_hflev(6,47)=   -14.632828_wp
      cbas_hflev(1,48)=  -945.544783_wp
      cbas_hflev(2,48)=  -141.029820_wp
      cbas_hflev(3,48)=  -131.750662_wp
      cbas_hflev(4,48)=   -27.521588_wp
      cbas_hflev(5,48)=   -23.526435_wp
      cbas_hflev(6,48)=   -16.024400_wp
      cbas_hflev(7,48)=    -4.425086_wp
      cbas_hflev(8,48)=    -3.050318_wp
      cbas_hflev(1,49) =  -986.039127_wp
      cbas_hflev(2,49) =  -148.329602_wp
      cbas_hflev(3,49) =  -138.982025_wp
      cbas_hflev(4,49) =   -29.431590_wp
      cbas_hflev(5,49) =   -25.333321_wp
      cbas_hflev(6,49) =   -17.581263_wp
      cbas_hflev(7,49) =    -4.945096_wp
      cbas_hflev(8,49) =    -3.504725_wp
      cbas_hflev(9,49) =    -1.066632_wp
      cbas_hflev(1,50) = -1028.462609_wp
      cbas_hflev(2,50) =  -155.822626_wp
      cbas_hflev(3,50) =  -146.261871_wp
      cbas_hflev(4,50) =   -31.394339_wp
      cbas_hflev(5,50) =   -27.165115_wp
      cbas_hflev(6,50) =   -19.156938_wp
      cbas_hflev(7,50) =    -5.482144_wp
      cbas_hflev(8,50) =    -3.969944_wp
      cbas_hflev(9,50) =    -1.377398_wp
      cbas_hflev(1,51) = -1071.885115_wp
      cbas_hflev(2,51) =  -163.525440_wp
      cbas_hflev(3,51) =  -153.721503_wp
      cbas_hflev(4,51) =   -33.415327_wp
      cbas_hflev(5,51) =   -29.048139_wp
      cbas_hflev(6,51) =   -20.782680_wp
      cbas_hflev(7,51) =    -6.026311_wp
      cbas_hflev(8,51) =    -4.440078_wp
      cbas_hflev(9,51) =    -1.692358_wp
      cbas_hflev(1,52) = -1116.072227_wp
      cbas_hflev(2,52) =  -171.395384_wp
      cbas_hflev(3,52) =  -161.402426_wp
      cbas_hflev(4,52) =   -35.513193_wp
      cbas_hflev(5,52) =   -31.019088_wp
      cbas_hflev(6,52) =   -22.496886_wp
      cbas_hflev(7,52) =    -6.608889_wp
      cbas_hflev(8,52) =    -4.949904_wp
      cbas_hflev(9,52) =    -2.046883_wp
      cbas_hflev(1,53) = -1177.186224_wp
      cbas_hflev(2,53) =  -180.949488_wp
      cbas_hflev(3,53) =  -169.660877_wp
      cbas_hflev(4,53) =   -37.934830_wp
      cbas_hflev(5,53) =   -33.122800_wp
      cbas_hflev(6,53) =   -24.286115_wp
      cbas_hflev(7,53) =    -7.244676_wp
      cbas_hflev(8,53) =    -5.473820_wp
      cbas_hflev(9,53) =    -2.401629_wp
      cbas_hflev(1,54) = -1224.397652_wp
      cbas_hflev(2,54) =  -189.340333_wp
      cbas_hflev(3,54) =  -177.782921_wp
      cbas_hflev(4,54) =   -40.175973_wp
      cbas_hflev(5,54) =   -35.222101_wp
      cbas_hflev(6,54) =   -26.119243_wp
      cbas_hflev(7,54) =    -7.856564_wp
      cbas_hflev(8,54) =    -6.008750_wp
      cbas_hflev(9,54) =    -2.778258_wp
      cbas_hflev(1, 55)= -1262.576547_wp
      cbas_hflev(2, 55)=  -196.743815_wp
      cbas_hflev(3, 55)=  -185.315843_wp
      cbas_hflev(4, 55)=   -42.418097_wp
      cbas_hflev(5, 55)=   -37.368061_wp
      cbas_hflev(6, 55)=   -28.085920_wp
      cbas_hflev(7, 55)=    -8.659781_wp
      cbas_hflev(8, 55)=    -6.739622_wp
      cbas_hflev(9, 55)=    -3.380502_wp
      cbas_hflev(10,55)=    -1.228517_wp
      cbas_hflev(11,55)=    -0.680752_wp
      cbas_hflev(1, 56)= -1311.680338_wp
      cbas_hflev(2, 56)=  -205.611145_wp
      cbas_hflev(3, 56)=  -193.816924_wp
      cbas_hflev(4, 56)=   -44.966581_wp
      cbas_hflev(5, 56)=   -39.744461_wp
      cbas_hflev(6, 56)=   -30.216149_wp
      cbas_hflev(7, 56)=    -9.514249_wp
      cbas_hflev(8, 56)=    -7.506200_wp
      cbas_hflev(9, 56)=    -3.997419_wp
      cbas_hflev(10,56)=    -1.510172_wp
      cbas_hflev(11,56)=    -0.900567_wp
      cbas_hflev(1, 57)= -1372.280316_wp
      cbas_hflev(2, 57)=  -216.277016_wp
      cbas_hflev(3, 57)=  -203.907421_wp
      cbas_hflev(4, 57)=   -47.844418_wp
      cbas_hflev(5, 57)=   -42.459156_wp
      cbas_hflev(6, 57)=   -32.553605_wp
      cbas_hflev(7, 57)=   -10.345538_wp
      cbas_hflev(8, 57)=    -8.258909_wp
      cbas_hflev(9, 57)=    -4.553588_wp
      cbas_hflev(1, 72)= -2213.645904_wp
      cbas_hflev(2, 72)=  -366.688200_wp
      cbas_hflev(3, 72)=  -348.985532_wp
      cbas_hflev(4, 72)=   -84.830032_wp
      cbas_hflev(5, 72)=   -76.585360_wp
      cbas_hflev(6, 72)=   -63.431921_wp
      cbas_hflev(7, 72)=   -17.630131_wp
      cbas_hflev(8, 72)=   -14.212758_wp
      cbas_hflev(9, 72)=    -8.753137_wp
      cbas_hflev(1, 73)= -2277.525256_wp
      cbas_hflev(2, 73)=  -378.519617_wp
      cbas_hflev(3, 73)=  -360.405013_wp
      cbas_hflev(4, 73)=   -87.994492_wp
      cbas_hflev(5, 73)=   -79.524638_wp
      cbas_hflev(6, 73)=   -66.160821_wp
      cbas_hflev(7, 73)=   -18.487062_wp
      cbas_hflev(8, 73)=   -14.956933_wp
      cbas_hflev(9, 73)=    -9.364571_wp
      cbas_hflev(1, 74)= -2342.330246_wp
      cbas_hflev(2, 74)=  -390.571409_wp
      cbas_hflev(3, 74)=  -372.032893_wp
      cbas_hflev(4, 74)=   -91.239797_wp
      cbas_hflev(5, 74)=   -82.538795_wp
      cbas_hflev(6, 74)=   -68.967486_wp
      cbas_hflev(7, 74)=   -19.373856_wp
      cbas_hflev(8, 74)=   -15.727283_wp
      cbas_hflev(9, 74)=   -10.002187_wp
      cbas_hflev(1, 75)= -2408.394783_wp
      cbas_hflev(2, 75)=  -402.936575_wp
      cbas_hflev(3, 75)=  -383.943938_wp
      cbas_hflev(4, 75)=   -94.654521_wp
      cbas_hflev(5, 75)=   -85.713161_wp
      cbas_hflev(6, 75)=   -71.938180_wp
      cbas_hflev(7, 75)=   -20.384014_wp
      cbas_hflev(8, 75)=   -16.618760_wp
      cbas_hflev(9, 75)=   -10.765282_wp
      cbas_hflev(1, 76)= -2475.611308_wp
      cbas_hflev(2, 76)=  -415.464765_wp
      cbas_hflev(3, 76)=  -395.986468_wp
      cbas_hflev(4, 76)=   -98.076899_wp
      cbas_hflev(5, 76)=   -88.882313_wp
      cbas_hflev(6, 76)=   -74.907027_wp
      cbas_hflev(7, 76)=   -21.344294_wp
      cbas_hflev(8, 76)=   -17.449423_wp
      cbas_hflev(9, 76)=   -11.469248_wp
      cbas_hflev(1, 77)= -2544.132094_wp
      cbas_hflev(2, 77)=  -428.228553_wp
      cbas_hflev(3, 77)=  -408.235971_wp
      cbas_hflev(4, 77)=  -101.578347_wp
      cbas_hflev(5, 77)=   -92.123900_wp
      cbas_hflev(6, 77)=   -77.949630_wp
      cbas_hflev(7, 77)=   -22.329022_wp
      cbas_hflev(8, 77)=   -18.304471_wp
      cbas_hflev(9, 77)=   -12.193095_wp
      cbas_hflev(1, 78)= -2613.096488_wp
      cbas_hflev(2, 78)=  -441.152576_wp
      cbas_hflev(3, 78)=  -420.681844_wp
      cbas_hflev(4, 78)=  -105.137875_wp
      cbas_hflev(5, 78)=   -95.429451_wp
      cbas_hflev(6, 78)=   -81.059442_wp
      cbas_hflev(7, 78)=   -23.332845_wp
      cbas_hflev(8, 78)=   -19.182828_wp
      cbas_hflev(9, 78)=   -12.943674_wp
      cbas_hflev(1, 79)= -2683.512776_wp
      cbas_hflev(2, 79)=  -454.317452_wp
      cbas_hflev(3, 79)=  -433.310346_wp
      cbas_hflev(4, 79)=  -108.768191_wp
      cbas_hflev(5, 79)=   -98.785511_wp
      cbas_hflev(6, 79)=   -84.218451_wp
      cbas_hflev(7, 79)=   -24.354862_wp
      cbas_hflev(8, 79)=   -20.062847_wp
      cbas_hflev(9, 79)=   -13.686616_wp
      cbas_hflev(1, 80)= -2754.748364_wp
      cbas_hflev(2, 80)=  -467.845242_wp
      cbas_hflev(3, 80)=  -446.308106_wp
      cbas_hflev(4, 80)=  -112.639779_wp
      cbas_hflev(5, 80)=  -102.381961_wp
      cbas_hflev(6, 80)=   -87.623964_wp
      cbas_hflev(7, 80)=   -25.578690_wp
      cbas_hflev(8, 80)=   -21.145629_wp
      cbas_hflev(9, 80)=   -14.641991_wp
      cbas_hflev(10,80)=    -4.193895_wp
      cbas_hflev(11,80)=    -2.751243_wp
      cbas_hflev(1, 81)= -2827.018225_wp
      cbas_hflev(2, 81)=  -481.601770_wp
      cbas_hflev(3, 81)=  -459.453771_wp
      cbas_hflev(4, 81)=  -116.631912_wp
      cbas_hflev(5, 81)=  -106.065182_wp
      cbas_hflev(6, 81)=   -91.144789_wp
      cbas_hflev(7, 81)=   -26.890410_wp
      cbas_hflev(8, 81)=   -22.303972_wp
      cbas_hflev(9, 81)=   -15.683246_wp
      cbas_hflev(10,81)=    -4.639016_wp
      cbas_hflev(11,81)=    -3.122645_wp
      cbas_hflev(12,81)=    -0.985432_wp
      cbas_hflev(1, 82)= -2900.606343_wp
      cbas_hflev(2, 82)=  -495.578042_wp
      cbas_hflev(3, 82)=  -472.677922_wp
      cbas_hflev(4, 82)=  -120.730979_wp
      cbas_hflev(5, 82)=  -109.792024_wp
      cbas_hflev(6, 82)=   -94.769449_wp
      cbas_hflev(7, 82)=   -28.248089_wp
      cbas_hflev(8, 82)=   -23.484894_wp
      cbas_hflev(9, 82)=   -16.768326_wp
      cbas_hflev(10,82)=    -5.090038_wp
      cbas_hflev(11,82)=    -3.492872_wp
      cbas_hflev(12,82)=    -1.250144_wp
      cbas_hflev(1, 83)= -2975.353478_wp
      cbas_hflev(2, 83)=  -509.756597_wp
      cbas_hflev(3, 83)=  -486.071646_wp
      cbas_hflev(4, 83)=  -124.895174_wp
      cbas_hflev(5, 83)=  -113.568438_wp
      cbas_hflev(6, 83)=   -98.455627_wp
      cbas_hflev(7, 83)=   -29.627671_wp
      cbas_hflev(8, 83)=   -24.679241_wp
      cbas_hflev(9, 83)=   -17.873706_wp
      cbas_hflev(10,83)=    -5.539140_wp
      cbas_hflev(11,83)=    -3.857371_wp
      cbas_hflev(12,83)=    -1.512035_wp
      cbas_hflev(1, 84)= -3050.371760_wp
      cbas_hflev(2, 84)=  -524.154675_wp
      cbas_hflev(3, 84)=  -499.687582_wp
      cbas_hflev(4, 84)=  -129.168353_wp
      cbas_hflev(5, 84)=  -117.442762_wp
      cbas_hflev(6, 84)=  -102.253755_wp
      cbas_hflev(7, 84)=   -31.066816_wp
      cbas_hflev(8, 84)=   -25.926638_wp
      cbas_hflev(9, 84)=   -19.038207_wp
      cbas_hflev(10,84)=    -6.019158_wp
      cbas_hflev(11,84)=    -4.249033_wp
      cbas_hflev(12,84)=    -1.804074_wp
      cbas_hflev(1, 85)= -3126.802743_wp
      cbas_hflev(2, 85)=  -538.771899_wp
      cbas_hflev(3, 85)=  -513.486907_wp
      cbas_hflev(4, 85)=  -133.511476_wp
      cbas_hflev(5, 85)=  -121.374707_wp
      cbas_hflev(6, 85)=  -106.120340_wp
      cbas_hflev(7, 85)=   -32.529169_wp
      cbas_hflev(8, 85)=   -27.188944_wp
      cbas_hflev(9, 85)=   -20.223834_wp
      cbas_hflev(10,85)=    -6.499404_wp
      cbas_hflev(11,85)=    -4.637214_wp
      cbas_hflev(12,85)=    -2.096094_wp
      cbas_hflev(1, 86)= -3230.313904_wp
      cbas_hflev(2, 86)=  -556.914740_wp
      cbas_hflev(3, 86)=  -536.678708_wp
      cbas_hflev(4, 86)=  -138.423569_wp
      cbas_hflev(5, 86)=  -128.673283_wp
      cbas_hflev(6, 86)=  -110.703144_wp
      cbas_hflev(7, 86)=   -33.922371_wp
      cbas_hflev(8, 86)=   -29.492852_wp
      cbas_hflev(9, 86)=   -21.333037_wp
      cbas_hflev(10,86)=    -6.907222_wp
      cbas_hflev(11,86)=    -5.226761_wp
      cbas_hflev(12,86)=    -2.327901_wp

   end subroutine set_cbas

   pure subroutine shellocc_ref(at,nsh,refocc)
      integer, intent(in) :: at
      integer, intent(in) :: nsh
      real(wp), allocatable, intent(out) :: refocc(:)
      real(wp) :: socc(max_shell)

      allocate(refocc(nsh),source=0.0_wp)

      select case (at)
       case (1)
         socc( 1)=  0.69264222808434_wp
         socc( 2)=  0.14159955297977_wp
         socc( 3)=  0.16575821893589_wp
       case (2)
         socc( 1)=  1.30320944384752_wp
         socc( 2)=  0.67445566751767_wp
         socc( 3)=  0.02233488863481_wp
       case (3)
         socc( 1)=  1.95899377872121_wp
         socc( 2)=  0.25267315485192_wp
         socc( 3)=  0.11620627212046_wp
         socc( 4)=  0.41258386001878_wp
         socc( 5)=  0.25954293428763_wp
       case (4)
         socc( 1)=  1.97111378215842_wp
         socc( 2)=  0.45810747906670_wp
         socc( 3)=  0.15300493595427_wp
         socc( 4)=  0.94930693774026_wp
         socc( 5)=  0.46846686508036_wp
       case (5)
         socc( 1)=  0.48354388386042_wp
         socc( 2)=  0.15681644999964_wp
         socc( 3)=  1.47322175838681_wp
         socc( 4)=  0.38482788015938_wp
         socc( 5)=  0.50159002759375_wp
       case (6)
         socc( 1)=  0.55666397324469_wp
         socc( 2)=  0.25120261811235_wp
         socc( 3)=  2.10755997506139_wp
         socc( 4)=  0.75746522555791_wp
         socc( 5)=  0.32710820802367_wp
       case (7)
         socc( 1)=  0.79624139865659_wp
         socc( 2)=  0.31749496440959_wp
         socc( 3)=  2.73635251741813_wp
         socc( 4)=  0.98096348058478_wp
         socc( 5)=  0.16894763893091_wp
       case (8)
         socc( 1)=  1.08460674785680_wp
         socc( 2)=  0.38134064737674_wp
         socc( 3)=  3.32762843994261_wp
         socc( 4)=  1.12937855100283_wp
         socc( 5)=  0.07704561382102_wp
       case (9)
         socc( 1)=  1.26298582965783_wp
         socc( 2)=  0.45839005699390_wp
         socc( 3)=  3.83052034186561_wp
         socc( 4)=  1.40814828911248_wp
         socc( 5)=  0.03995548237017_wp
       case (10)
         socc( 1)=  1.46505363326007_wp
         socc( 2)=  0.52503448452191_wp
         socc( 3)=  3.79760668215486_wp
         socc( 4)=  2.19277651011192_wp
         socc( 5)=  0.01952868995124_wp
       case (11)
         socc( 1)=  1.44272104623990_wp
         socc( 2)=  0.56147696803778_wp
         socc( 3)=  0.30286795060414_wp
         socc( 4)=  5.15471618358905_wp
         socc( 5)=  1.23531381129067_wp
         socc( 6)=  0.30290404023845_wp
       case (12)
         socc( 1)=  1.41118194136168_wp
         socc( 2)=  0.60257450561694_wp
         socc( 3)=  0.58202476396721_wp
         socc( 4)=  5.90203159422521_wp
         socc( 5)=  0.88913520405121_wp
         socc( 6)=  0.61305199077774_wp
       case (13)
         socc( 1)=  0.55409123445529_wp
         socc( 2)=  0.18700939319552_wp
         socc( 3)=  1.14200108985144_wp
         socc( 4)=  0.36773005186021_wp
         socc( 5)=  0.74916823063754_wp
       case (14)
         socc( 1)=  0.78489160465251_wp
         socc( 2)=  0.22437622129695_wp
         socc( 3)=  1.89887560555902_wp
         socc( 4)=  0.54518793303536_wp
         socc( 5)=  0.54666863545615_wp
       case (15)
         socc( 1)=  0.99441180454863_wp
         socc( 2)=  0.38348294255053_wp
         socc( 3)=  2.42166885021688_wp
         socc( 4)=  0.62108522469704_wp
         socc( 5)=  0.57935117798692_wp
       case (16)
         socc( 1)=  1.16875462402413_wp
         socc( 2)=  0.42250722184002_wp
         socc( 3)=  3.18695750632506_wp
         socc( 4)=  0.93092404067175_wp
         socc( 5)=  0.29085660713904_wp
       case (17)
         socc( 1)=  1.26904700169420_wp
         socc( 2)=  0.50124825150823_wp
         socc( 3)=  3.96697497126814_wp
         socc( 4)=  1.15243762904417_wp
         socc( 5)=  0.11029214648526_wp
       case (18)
         socc( 1)=  1.36124497167220_wp
         socc( 2)=  0.63500946581722_wp
         socc( 3)=  4.81587227054938_wp
         socc( 4)=  1.06582014894422_wp
         socc( 5)=  0.12205314301698_wp
       case (19)
         socc( 1)=  1.53522663338979_wp
         socc( 2)=  0.51632834060987_wp
         socc( 3)=  0.14469017236935_wp
         socc( 4)=  5.81075185616811_wp
         socc( 5)=  0.70558219720276_wp
         socc( 6)=  0.28742080026012_wp
       case (20)
         socc( 1)=  1.79844483420962_wp
         socc( 2)=  0.27675907011070_wp
         socc( 3)=  0.56912010580096_wp
         socc( 4)=  5.78388493742262_wp
         socc( 5)=  0.97402936072941_wp
         socc( 6)=  0.59776169172669_wp
       case (21)
         socc( 1)=  1.05624132296618_wp
         socc( 2)=  0.86464541756055_wp
         socc( 3)=  0.36326980759719_wp
         socc( 4)=  5.89679098392234_wp
         socc( 5)=  0.36452506454090_wp
         socc( 6)=  1.44469536638604_wp
         socc( 7)=  1.00983203702680_wp
       case (22)
         socc( 1)=  0.96375901454332_wp
         socc( 2)=  0.95827024977728_wp
         socc( 3)=  0.30602127120364_wp
         socc( 4)=  5.68047605524000_wp
         socc( 5)=  0.95761406606174_wp
         socc( 6)=  1.71990673440951_wp
         socc( 7)=  1.41395260876451_wp
       case (23)
         socc( 1)=  0.79929332407016_wp
         socc( 2)=  1.12033730632854_wp
         socc( 3)=  0.25672107345614_wp
         socc( 4)=  5.85964123820359_wp
         socc( 5)=  0.77931057740872_wp
         socc( 6)=  2.23857038294087_wp
         socc( 7)=  1.94612609759199_wp
       case (24)
         socc( 1)=  0.82462845140315_wp
         socc( 2)=  1.09418941812312_wp
         socc( 3)=  0.27451288721349_wp
         socc( 4)=  5.89703484799000_wp
         socc( 5)=  0.44878161103213_wp
         socc( 6)=  3.14653845696373_wp
         socc( 7)=  2.31431432727439_wp
       case (25)
         socc( 1)=  1.08108361907627_wp
         socc( 2)=  0.83075965228288_wp
         socc( 3)=  0.23005700106202_wp
         socc( 4)=  5.73086784715238_wp
         socc( 5)=  1.02936636175660_wp
         socc( 6)=  3.43462102064233_wp
         socc( 7)=  2.66324449802751_wp
       case (26)
         socc( 1)=  1.22593630105257_wp
         socc( 2)=  0.69723140299320_wp
         socc( 3)=  0.26753120338196_wp
         socc( 4)=  5.74501886069775_wp
         socc( 5)=  0.98523486849461_wp
         socc( 6)=  4.06455486747179_wp
         socc( 7)=  3.01449249590812_wp
       case (27)
         socc( 1)=  1.21903505020363_wp
         socc( 2)=  0.71617078781128_wp
         socc( 3)=  0.29865787302788_wp
         socc( 4)=  5.82268044144608_wp
         socc( 5)=  0.95032956216758_wp
         socc( 6)=  4.59872624986471_wp
         socc( 7)=  3.39440003547883_wp
       case (28)
         socc( 1)=  1.20548003361775_wp
         socc( 2)=  0.74543920617837_wp
         socc( 3)=  0.40683509440805_wp
         socc( 4)=  5.76626888605264_wp
         socc( 5)=  0.81076059239549_wp
         socc( 6)=  5.11651654687471_wp
         socc( 7)=  3.94869964047298_wp
       case (29)
         socc( 1)=  1.20406871070405_wp
         socc( 2)=  0.75784029468091_wp
         socc( 3)=  0.49742209666322_wp
         socc( 4)=  5.80300045156193_wp
         socc( 5)=  0.60450334809067_wp
         socc( 6)=  5.86388958363827_wp
         socc( 7)=  4.26927551466095_wp
       case (30)
         socc( 1)=  1.12234938727773_wp
         socc( 2)=  0.84672965854406_wp
         socc( 3)=  0.72400604003908_wp
         socc( 4)=  5.89216762262842_wp
         socc( 5)=  0.77667440854275_wp
         socc( 6)=  6.12511364494980_wp
         socc( 7)=  4.51295923801816_wp
       case (31)
         socc( 1)=  0.66219163255453_wp
         socc( 2)=  0.25657241747323_wp
         socc( 3)=  1.11150909382464_wp
         socc( 4)=  0.33620459465314_wp
         socc( 5)=  0.63352226149446_wp
       case (32)
         socc( 1)=  0.85984908336581_wp
         socc( 2)=  0.26060780725228_wp
         socc( 3)=  1.67044240548229_wp
         socc( 4)=  0.66685034189601_wp
         socc( 5)=  0.54225036200361_wp
       case (33)
         socc( 1)=  1.07398807439989_wp
         socc( 2)=  0.41816854255620_wp
         socc( 3)=  2.25303762419483_wp
         socc( 4)=  0.70112654135950_wp
         socc( 5)=  0.55367921748958_wp
       case (34)
         socc( 1)=  1.27903270513343_wp
         socc( 2)=  0.39984240042723_wp
         socc( 3)=  3.03138286397761_wp
         socc( 4)=  0.90378034037950_wp
         socc( 5)=  0.38596169008223_wp
       case (35)
         socc( 1)=  1.40434899781744_wp
         socc( 2)=  0.42022034962529_wp
         socc( 3)=  3.94291335821524_wp
         socc( 4)=  1.12941461820133_wp
         socc( 5)=  0.10310267614070_wp
       case (36)
         socc( 1)=  1.54065315393567_wp
         socc( 2)=  0.48393306127598_wp
         socc( 3)=  4.38005513779559_wp
         socc( 4)=  1.36203924680094_wp
         socc( 5)=  0.23331940019183_wp
       case (37)
         socc( 1)=  1.16316957848310_wp
         socc( 2)=  0.77396207237532_wp
         socc( 3)=  0.35983295313118_wp
         socc( 4)=  5.84628732578992_wp
         socc( 5)=  0.54124987542056_wp
         socc( 6)=  0.31549819479992_wp
       case (38)
         socc( 1)=  1.78170365219315_wp
         socc( 2)=  0.14376890876641_wp
         socc( 3)=  0.63779482765893_wp
         socc( 4)=  5.66476954768101_wp
         socc( 5)=  1.09162275855392_wp
         socc( 6)=  0.68034030514658_wp
       case (39)
         socc( 1)=  0.86282319345063_wp
         socc( 2)=  1.01405042367016_wp
         socc( 3)=  0.32985729489430_wp
         socc( 4)=  5.70344110701582_wp
         socc( 5)=  0.84245592374814_wp
         socc( 6)=  1.42454727573665_wp
         socc( 7)=  0.82282478148430_wp
       case (40)
         socc( 1)=  0.67435167245118_wp
         socc( 2)=  1.20342585264918_wp
         socc( 3)=  0.34601231372147_wp
         socc( 4)=  5.53238142398467_wp
         socc( 5)=  1.11086040916148_wp
         socc( 6)=  2.44933491569741_wp
         socc( 7)=  0.68363341233460_wp
       case (41)
         socc( 1)=  0.50287232383163_wp
         socc( 2)=  1.37134964592689_wp
         socc( 3)=  0.30190012907732_wp
         socc( 4)=  5.64743868967073_wp
         socc( 5)=  0.99359570886050_wp
         socc( 6)=  3.31220428462917_wp
         socc( 7)=  0.87063921800376_wp
       case (42)
         socc( 1)=  0.87261576286637_wp
         socc( 2)=  1.00425501802180_wp
         socc( 3)=  0.25130110590710_wp
         socc( 4)=  5.71428271095351_wp
         socc( 5)=  0.83658232692674_wp
         socc( 6)=  4.03217966641299_wp
         socc( 7)=  1.28878340891149_wp
       case (43)
         socc( 1)=  1.26586774575342_wp
         socc( 2)=  0.61353672533920_wp
         socc( 3)=  0.26452085786135_wp
         socc( 4)=  5.81864082075886_wp
         socc( 5)=  0.48460244840837_wp
         socc( 6)=  4.62486720062715_wp
         socc( 7)=  1.92796420125166_wp
       case (44)
         socc( 1)=  1.20126149901804_wp
         socc( 2)=  0.69449186606227_wp
         socc( 3)=  0.26071233998777_wp
         socc( 4)=  5.67965818913478_wp
         socc( 5)=  0.90345795952935_wp
         socc( 6)=  5.38350854797263_wp
         socc( 7)=  1.87690959829517_wp
       case (45)
         socc( 1)=  1.20797741909822_wp
         socc( 2)=  0.70089251858493_wp
         socc( 3)=  0.29796640283600_wp
         socc( 4)=  5.78766562168797_wp
         socc( 5)=  0.81506857231596_wp
         socc( 6)=  5.93396731971260_wp
         socc( 7)=  2.25646214576433_wp
       case (46)
         socc( 1)=  1.14953967553598_wp
         socc( 2)=  0.77639261749056_wp
         socc( 3)=  0.34493408116223_wp
         socc( 4)=  5.73173722691006_wp
         socc( 5)=  0.74227886758343_wp
         socc( 6)=  7.05728018989357_wp
         socc( 7)=  2.19783734142418_wp
       case (47)
         socc( 1)=  1.47715411867414_wp
         socc( 2)=  0.46112075991231_wp
         socc( 3)=  0.46976828745373_wp
         socc( 4)=  5.83681422266467_wp
         socc( 5)=  0.56839763037875_wp
         socc( 6)=  8.25936505136489_wp
         socc( 7)=  1.92737992955151_wp
       case (48)
         socc( 1)=  1.44744227031352_wp
         socc( 2)=  0.50321505137676_wp
         socc( 3)=  0.76605254697916_wp
         socc( 4)=  5.84237560645286_wp
         socc( 5)=  0.73805996216542_wp
         socc( 6)=  8.86501958024778_wp
         socc( 7)=  1.83783498246450_wp
       case (49)
         socc( 1)=  0.70140157508278_wp
         socc( 2)=  0.17354305838454_wp
         socc( 3)=  1.04740931228563_wp
         socc( 4)=  0.35347741827462_wp
         socc( 5)=  0.72416863597243_wp
       case (50)
         socc( 1)=  0.88138857934851_wp
         socc( 2)=  0.30270792750431_wp
         socc( 3)=  1.70765743210573_wp
         socc( 4)=  0.56001837529799_wp
         socc( 5)=  0.54822768574346_wp
       case (51)
         socc( 1)=  1.07923018725516_wp
         socc( 2)=  0.48702839024217_wp
         socc( 3)=  2.07653831010358_wp
         socc( 4)=  0.87014657518830_wp
         socc( 5)=  0.48705653721079_wp
       case (52)
         socc( 1)=  1.07764207628998_wp
         socc( 2)=  0.64644728669200_wp
         socc( 3)=  2.50901541843569_wp
         socc( 4)=  1.26324679714389_wp
         socc( 5)=  0.50364842143843_wp
       case (53)
         socc( 1)=  1.30238990024541_wp
         socc( 2)=  0.55688322008723_wp
         socc( 3)=  3.88262669174735_wp
         socc( 4)=  1.13614684343849_wp
         socc( 5)=  0.12195334448153_wp
       case (54)
         socc( 1)=  1.49117207162071_wp
         socc( 2)=  0.56093418226970_wp
         socc( 3)=  4.53368436069951_wp
         socc( 4)=  1.12237345264925_wp
         socc( 5)=  0.29183593276083_wp
       case (55)
         socc( 1)=  1.20018212883561_wp
         socc( 2)=  0.66979530958272_wp
         socc( 3)=  0.52027546654330_wp
         socc( 4)=  5.88465637946255_wp
         socc( 5)=  0.37991495530054_wp
         socc( 6)=  0.34517576027529_wp
       case (56)
         socc( 1)=  1.72761522363496_wp
         socc( 2)=  0.20769891987911_wp
         socc( 3)=  0.54523299736967_wp
         socc( 4)=  5.17593075925072_wp
         socc( 5)=  1.64479764094805_wp
         socc( 6)=  0.69872444771662_wp
       case (57)
         socc( 1)=  1.46529473169990_wp
         socc( 2)=  0.35684994359811_wp
         socc( 3)=  0.30050414785488_wp
         socc( 4)=  5.65562158477479_wp
         socc( 5)=  0.47339987795626_wp
         socc( 6)=  1.94126626446140_wp
         socc( 7)=  0.80706344965465_wp
       case (72)
         socc( 1)=  1.02888520395055_wp
         socc( 2)=  0.86766041482196_wp
         socc( 3)=  0.49634867713150_wp
         socc( 4)=  5.81775995212443_wp
         socc( 5)=  0.38607398667741_wp
         socc( 6)=  2.55936780222997_wp
         socc( 7)=  0.84390396306419_wp
       case (73)
         socc( 1)=  1.14684318757690_wp
         socc( 2)=  0.74184181626919_wp
         socc( 3)=  0.37576199210298_wp
         socc( 4)=  5.65817508760807_wp
         socc( 5)=  0.96009271251448_wp
         socc( 6)=  2.90642094143716_wp
         socc( 7)=  1.21086426249122_wp
       case (74)
         socc( 1)=  1.08250965966898_wp
         socc( 2)=  0.80853928654802_wp
         socc( 3)=  0.35218957244199_wp
         socc( 4)=  5.62687129950991_wp
         socc( 5)=  0.99645393669446_wp
         socc( 6)=  3.93350253188128_wp
         socc( 7)=  1.19993371325537_wp
       case (75)
         socc( 1)=  1.23755244004358_wp
         socc( 2)=  0.65059008487721_wp
         socc( 3)=  0.31310114116509_wp
         socc( 4)=  5.66081470189635_wp
         socc( 5)=  1.01994427206965_wp
         socc( 6)=  4.53555935561923_wp
         socc( 7)=  1.58243800432889_wp
       case (76)
         socc( 1)=  1.15351477996358_wp
         socc( 2)=  0.74317562636339_wp
         socc( 3)=  0.34534774171001_wp
         socc( 4)=  5.59762937280367_wp
         socc( 5)=  0.97786153395086_wp
         socc( 6)=  5.52677596562903_wp
         socc( 7)=  1.65569497957946_wp
       case (77)
         socc( 1)=  0.95700328114132_wp
         socc( 2)=  0.94974278534452_wp
         socc( 3)=  0.38613265161768_wp
         socc( 4)=  5.75686633214355_wp
         socc( 5)=  0.87019632783512_wp
         socc( 6)=  6.68281234310795_wp
         socc( 7)=  1.39724627880986_wp
       case (78)
         socc( 1)=  1.62396906180815_wp
         socc( 2)=  0.27135616269979_wp
         socc( 3)=  0.49722855302172_wp
         socc( 4)=  5.55802102341796_wp
         socc( 5)=  0.87132355327770_wp
         socc( 6)=  7.65933830284699_wp
         socc( 7)=  1.51876334292769_wp
       case (79)
         socc( 1)=  1.19971391814096_wp
         socc( 2)=  0.73584211247308_wp
         socc( 3)=  0.61639472645522_wp
         socc( 4)=  5.71716576508113_wp
         socc( 5)=  0.65941930341563_wp
         socc( 6)=  8.77099785810800_wp
         socc( 7)=  1.30046631632598_wp
       case (80)
         socc( 1)=  0.80989406331215_wp
         socc( 2)=  1.14406917132244_wp
         socc( 3)=  0.93849154566692_wp
         socc( 4)=  5.77696867557387_wp
         socc( 5)=  0.72647677151064_wp
         socc( 6)=  9.31903323367983_wp
         socc( 7)=  1.28506653893413_wp
       case (81)
         socc( 1)=  0.84549117650825_wp
         socc( 2)=  0.27623273506597_wp
         socc( 3)=  0.99814605463893_wp
         socc( 4)=  0.33711577835634_wp
         socc( 5)=  0.54301425543051_wp
       case (82)
         socc( 1)=  0.95746591128469_wp
         socc( 2)=  0.42149251532093_wp
         socc( 3)=  1.58930771841450_wp
         socc( 4)=  0.55141264480076_wp
         socc( 5)=  0.48032121017912_wp
       case (83)
         socc( 1)=  0.94547616956564_wp
         socc( 2)=  0.72870343620536_wp
         socc( 3)=  1.96188878715981_wp
         socc( 4)=  0.89319491934738_wp
         socc( 5)=  0.47073668772181_wp
       case (84)
         socc( 1)=  0.99760576293602_wp
         socc( 2)=  0.81343627348347_wp
         socc( 3)=  2.30864520742057_wp
         socc( 4)=  1.45482731964036_wp
         socc( 5)=  0.42548543651957_wp
       case (85)
         socc( 1)=  1.32802176831061_wp
         socc( 2)=  0.58020111845373_wp
         socc( 3)=  3.45775985791178_wp
         socc( 4)=  1.50842171014431_wp
         socc( 5)=  0.12559554517957_wp
       case (86)
         socc( 1)=  1.40549915956233_wp
         socc( 2)=  0.67064049817754_wp
         socc( 3)=  4.25809761136944_wp
         socc( 4)=  1.50416161699423_wp
         socc( 5)=  0.16160111389646_wp
      end select

      refocc(1:nsh) = socc(1:nsh)

   end subroutine shellocc_ref

#endif
end module xtb_ptb_param
